<?php
namespace BixolaCore\Widgets;

use \Elementor\Widget_Base;
use \Elementor\Controls_Manager;
use \Elementor\Group_Control_Border;
use \Elementor\Group_Control_Typography;
use \Elementor\Group_Control_Box_Shadow;
use \Elementor\Repeater;
use \Elementor\Utils;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Bixola Core
 * @since 1.0.0
 */
class Bixola_Team extends Widget_Base {

	/**
	 * Retrieve the widget name.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'bixola_team';
	}

	/**
	 * Retrieve the widget title.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Team', 'bixola-core' );
	}

	/**
	 * Retrieve the widget icon.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'bixola-icon';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * Note that currently Elementor supports only one category.
	 * When multiple categories passed, Elementor uses the first one.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'bixola_core' ];
	}

	/**
	 * Retrieve the list of scripts the widget depended on.
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'bixola-core' ];
	}

	/**
	 * Register the widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function register_controls() {
		
        $this->start_controls_section(
            '_content_design_layout',
            [
                'label' => esc_html__('Design Layout', 'bixola-core'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'design_style',
            [
                'label' => esc_html__('Select Layout', 'bixola-core'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'layout-1' => esc_html__('Layout 1', 'bixola-core'),
                    'layout-2' => esc_html__('Layout 2', 'bixola-core'),
                    'layout-3' => esc_html__('Layout 3', 'bixola-core'),
                    'layout-4' => esc_html__('Layout 4', 'bixola-core'),
                    'layout-5' => esc_html__('Layout 5', 'bixola-core'),
                    'layout-6' => esc_html__('Layout 6', 'bixola-core'),
                ],
                'default' => 'layout-1',
            ]
        );

        $this->add_control(
            'section_shape_switch',
            [
                'label' => esc_html__( 'Shape Switch', 'bixola-core' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Show', 'bixola-core' ),
                'label_off' => esc_html__( 'Hide', 'bixola-core' ),
                'return_value' => 'yes',
                'default' => 'yes',
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3' ],
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_content_title',
            [
                'label' => esc_html__( 'Title & Content',  'bixola-core'  ),
                'tab' => Controls_Manager::TAB_CONTENT,
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-2', 'layout-3', 'layout-4' ],
                ],
            ]
        );
        
        $this->add_control(
            'section_subheading',
            [
                'label' => esc_html__( 'Subheading', 'bixola-core' ),
                'description' => bixola_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'Team Members', 'bixola-core' ),
                'label_block' => true,
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3', 'layout-4' ],
                ],
            ]
        );

        $this->add_control(
            'section_title',
            [
                'label' => esc_html__( 'Title', 'bixola-core' ),
                'description' => bixola_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'Meet Our Experts', 'bixola-core' ),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'section_description',
            [
                'label' => esc_html__( 'Description', 'bixola-core' ),
                'description' => bixola_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXTAREA,
                'default' => __( 'Lorem ipsum dolor sit amet labore et consectetuer adipiscing elit sed do eiusmod tempor incididunt.', 'bixola-core' ),
                'label_block' => true,
                'condition' => [
                    'design_style' => [ 'layout-2' ],
                ],
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
            '_content_teams',
            [
                'label' => esc_html__('Members', 'bixola-core'),
            ]
        );

        $repeater = new Repeater();

        $repeater->start_controls_tabs( '_tab_style_member_box_item' );

        $repeater->start_controls_tab(
            'tab_member_info',
            [
                'label' => __( 'Information', 'bixola-core' ),
            ]
        );

        $repeater->add_control(
            'image',
            [
                'type' => Controls_Manager::MEDIA,
                'label' => __( 'Image', 'bixola-core' ),
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );                      

        $repeater->add_control(
            'name',
            [
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'label' => __( 'Name', 'bixola-core' ),
                'default' => __( 'Courtney Henry', 'bixola-core' ),
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );

        $repeater->add_control(
            'designation',
            [
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'show_label' => true,
                'label' => __( 'Designation', 'bixola-core' ),
                'default' => __( 'Software Tester', 'bixola-core' ),
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );

        $repeater->add_control(
            'bixola_link_switcher',
            [
                'label' => esc_html__( 'Show Link', 'bixola-core' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Yes', 'bixola-core' ),
                'label_off' => esc_html__( 'No', 'bixola-core' ),
                'return_value' => 'yes',
                'default' => 'yes',
                'separator' => 'before',
            ]
        );

        $repeater->add_control(
            'bixola_link_type',
            [
                'label' => esc_html__( 'Service Link Type', 'bixola-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '1' => 'Custom Link',
                    '2' => 'Internal Page',
                ],
                'default' => '1',
                'condition' => [
                    'bixola_link_switcher' => 'yes'
                ]
            ]
        );

        $repeater->add_control(
            'bixola_link',
            [
                'label' => esc_html__( 'Service Link link', 'bixola-core' ),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => esc_html__( 'https://your-link.com', 'bixola-core' ),
                'show_external' => true,
                'default' => [
                    'url' => '#',
                    'is_external' => false,
                    'nofollow' => false,
                ],
                'condition' => [
                    'bixola_link_type' => '1',
                    'bixola_link_switcher' => 'yes',
                ]
            ]
        );

        $repeater->add_control(
            'bixola_page_link',
            [
                'label' => esc_html__( 'Select Service Link Page', 'bixola-core' ),
                'type' => Controls_Manager::SELECT2,
                'label_block' => true,
                'options' => bixola_get_all_pages(),
                'condition' => [
                    'bixola_link_type' => '2',
                    'bixola_link_switcher' => 'yes',
                ]
            ]
        );

        $repeater->add_control(
            'animation_delay',
            [
                'label' => esc_html__( 'Animation Delay', 'bixola-core' ),
                'type' => Controls_Manager::SELECT,
                'default' => '100ms',
                'options' => [
                    '100ms' => esc_html__( '100ms', 'bixola-core' ),
                    '200ms' => esc_html__( '200ms', 'bixola-core' ),
                    '300ms' => esc_html__( '300ms', 'bixola-core' ),
                    '400ms' => esc_html__( '400ms', 'bixola-core' ),
                    '500ms' => esc_html__( '500ms', 'bixola-core' ),
                    '600ms' => esc_html__( '600ms', 'bixola-core' ),
                    '700ms' => esc_html__( '700ms', 'bixola-core' ),
                    '800ms' => esc_html__( '800ms', 'bixola-core' ),
                    '900ms' => esc_html__( '900ms', 'bixola-core' ),
                ],
            ]
        );

        $repeater->end_controls_tab();

        $repeater->start_controls_tab(
            '_tab_member_links',
            [
                'label' => __( 'Links', 'bixola-core' ),
            ]
        );

        $repeater->add_control(
            'show_social',
            [
                'label' => __( 'Show Options?', 'bixola-core' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => __( 'Yes', 'bixola-core' ),
                'label_off' => __( 'No', 'bixola-core' ),
                'return_value' => 'yes',
                'style_transfer' => true,
            ]
        );

        $repeater->add_control(
            'web_title',
            [
                'type' => Controls_Manager::TEXT,
                'label_block' => false,
                'label' => __( 'Website Address', 'bixola-core' ),
                'placeholder' => __( 'Add your profile link', 'bixola-core' ),
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );        

        $repeater->add_control(
            'email_title',
            [
                'type' => Controls_Manager::TEXT,
                'label_block' => false,
                'label' => __( 'Email', 'bixola-core' ),
                'placeholder' => __( 'Add your email link', 'bixola-core' ),
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );           

        $repeater->add_control(
            'phone_title',
            [
                'type' => Controls_Manager::TEXT,
                'label_block' => false,
                'label' => __( 'Phone', 'bixola-core' ),
                'placeholder' => __( 'Add your phone link', 'bixola-core' ),
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );        

        $repeater->add_control(
            'facebook_title',
            [
                'type' => Controls_Manager::TEXT,
                'label_block' => false,
                'label' => __( 'Facebook', 'bixola-core' ),
                'default' => __( '#', 'bixola-core' ),
                'placeholder' => __( 'Add your facebook link', 'bixola-core' ),
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );                

        $repeater->add_control(
            'twitter_title',
            [
                'type' => Controls_Manager::TEXT,
                'label_block' => false,
                'label' => __( 'Twitter', 'bixola-core' ),
                'default' => __( '#', 'bixola-core' ),
                'placeholder' => __( 'Add your twitter link', 'bixola-core' ),
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );

        $repeater->add_control(
            'instagram_title',
            [
                'type' => Controls_Manager::TEXT,
                'label_block' => false,
                'label' => __( 'Instagram', 'bixola-core' ),
                'default' => __( '#', 'bixola-core' ),
                'placeholder' => __( 'Add your instagram link', 'bixola-core' ),
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );       

        $repeater->add_control(
            'linkedin_title',
            [
                'type' => Controls_Manager::TEXT,
                'label_block' => false,
                'label' => __( 'LinkedIn', 'bixola-core' ),
                'placeholder' => __( 'Add your linkedin link', 'bixola-core' ),
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );        

        $repeater->add_control(
            'youtube_title',
            [
                'type' => Controls_Manager::TEXT,
                'label_block' => false,
                'label' => __( 'Youtube', 'bixola-core' ),
                'placeholder' => __( 'Add your youtube link', 'bixola-core' ),
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );    

        $repeater->add_control(
            'flickr_title',
            [
                'type' => Controls_Manager::TEXT,
                'label_block' => false,
                'label' => __( 'Flickr', 'bixola-core' ),
                'placeholder' => __( 'Add your flickr link', 'bixola-core' ),
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );        

        $repeater->add_control(
            'vimeo_title',
            [
                'type' => Controls_Manager::TEXT,
                'label_block' => false,
                'label' => __( 'Vimeo', 'bixola-core' ),
                'placeholder' => __( 'Add your vimeo link', 'bixola-core' ),
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );

        $repeater->add_control(
            'behance_title',
            [
                'type' => Controls_Manager::TEXT,
                'label_block' => false,
                'label' => __( 'Behance', 'bixola-core' ),
                'placeholder' => __( 'Add your hehance link', 'bixola-core' ),
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );        

        $repeater->add_control(
            'dribble_title',
            [
                'type' => Controls_Manager::TEXT,
                'label_block' => false,
                'label' => __( 'Dribbble', 'bixola-core' ),
                'placeholder' => __( 'Add your dribbble link', 'bixola-core' ),
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );        

        $repeater->add_control(
            'pinterest_title',
            [
                'type' => Controls_Manager::TEXT,
                'label_block' => false,
                'label' => __( 'Pinterest', 'bixola-core' ),
                'placeholder' => __( 'Add your pinterest link', 'bixola-core' ),
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );

        $repeater->add_control(
            'gitub_title',
            [
                'type' => Controls_Manager::TEXT,
                'label_block' => false,
                'label' => __( 'Github', 'bixola-core' ),
                'placeholder' => __( 'Add your github link', 'bixola-core' ),
                'dynamic' => [
                    'active' => true,
                ]
            ]
        ); 

        $repeater->end_controls_tab();

        $repeater->end_controls_tabs();

        // REPEATER
        $this->add_control(
            'team_list',
            [
                'show_label' => false,
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'title_field' => '<# print(name || "Carousel Item"); #>',
                'default' => [
                    [
                        'image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                    ],
                    [
                        'image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                    ],
                    [
                        'image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                    ],
                    [
                        'image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                    ],
                    [
                        'image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                    ]
                ]
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_content_button',
            [
                'label' => esc_html__( 'Button', 'bixola-core' ),
                'tab' => Controls_Manager::TAB_CONTENT,
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3', 'layout-4' ],
                ],
            ]
        );
        
        $this->add_control(
            'bixola_button_text',
            [
                'label' => esc_html__( 'Button Text', 'bixola-core' ),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__( 'View All', 'bixola-core' ),
                'label_block' => true,
            ]
        );
        
        $this->add_control(
            'bixola_button_link_type',
            [
                'label' => esc_html__( 'Button Link Type', 'bixola-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '1' => 'Custom Link',
                    '2' => 'Internal Page',
                ],
                'default' => '1',
                'label_block' => true,
            ]
        );
        
        $this->add_control(
            'bixola_button_link',
            [
                'label' => esc_html__( 'Button link', 'bixola-core' ),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => esc_html__('https://your-link.com', 'bixola-core'),
                'show_external' => false,
                'default' => [
                    'url' => '#',
                    'is_external' => true,
                    'nofollow' => true,
                    'custom_attributes' => '',
                ],
                'condition' => [
                    'bixola_button_link_type' => '1',
                ],
                'label_block' => true,
            ]
        );

        $this->add_control(
            'bixola_button_page_link',
            [
                'label' => esc_html__( 'Select Button Page', 'bixola-core' ),
                'type' => Controls_Manager::SELECT2,
                'label_block' => true,
                'options' => bixola_get_all_pages(),
                'condition' => [
                    'bixola_button_link_type' => '2',
                ]
            ]
        );
        
        $this->add_control(
            'bixola_icon',
            [
                'label' => esc_html__( 'Icon', 'bixola-core' ),
                'type' => Controls_Manager::ICONS,
                'fa4compatibility' => 'icon',
                'label_block' => true,
                'default' => [
                    'value' => 'icon-right-arrow1',
                    'library' => 'solid',
                ],
            ]
        );
        
        $this->add_control(
            'bixola_button_icon_alignment',
            [
                'label'   => esc_html__( 'Icon Position', 'bixola-core' ),
                'type'    => Controls_Manager::SELECT,
                'default' => 'before-icon',
                'options' => [
                    'before-icon' => esc_html__( 'Before', 'bixola-core' ),
                    'after-icon'  => esc_html__( 'After', 'bixola-core' ),
                ],
            ]
        );
        
        $this->add_control(
            'bixola_button_icon_spacing',
            [
                'label'     => esc_html__( 'Icon Spacing', 'bixola-core' ),
                'type'      => Controls_Manager::SLIDER,
                'range'     => [
                    'px' => [
                        'max' => 100,
                    ],
                ],
                'default'   => [
                    'size' => 16,
                ],
                'selectors' => [
                    '{{WRAPPER}} .before-icon span' => 'padding-right: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .before-icon i' => 'padding-right: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .after-icon span'  => 'padding-left: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .after-icon i'  => 'padding-left: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        
        $this->add_control(
            'bixola_button_icon_size',
            [
                'label'     => esc_html__( 'Icon Size', 'bixola-core' ),
                'type'      => Controls_Manager::SLIDER,
                'range'     => [
                    'px' => [
                        'max' => 100,
                    ],
                ],
                'default'   => [
                    'size' => 14,
                ],
                'selectors' => [
                    '{{WRAPPER}} .before-icon span' => 'font-size: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .before-icon i' => 'font-size: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .after-icon span'  => 'font-size: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .after-icon i'  => 'font-size: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
            '_content_column',
            [
                'label' => esc_html__( 'Column',  'delishs-core'  ),
                'tab' => Controls_Manager::TAB_CONTENT,
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3', 'layout-4', 'layout-5' ],
                ],
            ]
        );
        
        $this->add_control(
            'column_desktop',
            [
                'label' => esc_html__( 'Columns for Desktop', 'delishs-core' ),
                'description' => esc_html__( 'Screen width equal to or greater than 1200px', 'delishs-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    12 => esc_html__( '1 Columns', 'delishs-core' ),
                    6 => esc_html__( '2 Columns', 'delishs-core' ),
                    4 => esc_html__( '3 Columns', 'delishs-core' ),
                    3 => esc_html__( '4 Columns', 'delishs-core' ),
                    5 => esc_html__( '5 Columns (For Carousel Item)', 'delishs-core' ),
                    2 => esc_html__( '6 Columns', 'delishs-core' ),
                    1 => esc_html__( '12 Columns', 'delishs-core' ),
                ],
                'separator' => 'before',
                'default' => 4,
                'style_transfer' => true,
            ]
        );

        $this->add_control(
            'column_laptop',
            [
                'label' => esc_html__( 'Columns for Large', 'delishs-core' ),
                'description' => esc_html__( 'Screen width equal to or greater than 992px', 'delishs-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    12 => esc_html__( '1 Columns', 'delishs-core' ),
                    6 => esc_html__( '2 Columns', 'delishs-core' ),
                    4 => esc_html__( '3 Columns', 'delishs-core' ),
                    3 => esc_html__( '4 Columns', 'delishs-core' ),
                    5 => esc_html__( '5 Columns (For Carousel Item)', 'delishs-core' ),
                    2 => esc_html__( '6 Columns', 'delishs-core' ),
                    1 => esc_html__( '12 Columns', 'delishs-core' ),
                ],
                'separator' => 'before',
                'default' => 4,
                'style_transfer' => true,
            ]
        );

        $this->add_control(
            'column_tablet',
            [
                'label' => esc_html__( 'Columns for Tablet', 'delishs-core' ),
                'description' => esc_html__( 'Screen width equal to or greater than 768px', 'delishs-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    12 => esc_html__( '1 Columns', 'delishs-core' ),
                    6 => esc_html__( '2 Columns', 'delishs-core' ),
                    4 => esc_html__( '3 Columns', 'delishs-core' ),
                    3 => esc_html__( '4 Columns', 'delishs-core' ),
                    5 => esc_html__( '5 Columns (For Carousel Item)', 'delishs-core' ),
                    2 => esc_html__( '6 Columns', 'delishs-core' ),
                    1 => esc_html__( '12 Columns', 'delishs-core' ),
                ],
                'separator' => 'before',
                'default' => 6,
                'style_transfer' => true,
            ]
        );

        $this->add_control(
            'column_mobile',
            [
                'label' => esc_html__( 'Columns for Mobile', 'delishs-core' ),
                'description' => esc_html__( 'Screen width less than 767px', 'delishs-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    12 => esc_html__( '1 Columns', 'delishs-core' ),
                    6 => esc_html__( '2 Columns', 'delishs-core' ),
                    4 => esc_html__( '3 Columns', 'delishs-core' ),
                    3 => esc_html__( '4 Columns', 'delishs-core' ),
                    5 => esc_html__( '5 Columns (For Carousel Item)', 'delishs-core' ),
                    2 => esc_html__( '6 Columns', 'delishs-core' ),
                    1 => esc_html__( '12 Columns', 'delishs-core' ),
                ],
                'separator' => 'before',
                'default' => 12,
                'style_transfer' => true,
            ]
        );
        
        $this->end_controls_section();

		$this->start_controls_section(
			'design_layout_style',
			[
				'label' => __( 'Design Layout', 'bixola-core' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

        $this->add_responsive_control(
            'design_layout_margin',
            [
                'label' => esc_html__( 'Margin', 'bixola-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-section' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'design_layout_padding',
            [
                'label' => esc_html__( 'Padding', 'bixola-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-section' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'design_layout_background',
            [
                'label' => esc_html__( 'Background', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-section' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            '_heading_arrow_dots',
            [
                'label' => esc_html__( 'Arrow & Dots', 'bixola-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
                'condition' => [
                    'design_style' => [ 'layout-2', 'layout-6' ],
                ],
            ]
        );

        $this->start_controls_tabs( '_tabs_arrow' );
        
        $this->start_controls_tab(
            'arrow_normal_tab',
            [
                'label' => esc_html__( 'Normal', 'bixola-core' ),
                'condition' => [
                    'design_style' => [ 'layout-2', 'layout-6' ],
                ],
            ]
        );

        $this->add_control(
            'dots_color',
            [
                'label' => esc_html__( 'Color', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .team-two__carousel.owl-theme .owl-nav .owl-next' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .team-two__carousel.owl-theme .owl-nav .owl-prev' => 'color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => [ 'layout-2', 'layout-6' ],
                ],
            ]
        );
        
        $this->add_control(
            'dot_background',
            [
                'label' => esc_html__( 'Background', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .team-two__carousel.owl-theme .owl-nav .owl-next' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .team-two__carousel.owl-theme .owl-nav .owl-prev' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .carousel-dot-style.owl-carousel .owl-dots .owl-dot' => 'background-color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => [ 'layout-2', 'layout-6' ],
                ],
            ]
        );

        $this->add_control(
            'dots_border_color',
            [
                'label' => esc_html__( 'Border Color', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .team-two__carousel.owl-theme .owl-nav .owl-next' => 'border-color: {{VALUE}}',
                    '{{WRAPPER}} .team-two__carousel.owl-theme .owl-nav .owl-prev' => 'border-color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => [ 'layout-2', 'layout-6' ],
                ],
            ]
        );

        $this->end_controls_tab();
        
        $this->start_controls_tab(
            'arrow_hover_tab',
            [
                'label' => esc_html__( 'Hover', 'bixola-core' ),
                'condition' => [
                    'design_style' => [ 'layout-2', 'layout-6' ],
                ],
            ]
        );

        $this->add_control(
            'dots_color_hover',
            [
                'label' => esc_html__( 'Color', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .team-two__carousel.owl-theme .owl-nav .owl-next:hover' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .team-two__carousel.owl-theme .owl-nav .owl-prev:hover' => 'color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => [ 'layout-2', 'layout-6' ],
                ],
            ]
        );

        $this->add_control(
            'dot_background_hover',
            [
                'label' => esc_html__( 'Background', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .team-two__carousel.owl-theme .owl-nav .owl-next:hover' => 'background-color: {{VALUE}}!important',
                    '{{WRAPPER}} .team-two__carousel.owl-theme .owl-nav .owl-prev:hover' => 'background-color: {{VALUE}}!important',
                    '{{WRAPPER}} .carousel-dot-style.owl-carousel .owl-dot.active' => 'background-color: {{VALUE}}!important',
                ],
                'condition' => [
                    'design_style' => [ 'layout-2', 'layout-6' ],
                ],
            ]
        );

        $this->add_control(
            'dots_border_color_hover',
            [
                'label' => esc_html__( 'Border Color', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .team-two__carousel.owl-theme .owl-nav .owl-next:hover' => 'border-color: {{VALUE}}',
                    '{{WRAPPER}} .team-two__carousel.owl-theme .owl-nav .owl-prev:hover' => 'border-color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => [ 'layout-2', 'layout-6' ],
                ],
            ]
        );
        
        $this->end_controls_tab();
        
        $this->end_controls_tabs();

		$this->end_controls_section();

        $this->start_controls_section(
            '_style_title',
            [
                'label' => esc_html__( 'Title & Content',  'bixola-core'  ),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            '_heading_style_section_title',
            [
                'label' => esc_html__( 'Title', 'bixola-core' ),
                'type' => Controls_Manager::HEADING,
            ]
        );
        
        $this->add_responsive_control(
            'section_title_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'bixola-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .section-title__title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .section__title-two' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .section-title-three__title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        
        $this->add_control(
            'section_title_color',
            [
                'label' => __( 'Color', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .section-title__title' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .section__title-two' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .section-title-three__title' => 'color: {{VALUE}}',
                ],
            ]
        );
        
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'section_title_typography',
                'selector' => '{{WRAPPER}} .section-title__title, .section__title-two, .section-title-three__title',
            ]
        );
        
        $this->add_control(
            '_heading_style_section_subheading',
            [
                'label' => esc_html__( 'Subheading', 'bixola-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3' ],
                ],
            ]
        );
        
        $this->add_responsive_control(
            'section_subheading_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'bixola-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .section-title__tagline' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .section-title-three__tagline' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3' ],
                ],
            ]
        );
        
        $this->add_control(
            'section_subheading_color',
            [
                'label' => __( 'Color', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .section-title__tagline' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .section-title-three__tagline' => 'color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3' ],
                ],
            ]
        );
        
        $this->add_control(
            'section_subheading_border_color',
            [
                'label' => __( 'Border Color', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .section-title__tagline::before' => 'background-color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1' ],
                ],
            ]
        );
        
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'section_subheading_typography',
                'selector' => '{{WRAPPER}} .section-title__tagline, .section-title-three__tagline',
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3' ],
                ],
            ]
        );

        $this->add_control(
            '_heading_style_section_description',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Description', 'bixola-core' ),
                'separator' => 'before',
                'condition' => [
                    'design_style' => [ 'layout-2' ],
                ],
            ]
        );
        
        $this->add_responsive_control(
            'section_description_spacing',
            [
                'label' => __( 'Bottom Spacing', 'bixola-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .team-two__text' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'design_style' => [ 'layout-2' ],
                ],
            ]
        );
        
        $this->add_control(
            'section_description_color',
            [
                'label' => __( 'Color', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .team-two__text' => 'color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => [ 'layout-2' ],
                ],
            ]
        );
        
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'section_description_typography',
                'selector' => '{{WRAPPER}} .team-two__text',
                'condition' => [
                    'design_style' => [ 'layout-2' ],
                ],
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
			'team_member_style',
			[
				'label' => __( 'Members', 'bixola-core' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

        // Icon
        $this->add_control(
            '_heading_icon',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Icon', 'bixola-core' ),
            ]
        );

        $this->start_controls_tabs( 'tabs_icon_style' );

        $this->start_controls_tab(
            'icon_tab',
            [
                'label' => esc_html__( 'Normal', 'bixola-core' ),
            ]
        );

        $this->add_control(
            'icon_color',
            [
                'label'     => esc_html__( 'Color', 'bixola-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .team-one__social a'    => 'color: {{VALUE}}',
                    '{{WRAPPER}} .team-two__social a'    => 'color: {{VALUE}}',
                    '{{WRAPPER}} .team-three__social a'    => 'color: {{VALUE}}',
                    '{{WRAPPER}} .team-six__social a'    => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'icon_background',
            [
                'label'     => esc_html__( 'Background', 'bixola-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .team-one__social a' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .team-three__social a' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .team-six__social a' => 'background-color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3', 'layout-4', 'layout-5', 'layout-6' ],
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'label'    => esc_html__( 'Border', 'bixola-core' ),
                'name'     => 'icon_border',
                'selector' => '{{WRAPPER}} .team-one__social a, .team-three__social a, .team-six__social a',
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3', 'layout-4', 'layout-5', 'layout-6' ],
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'icon_hover_tab',
            [
                'label' => esc_html__( 'Hover', 'bixola-core' ),
            ]
        );

        $this->add_control(
            'icon_color_hover',
            [
                'label'     => esc_html__( 'Color', 'bixola-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .team-one__social a:hover' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .team-two__social a:hover' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .team-three__social a:hover' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .team-six__social a:hover' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'icon_background_hover',
            [
                'label'     => esc_html__( 'Background', 'bixola-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .team-one__social a:hover' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .team-three__social a:hover' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .team-six__social a:hover' => 'background-color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3', 'layout-4', 'layout-5', 'layout-6' ],
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'label'    => esc_html__( 'Border', 'bixola-core' ),
                'name'     => 'icon_border_hover',
                'selector' => '{{WRAPPER}} .team-one__social a:hover, .team-three__social a:hover, .team-six__social a:hover',
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3', 'layout-4', 'layout-5', 'layout-6' ],
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        // Name
        $this->add_control(
            '_heading_member_name',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Name', 'bixola-core' ),
                'separator' => 'before'
            ]
        );

        $this->add_responsive_control(
            'member_name_spacing',
            [
                'label' => __( 'Bottom Spacing', 'bixola-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .team-one__name' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .team-two__name' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .team-three__title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .team-six__title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

		$this->add_control(
            'member_name_color',
            [
                'label' => __( 'Color', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .team-one__name a' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .team-two__name a' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .team-three__title a' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .team-six__title a' => 'color: {{VALUE}}',
                ],
            ]
        );

		$this->add_control(
            'member_name_color_hover',
            [
                'label' => __( 'Color (Hover)', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .team-one__name a:hover' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .team-two__name a:hover' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .team-three__title a:hover' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .team-six__title a:hover' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'member_name_typography',
                'selector' => '{{WRAPPER}} .team-one__name, .team-two__name, .team-three__title, .team-six__title',
            ]
        );

        // Name
        $this->add_control(
            '_heading_member_designation',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Designation', 'bixola-core' ),
                'separator' => 'before'
            ]
        );

		$this->add_control(
            'member_designation_color',
            [
                'label' => __( 'Color', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .team-one__sub-title' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .team-two__sub-title' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .team-three__sub-title' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .team-six__sub-title' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'member_designation_typography',
                'selector' => '{{WRAPPER}} .team-one__sub-title, .team-two__sub-title, .team-three__sub-title, .team-six__sub-title',
            ]
        );

        $this->add_control(
            '_heading_name_layout',
            [
                'label' => esc_html__( 'Layout', 'bixola-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-2', 'layout-3', 'layout-4', 'layout-5', 'layout-6' ],
                ],
            ]
        );

		$this->add_control(
            'member_name_background',
            [
                'label' => __( 'Background', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .team-one__content' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .team-two__single' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .team-six__content' => 'background-color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-2', 'layout-4', 'layout-5', 'layout-6' ],
                ],
            ]
        );

		$this->add_control(
            'member_image_hover',
            [
                'label' => __( 'Image Hover', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .team-one__img:before' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .team-two__img:before' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .team-three__img:before' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .team-six__img:before' => 'background-color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-2', 'layout-3', 'layout-4', 'layout-5', 'layout-6' ],
                ],
            ]
        );

        $this->add_responsive_control(
            'member_name_padding',
            [
                'label' => __( 'Padding', 'bixola-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .team-one__content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .team-two__single' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .team-six__content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-2', 'layout-4', 'layout-5', 'layout-6' ],
                ],
            ]
        );

        $this->add_responsive_control(
            'member_name_border_radius',
            [
                'label' => __( 'Border Radius', 'bixola-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .team-one__content' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .team-six__content' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-4', 'layout-5', 'layout-6' ],
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name' => 'member_name_border',
                'selector' => '{{WRAPPER}} .team-one__content, .team-two__single, .team-six__content',
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-2', 'layout-4', 'layout-5', 'layout-6' ],
                ],
            ]
        );

		$this->end_controls_section();

        $this->start_controls_section(
            '_style_section_button',
            [
                'label' => esc_html__( 'Button', 'bixola-core' ),
                'tab' => Controls_Manager::TAB_STYLE,
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3', 'layout-4' ],
                ],
            ]
        );
        
        $this->start_controls_tabs( 'tabs_style__button' );
        
        $this->start_controls_tab(
            'section_button_tab',
            [
                'label' => esc_html__( 'Normal', 'bixola-core' ),
            ]
        );
        
        $this->add_control(
            'section_button_color',
            [
                'label'     => esc_html__( 'Color', 'bixola-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .thm-btn'    => 'color: {{VALUE}}',
                    '{{WRAPPER}} .team-three__btn a'    => 'color: {{VALUE}}',
                ],
            ]
        );
        
        $this->add_control(
            'section_button_background',
            [
                'label'     => esc_html__( 'Background', 'bixola-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .thm-btn' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .team-three__btn a' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        
        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'label'    => esc_html__( 'Border', 'bixola-core' ),
                'name'     => 'section_button_border',
                'selector' => '{{WRAPPER}} .thm-btn, .team-three__btn a',
            ]
        );
        
        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name'     => 'section_button_box_shadow',
                'selector' => '{{WRAPPER}} .thm-btn, .team-three__btn a',
            ]
        );
        
        $this->end_controls_tab();
        
        $this->start_controls_tab(
            'section_button_hover_tab',
            [
                'label' => esc_html__( 'Hover', 'bixola-core' ),
            ]
        );
        
        $this->add_control(
            'section_button_color_hover',
            [
                'label'     => esc_html__( 'Color', 'bixola-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .thm-btn:hover' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .team-three__btn a:hover' => 'color: {{VALUE}};',
                ],
            ]
        );
        
        $this->add_control(
            'section_button_background_hover',
            [
                'label'     => esc_html__( 'Background', 'bixola-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .thm-btn:after, .thm-btn:before' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .team-three__btn a:after, .team-three__btn a:before' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        
        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'label'    => esc_html__( 'Border', 'bixola-core' ),
                'name'     => 'section_button_border_hover',
                'selector' => '{{WRAPPER}} .thm-btn:hover, .team-three__btn a:hover',
            ]
        );
        
        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name'     => 'section_button_box_shadow_hover',
                'selector' => '{{WRAPPER}} .thm-btn:hover, .team-three__btn a:hover',
            ]
        );
        
        $this->end_controls_tab();
        
        $this->end_controls_tabs();
        
        $this->add_control(
            'section_button_border_radius',
            [
                'label'     => esc_html__( 'Border Radius', 'bixola-core' ),
                'type'      => Controls_Manager::SLIDER,
                'selectors' => [
                    '{{WRAPPER}} .thm-btn' => 'border-radius: {{SIZE}}px;',
                    '{{WRAPPER}} .thm-btn:before' => 'border-radius: {{SIZE}}px;',
                    '{{WRAPPER}} .team-three__btn a' => 'border-radius: {{SIZE}}px;',
                    '{{WRAPPER}} .team-three__btn a:before' => 'border-radius: {{SIZE}}px;',
                ],
            ]
        );
        
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'label'    => esc_html__( 'Typography', 'bixola-core' ),
                'name'     => 'section_button_typography',
                'selector' => '{{WRAPPER}} .thm-btn, .team-three__btn a',
            ]
        );
        
        $this->add_control(
            'section_button_padding',
            [
                'label'      => esc_html__( 'Padding', 'bixola-core' ),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .thm-btn' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .team-three__btn a' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        
        $this->add_control(
            'section_button_margin',
            [
                'label'      => esc_html__( 'Margin', 'bixola-core' ),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .thm-btn' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .team-three__btn a' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        
        $this->end_controls_section();
	}

	/**
	 * Render the widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();
        $this->add_render_attribute('title_args', 'class', 'section-title__title');
        

		?>

            <?php if ( $settings['design_style']  == 'layout-1' ) : 

                if ('2' == $settings['bixola_button_link_type']) {
                    $this->add_render_attribute('bixola-button-arg', 'href', get_permalink($settings['bixola_button_page_link']));
                    $this->add_render_attribute('bixola-button-arg', 'target', '_self');
                    $this->add_render_attribute('bixola-button-arg', 'rel', 'nofollow');
                    $this->add_render_attribute('bixola-button-arg', 'class', 'text-domain-el-button team-one__btn thm-btn');
                } else {
                    if ( ! empty( $settings['bixola_button_link']['url'] ) ) {
                        $this->add_link_attributes( 'bixola-button-arg', $settings['bixola_button_link'] );
                        $this->add_render_attribute('bixola-button-arg', 'class', 'text-domain-el-button team-one__btn thm-btn');
                    }
                }

                ?>

                <section class="bixola-el-section team-one">
                    <?php if ( !empty( $settings['section_shape_switch'] ) ) : ?>
                        <div class="team-one__shape-1 img-bounce">
                            <img src="<?php print get_template_directory_uri(); ?>/assets/images/shapes/team-one-shape-1.png" alt="<?php print esc_attr( 'shape', 'bixola-core' ); ?>">
                        </div>
                        <div class="team-one__shape-2 zoominout">
                            <img src="<?php print get_template_directory_uri(); ?>/assets/images/shapes/team-one-shape-2.png" alt="<?php print esc_attr( 'shape', 'bixola-core' ); ?>">
                        </div>
                    <?php endif; ?>
                    <div class="container">
                        <div class="section-title text-center">
                            <?php if ( !empty( $settings['section_subheading'] ) ) : ?>
                                <div class="section-title__tagline-box">
                                    <p class="section-title__tagline">
                                        <?php print twinkle_kses($settings['section_subheading']); ?>
                                    </p>
                                </div>
                            <?php endif; ?>
                            <?php if ( !empty( $settings['section_title'] ) ) : ?>
                                <h2 class="section-title__title">
                                    <?php print twinkle_kses($settings['section_title']); ?>
                                </h2>
                            <?php endif; ?>
                        </div>
                        <div class="row">
                            <?php foreach ( $settings['team_list'] as $item ) :

                                if ( !empty($item['image']['url']) ) {
                                    $bixola_team_image_url = !empty($item['image']['id']) ? wp_get_attachment_image_url( $item['image']['id'], 'full') : $item['image']['url'];
                                    $bixola_team_image_alt = get_post_meta($item["image"]["id"], "_wp_attachment_image_alt", true);
                                }
                                
                                // Link
                                if ('2' == $item['bixola_link_type']) {
                                    $link = get_permalink($item['bixola_page_link']);
                                    $target = '_self';
                                    $rel = 'nofollow';
                                } else {
                                    $link = !empty($item['bixola_link']['url']) ? $item['bixola_link']['url'] : '';
                                    $target = !empty($item['bixola_link']['is_external']) ? '_blank' : '';
                                    $rel = !empty($item['bixola_link']['nofollow']) ? 'nofollow' : '';
                                }

                            ?>
                            <div class="col-xl-<?php print esc_attr($settings['column_desktop']); ?> col-lg-<?php print esc_attr($settings['column_laptop']); ?> col-md-<?php print esc_attr($settings['column_tablet']); ?> col-sm-<?php print esc_attr($settings['column_mobile']); ?> wow fadeInUp" data-wow-delay="<?php print esc_attr($item['animation_delay']); ?>">
                                <div class="team-one__single">
                                    <div class="team-one__img-box">
                                        <div class="team-one__img">
                                            <img src="<?php echo esc_url($bixola_team_image_url); ?>" alt="<?php echo esc_attr($bixola_team_image_alt); ?>">
                                        </div>
                                        <div class="team-one__content">
                                            <h3 class="team-one__name">
                                                <a target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>" href="<?php echo esc_url($link); ?>">
                                                    <?php echo twinkle_kses( $item['name'] ); ?>
                                                </a>
                                            </h3>
                                            <?php if( !empty($item['designation']) ) : ?>
                                                <p class="team-one__sub-title">
                                                    <?php echo bixola_kses( $item['designation'] ); ?>
                                                </p>
                                            <?php endif; ?>
                                        </div>
                                        <div class="team-one__social">
                                            <?php if( !empty($item['facebook_title'] ) ) : ?>
                                                <a href="<?php echo esc_url( $item['facebook_title'] ); ?>"><i class="fa-brands fa-facebook-f"></i></a>
                                            <?php endif; ?>

                                            <?php if( !empty($item['twitter_title'] ) ) : ?>
                                                <a href="<?php echo esc_url( $item['twitter_title'] ); ?>"><i class="fa-brands fa-x-twitter"></i></a>
                                            <?php endif; ?>

                                            <?php if( !empty($item['pinterest_title'] ) ) : ?>
                                                <a href="<?php echo esc_url( $item['pinterest_title'] ); ?>"><i class="fa-brands fa-pinterest-p"></i></a>
                                            <?php endif; ?>

                                            <?php if( !empty($item['instagram_title'] ) ) : ?>
                                                <a href="<?php echo esc_url( $item['instagram_title'] ); ?>"><i class="fa-brands fa-instagram"></i></a>
                                            <?php endif; ?>

                                            <?php if( !empty($item['linkedin_title'] ) ) : ?>
                                                <a href="<?php echo esc_url( $item['linkedin_title'] ); ?>"><i class="fa-brands fa-linkedin-in"></i></a>
                                            <?php endif; ?>

                                            <?php if( !empty($item['youtube_title'] ) ) : ?>
                                                <a href="<?php echo esc_url( $item['youtube_title'] ); ?>"><i class="fa-brands fa-youtube"></i></a>
                                            <?php endif; ?>

                                            <?php if( !empty($item['flickr_title'] ) ) : ?>
                                                <a href="<?php echo esc_url( $item['flickr_title'] ); ?>"><i class="fa-brands fa-flickr"></i></a>
                                            <?php endif; ?>

                                            <?php if( !empty($item['vimeo_title'] ) ) : ?>
                                                <a href="<?php echo esc_url( $item['vimeo_title'] ); ?>"><i class="fa-brands fa-vimeo-v"></i></a>
                                            <?php endif; ?>

                                            <?php if( !empty($item['behance_title'] ) ) : ?>
                                                <a href="<?php echo esc_url( $item['behance_title'] ); ?>"><i class="fa-brands fa-behance"></i></a>
                                            <?php endif; ?>

                                            <?php if( !empty($item['dribble_title'] ) ) : ?>
                                                <a href="<?php echo esc_url( $item['dribble_title'] ); ?>"><i class="fa-brands fa-dribbble"></i></a>
                                            <?php endif; ?>

                                            <?php if( !empty($item['phone_title'] ) ) : ?>
                                                <a href="tel:<?php echo esc_url( $item['phone_title'] ); ?>"><i class="fa-solid fa-phone"></i></a>
                                            <?php endif; ?>  

                                            <?php if( !empty($item['gitub_title'] ) ) : ?>
                                                <a href="<?php echo esc_url( $item['gitub_title'] ); ?>"><i class="fa-brands fa-github"></i></a>
                                            <?php endif; ?>

                                            <?php if( !empty($item['web_title'] ) ) : ?>
                                                <a href="<?php echo esc_url( $item['web_title'] ); ?>"><i class="fa-solid fa-globe"></i></a>
                                            <?php endif; ?>  

                                            <?php if( !empty($item['email_title'] ) ) : ?>
                                                <a href="mailto:<?php echo esc_url( $item['email_title'] ); ?>"><i class="fa-solid fa-envelope"></i></a>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                        <div class="team-one__btn-box">
                            <a <?php echo $this->get_render_attribute_string( 'bixola-button-arg' ); ?>>
                                <?php if ( !empty($settings['bixola_icon']['value']) ) : ?>
                                    <span class="<?php echo esc_attr( $settings['bixola_button_icon_alignment'] ); ?>">
                                        <?php bixola_render_icon($settings, 'bixola_icon'); ?>
                                    </span>
                                <?php endif; ?>
                                <?php print twinkle_kses($settings['bixola_button_text']); ?>
                            </a>
                        </div>
                    </div>
                </section>

            <?php elseif ( $settings['design_style']  == 'layout-2' ): ?>

                <section class="bixola-el-section team-two">
                    <div class="container">
                        <div class="row">
                            <div class="col-xl-4">
                                <div class="team-two__left">
                                    <?php if ( !empty( $settings['section_title'] ) ) : ?>
                                        <h3 class="section__title-two">
                                            <?php print twinkle_kses($settings['section_title']); ?>
                                        </h3>
                                    <?php endif; ?>
                                    <?php if ( !empty( $settings['section_description'] ) ) : ?>
                                        <p class="team-two__text">
                                            <?php print twinkle_kses($settings['section_description']); ?>
                                        </p>
                                    <?php endif; ?>
                                </div>
                            </div>
                            <div class="col-xl-8">
                                <div class="team-two__right">
                                    <div class="team-two__carousel owl-carousel owl-theme thm-owl__carousel">
                                        <?php foreach ( $settings['team_list'] as $item ) :

                                            if ( !empty($item['image']['url']) ) {
                                                $bixola_team_image_url = !empty($item['image']['id']) ? wp_get_attachment_image_url( $item['image']['id'], 'full') : $item['image']['url'];
                                                $bixola_team_image_alt = get_post_meta($item["image"]["id"], "_wp_attachment_image_alt", true);
                                            }
                                            
                                            // Link
                                            if ('2' == $item['bixola_link_type']) {
                                                $link = get_permalink($item['bixola_page_link']);
                                                $target = '_self';
                                                $rel = 'nofollow';
                                            } else {
                                                $link = !empty($item['bixola_link']['url']) ? $item['bixola_link']['url'] : '';
                                                $target = !empty($item['bixola_link']['is_external']) ? '_blank' : '';
                                                $rel = !empty($item['bixola_link']['nofollow']) ? 'nofollow' : '';
                                            }

                                        ?>
                                        <div class="item">
                                            <div class="team-two__single">
                                                <?php if( !empty($bixola_team_image_url) ) : ?>
                                                    <div class="team-two__img-box">
                                                        <div class="team-two__img">
                                                            <img src="<?php echo esc_url($bixola_team_image_url); ?>" alt="<?php echo esc_attr($bixola_team_image_alt); ?>">
                                                        </div>
                                                    </div>
                                                <?php endif; ?>
                                                <div class="team-two__content">
                                                    <div class="team-two__name-box">
                                                        <h3 class="team-two__name">
                                                            <a target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>" href="<?php echo esc_url($link); ?>">
                                                                <?php echo bixola_kses( $item['name'] ); ?>
                                                            </a>
                                                        </h3>
                                                        <?php if( !empty($item['designation']) ) : ?>
                                                            <p class="team-two__sub-title">
                                                                <?php echo bixola_kses( $item['designation'] ); ?>
                                                            </p>
                                                        <?php endif; ?>
                                                    </div>
                                                    <div class="team-two__social">
                                                        <?php if( !empty($item['facebook_title'] ) ) : ?>
                                                            <a href="<?php echo esc_url( $item['facebook_title'] ); ?>"><i class="fa-brands fa-facebook-f"></i></a>
                                                        <?php endif; ?>

                                                        <?php if( !empty($item['twitter_title'] ) ) : ?>
                                                            <a href="<?php echo esc_url( $item['twitter_title'] ); ?>"><i class="fa-brands fa-x-twitter"></i></a>
                                                        <?php endif; ?>

                                                        <?php if( !empty($item['pinterest_title'] ) ) : ?>
                                                            <a href="<?php echo esc_url( $item['pinterest_title'] ); ?>"><i class="fa-brands fa-pinterest-p"></i></a>
                                                        <?php endif; ?>

                                                        <?php if( !empty($item['instagram_title'] ) ) : ?>
                                                            <a href="<?php echo esc_url( $item['instagram_title'] ); ?>"><i class="fa-brands fa-instagram"></i></a>
                                                        <?php endif; ?>

                                                        <?php if( !empty($item['linkedin_title'] ) ) : ?>
                                                            <a href="<?php echo esc_url( $item['linkedin_title'] ); ?>"><i class="fa-brands fa-linkedin-in"></i></a>
                                                        <?php endif; ?>

                                                        <?php if( !empty($item['youtube_title'] ) ) : ?>
                                                            <a href="<?php echo esc_url( $item['youtube_title'] ); ?>"><i class="fa-brands fa-youtube"></i></a>
                                                        <?php endif; ?>

                                                        <?php if( !empty($item['flickr_title'] ) ) : ?>
                                                            <a href="<?php echo esc_url( $item['flickr_title'] ); ?>"><i class="fa-brands fa-flickr"></i></a>
                                                        <?php endif; ?>

                                                        <?php if( !empty($item['vimeo_title'] ) ) : ?>
                                                            <a href="<?php echo esc_url( $item['vimeo_title'] ); ?>"><i class="fa-brands fa-vimeo-v"></i></a>
                                                        <?php endif; ?>

                                                        <?php if( !empty($item['behance_title'] ) ) : ?>
                                                            <a href="<?php echo esc_url( $item['behance_title'] ); ?>"><i class="fa-brands fa-behance"></i></a>
                                                        <?php endif; ?>

                                                        <?php if( !empty($item['dribble_title'] ) ) : ?>
                                                            <a href="<?php echo esc_url( $item['dribble_title'] ); ?>"><i class="fa-brands fa-dribbble"></i></a>
                                                        <?php endif; ?>

                                                        <?php if( !empty($item['phone_title'] ) ) : ?>
                                                            <a href="tel:<?php echo esc_url( $item['phone_title'] ); ?>"><i class="fa-solid fa-phone"></i></a>
                                                        <?php endif; ?>  

                                                        <?php if( !empty($item['gitub_title'] ) ) : ?>
                                                            <a href="<?php echo esc_url( $item['gitub_title'] ); ?>"><i class="fa-brands fa-github"></i></a>
                                                        <?php endif; ?>

                                                        <?php if( !empty($item['web_title'] ) ) : ?>
                                                            <a href="<?php echo esc_url( $item['web_title'] ); ?>"><i class="fa-solid fa-globe"></i></a>
                                                        <?php endif; ?>  

                                                        <?php if( !empty($item['email_title'] ) ) : ?>
                                                            <a href="mailto:<?php echo esc_url( $item['email_title'] ); ?>"><i class="fa-solid fa-envelope"></i></a>
                                                        <?php endif; ?>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                        <?php endforeach; ?>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </section>

            <?php elseif ( $settings['design_style']  == 'layout-3' ) : 

                if ('2' == $settings['bixola_button_link_type']) {
                    $this->add_render_attribute('bixola-button-arg', 'href', get_permalink($settings['bixola_button_page_link']));
                    $this->add_render_attribute('bixola-button-arg', 'target', '_self');
                    $this->add_render_attribute('bixola-button-arg', 'rel', 'nofollow');
                    $this->add_render_attribute('bixola-button-arg', 'class', 'text-domain-el-button');
                } else {
                    if ( ! empty( $settings['bixola_button_link']['url'] ) ) {
                        $this->add_link_attributes( 'bixola-button-arg', $settings['bixola_button_link'] );
                        $this->add_render_attribute('bixola-button-arg', 'class', 'text-domain-el-button');
                    }
                }
                
                ?>

                <section class="bixola-el-section team-three">
                    <?php if ( !empty( $settings['section_shape_switch'] ) ) : ?>
                        <div class="team-three__shape-1 img-bounce">
                            <img src="<?php print get_template_directory_uri(); ?>/assets/images/shapes/team-three-shape-1.png" alt="<?php print esc_attr( 'shape', 'bixola' ) ?>">
                        </div>
                        <div class="team-three__shape-2 float-bob-y">
                            <img src="<?php print get_template_directory_uri(); ?>/assets/images/shapes/team-three-shape-2.png" alt="<?php print esc_attr( 'shape', 'bixola' ) ?>">
                        </div>
                    <?php endif; ?>
                    <div class="container">
                        <div class="section-title-three text-center">
                            <?php if ( !empty( $settings['section_subheading'] ) ) : ?>
                                <div class="section-title-three__tagline-box">
                                    <p class="section-title-three__tagline">
                                        <?php print twinkle_kses($settings['section_subheading']); ?>
                                    </p>
                                </div>
                            <?php endif; ?>
                            <?php if ( !empty( $settings['section_title'] ) ) : ?>
                                <h2 class="section-title-three__title">
                                    <?php print twinkle_kses($settings['section_title']); ?>
                                </h2>
                            <?php endif; ?>
                        </div>
                        <div class="row">
                            <?php foreach ( $settings['team_list'] as $item ) :

                                if ( !empty($item['image']['url']) ) {
                                    $bixola_team_image_url = !empty($item['image']['id']) ? wp_get_attachment_image_url( $item['image']['id'], 'full') : $item['image']['url'];
                                    $bixola_team_image_alt = get_post_meta($item["image"]["id"], "_wp_attachment_image_alt", true);
                                }
                                
                                // Link
                                if ('2' == $item['bixola_link_type']) {
                                    $link = get_permalink($item['bixola_page_link']);
                                    $target = '_self';
                                    $rel = 'nofollow';
                                } else {
                                    $link = !empty($item['bixola_link']['url']) ? $item['bixola_link']['url'] : '';
                                    $target = !empty($item['bixola_link']['is_external']) ? '_blank' : '';
                                    $rel = !empty($item['bixola_link']['nofollow']) ? 'nofollow' : '';
                                }

                            ?>
                            <div class="col-xl-<?php print esc_attr($settings['column_desktop']); ?> col-lg-<?php print esc_attr($settings['column_laptop']); ?> col-md-<?php print esc_attr($settings['column_tablet']); ?> col-sm-<?php print esc_attr($settings['column_mobile']); ?> wow fadeInLeft" data-wow-delay="<?php print esc_attr($item['animation_delay']); ?>">
                                <div class="team-three__single">
                                    <div class="team-three__img-box">
                                        <?php if( !empty( $bixola_team_image_url ) ) : ?>
                                            <div class="team-three__img">
                                                <img src="<?php echo esc_url($bixola_team_image_url); ?>" alt="<?php echo esc_attr($bixola_team_image_alt); ?>">
                                            </div>
                                        <?php endif; ?>
                                        <div class="team-three__social">
                                            <?php if( !empty($item['facebook_title'] ) ) : ?>
                                                <a href="<?php echo esc_url( $item['facebook_title'] ); ?>"><i class="fa-brands fa-facebook-f"></i></a>
                                            <?php endif; ?>

                                            <?php if( !empty($item['twitter_title'] ) ) : ?>
                                                <a href="<?php echo esc_url( $item['twitter_title'] ); ?>"><i class="fa-brands fa-x-twitter"></i></a>
                                            <?php endif; ?>

                                            <?php if( !empty($item['pinterest_title'] ) ) : ?>
                                                <a href="<?php echo esc_url( $item['pinterest_title'] ); ?>"><i class="fa-brands fa-pinterest-p"></i></a>
                                            <?php endif; ?>

                                            <?php if( !empty($item['instagram_title'] ) ) : ?>
                                                <a href="<?php echo esc_url( $item['instagram_title'] ); ?>"><i class="fa-brands fa-instagram"></i></a>
                                            <?php endif; ?>

                                            <?php if( !empty($item['linkedin_title'] ) ) : ?>
                                                <a href="<?php echo esc_url( $item['linkedin_title'] ); ?>"><i class="fa-brands fa-linkedin-in"></i></a>
                                            <?php endif; ?>

                                            <?php if( !empty($item['youtube_title'] ) ) : ?>
                                                <a href="<?php echo esc_url( $item['youtube_title'] ); ?>"><i class="fa-brands fa-youtube"></i></a>
                                            <?php endif; ?>

                                            <?php if( !empty($item['flickr_title'] ) ) : ?>
                                                <a href="<?php echo esc_url( $item['flickr_title'] ); ?>"><i class="fa-brands fa-flickr"></i></a>
                                            <?php endif; ?>

                                            <?php if( !empty($item['vimeo_title'] ) ) : ?>
                                                <a href="<?php echo esc_url( $item['vimeo_title'] ); ?>"><i class="fa-brands fa-vimeo-v"></i></a>
                                            <?php endif; ?>

                                            <?php if( !empty($item['behance_title'] ) ) : ?>
                                                <a href="<?php echo esc_url( $item['behance_title'] ); ?>"><i class="fa-brands fa-behance"></i></a>
                                            <?php endif; ?>

                                            <?php if( !empty($item['dribble_title'] ) ) : ?>
                                                <a href="<?php echo esc_url( $item['dribble_title'] ); ?>"><i class="fa-brands fa-dribbble"></i></a>
                                            <?php endif; ?>

                                            <?php if( !empty($item['phone_title'] ) ) : ?>
                                                <a href="tel:<?php echo esc_url( $item['phone_title'] ); ?>"><i class="fa-solid fa-phone"></i></a>
                                            <?php endif; ?>  

                                            <?php if( !empty($item['gitub_title'] ) ) : ?>
                                                <a href="<?php echo esc_url( $item['gitub_title'] ); ?>"><i class="fa-brands fa-github"></i></a>
                                            <?php endif; ?>

                                            <?php if( !empty($item['web_title'] ) ) : ?>
                                                <a href="<?php echo esc_url( $item['web_title'] ); ?>"><i class="fa-solid fa-globe"></i></a>
                                            <?php endif; ?>  

                                            <?php if( !empty($item['email_title'] ) ) : ?>
                                                <a href="mailto:<?php echo esc_url( $item['email_title'] ); ?>"><i class="fa-solid fa-envelope"></i></a>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                    <div class="team-three__content">
                                        <h3 class="team-three__title">
                                            <a target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>" href="<?php echo esc_url($link); ?>">
                                                <?php echo bixola_kses( $item['name'] ); ?>
                                            </a>
                                        </h3>
                                        <?php if( !empty($item['designation']) ) : ?>
                                            <p class="team-three__sub-title"><?php echo bixola_kses( $item['designation'] ); ?></p>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                        <div class="team-three__btn-box">
                            <div class="team-three__btn">
                                <a <?php echo $this->get_render_attribute_string( 'bixola-button-arg' ); ?>>
                                    <?php if ( !empty($settings['bixola_icon']['value']) ) : ?>
                                        <span class="<?php echo esc_attr( $settings['bixola_button_icon_alignment'] ); ?>">
                                            <?php bixola_render_icon($settings, 'bixola_icon'); ?>
                                        </span>
                                    <?php endif; ?>
                                    <?php print twinkle_kses($settings['bixola_button_text']); ?>
                                </a>
                            </div>
                        </div>
                    </div>
                </section>

            <?php elseif ( $settings['design_style']  == 'layout-4' ) : 
                if ('2' == $settings['bixola_button_link_type']) {
                    $this->add_render_attribute('bixola-button-arg', 'href', get_permalink($settings['bixola_button_page_link']));
                    $this->add_render_attribute('bixola-button-arg', 'target', '_self');
                    $this->add_render_attribute('bixola-button-arg', 'rel', 'nofollow');
                    $this->add_render_attribute('bixola-button-arg', 'class', 'text-domain-el-button team-six__btn thm-btn');
                } else {
                    if ( ! empty( $settings['bixola_button_link']['url'] ) ) {
                        $this->add_link_attributes( 'bixola-button-arg', $settings['bixola_button_link'] );
                        $this->add_render_attribute('bixola-button-arg', 'class', 'text-domain-el-button team-six__btn thm-btn');
                    }
                }
                
                ?>

                <section class="bixola-el-section team-six">
                    <div class="container">
                        <div class="team-six__top">
                            <div class="team-six__left">
                                <div class="section-title-three text-left">
                                    <?php if ( !empty( $settings['section_subheading'] ) ) : ?>
                                        <div class="section-title-three__tagline-box">
                                            <p class="section-title-three__tagline">
                                                <?php print twinkle_kses($settings['section_subheading']); ?>
                                            </p>
                                        </div>
                                    <?php endif; ?>
                                    <?php if ( !empty( $settings['section_title'] ) ) : ?>
                                        <h2 class="section-title-three__title">
                                            <?php print twinkle_kses($settings['section_title']); ?>
                                        </h2>
                                    <?php endif; ?>
                                </div>
                            </div>
                            <div class="team-six__btn-box">
                                <a <?php echo $this->get_render_attribute_string( 'bixola-button-arg' ); ?>>
                                    <?php if ( !empty($settings['bixola_icon']['value']) ) : ?>
                                        <span class="<?php echo esc_attr( $settings['bixola_button_icon_alignment'] ); ?>">
                                            <?php bixola_render_icon($settings, 'bixola_icon'); ?>
                                        </span>
                                    <?php endif; ?>
                                    <?php print twinkle_kses($settings['bixola_button_text']); ?>
                                </a>
                            </div>
                        </div>
                        <div class="team-six__bottom">
                            <div class="row">
                                <?php foreach ( $settings['team_list'] as $item ) :

                                    if ( !empty($item['image']['url']) ) {
                                        $bixola_team_image_url = !empty($item['image']['id']) ? wp_get_attachment_image_url( $item['image']['id'], 'full') : $item['image']['url'];
                                        $bixola_team_image_alt = get_post_meta($item["image"]["id"], "_wp_attachment_image_alt", true);
                                    }

                                    // Link
                                    if ('2' == $item['bixola_link_type']) {
                                        $link = get_permalink($item['bixola_page_link']);
                                        $target = '_self';
                                        $rel = 'nofollow';
                                    } else {
                                        $link = !empty($item['bixola_link']['url']) ? $item['bixola_link']['url'] : '';
                                        $target = !empty($item['bixola_link']['is_external']) ? '_blank' : '';
                                        $rel = !empty($item['bixola_link']['nofollow']) ? 'nofollow' : '';
                                    }

                                ?>
                                <div class="col-xl-<?php print esc_attr($settings['column_desktop']); ?> col-lg-<?php print esc_attr($settings['column_laptop']); ?> col-md-<?php print esc_attr($settings['column_tablet']); ?> col-sm-<?php print esc_attr($settings['column_mobile']); ?> wow fadeInLeft" data-wow-delay="<?php print esc_attr($item['animation_delay']); ?>">
                                    <div class="team-six__single">
                                        <div class="team-six__img-box">
                                            <div class="team-six__img">
                                                <?php if( !empty($bixola_team_image_url ) ) : ?>
                                                    <img src="<?php echo esc_url($bixola_team_image_url); ?>" alt="<?php echo esc_attr($bixola_team_image_alt); ?>">
                                                <?php endif; ?>
                                                <div class="team-six__social">
                                                    <?php if( !empty($item['facebook_title'] ) ) : ?>
                                                        <a href="<?php echo esc_url( $item['facebook_title'] ); ?>"><i class="fa-brands fa-facebook-f"></i></a>
                                                    <?php endif; ?>

                                                    <?php if( !empty($item['twitter_title'] ) ) : ?>
                                                        <a href="<?php echo esc_url( $item['twitter_title'] ); ?>"><i class="fa-brands fa-x-twitter"></i></a>
                                                    <?php endif; ?>

                                                    <?php if( !empty($item['pinterest_title'] ) ) : ?>
                                                        <a href="<?php echo esc_url( $item['pinterest_title'] ); ?>"><i class="fa-brands fa-pinterest-p"></i></a>
                                                    <?php endif; ?>

                                                    <?php if( !empty($item['instagram_title'] ) ) : ?>
                                                        <a href="<?php echo esc_url( $item['instagram_title'] ); ?>"><i class="fa-brands fa-instagram"></i></a>
                                                    <?php endif; ?>

                                                    <?php if( !empty($item['linkedin_title'] ) ) : ?>
                                                        <a href="<?php echo esc_url( $item['linkedin_title'] ); ?>"><i class="fa-brands fa-linkedin-in"></i></a>
                                                    <?php endif; ?>

                                                    <?php if( !empty($item['youtube_title'] ) ) : ?>
                                                        <a href="<?php echo esc_url( $item['youtube_title'] ); ?>"><i class="fa-brands fa-youtube"></i></a>
                                                    <?php endif; ?>

                                                    <?php if( !empty($item['flickr_title'] ) ) : ?>
                                                        <a href="<?php echo esc_url( $item['flickr_title'] ); ?>"><i class="fa-brands fa-flickr"></i></a>
                                                    <?php endif; ?>

                                                    <?php if( !empty($item['vimeo_title'] ) ) : ?>
                                                        <a href="<?php echo esc_url( $item['vimeo_title'] ); ?>"><i class="fa-brands fa-vimeo-v"></i></a>
                                                    <?php endif; ?>

                                                    <?php if( !empty($item['behance_title'] ) ) : ?>
                                                        <a href="<?php echo esc_url( $item['behance_title'] ); ?>"><i class="fa-brands fa-behance"></i></a>
                                                    <?php endif; ?>

                                                    <?php if( !empty($item['dribble_title'] ) ) : ?>
                                                        <a href="<?php echo esc_url( $item['dribble_title'] ); ?>"><i class="fa-brands fa-dribbble"></i></a>
                                                    <?php endif; ?>

                                                    <?php if( !empty($item['phone_title'] ) ) : ?>
                                                        <a href="tel:<?php echo esc_url( $item['phone_title'] ); ?>"><i class="fa-solid fa-phone"></i></a>
                                                    <?php endif; ?>  

                                                    <?php if( !empty($item['gitub_title'] ) ) : ?>
                                                        <a href="<?php echo esc_url( $item['gitub_title'] ); ?>"><i class="fa-brands fa-github"></i></a>
                                                    <?php endif; ?>

                                                    <?php if( !empty($item['web_title'] ) ) : ?>
                                                        <a href="<?php echo esc_url( $item['web_title'] ); ?>"><i class="fa-solid fa-globe"></i></a>
                                                    <?php endif; ?>  

                                                    <?php if( !empty($item['email_title'] ) ) : ?>
                                                        <a href="mailto:<?php echo esc_url( $item['email_title'] ); ?>"><i class="fa-solid fa-envelope"></i></a>
                                                    <?php endif; ?>
                                                </div>
                                            </div>
                                        </div>
                                        <div class="team-six__content">
                                            <h3 class="team-six__title">
                                                <a target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>" href="<?php echo esc_url($link); ?>">
                                                    <?php echo bixola_kses( $item['name'] ); ?>
                                                </a>
                                            </h3>
                                            <?php if( !empty($item['designation']) ) : ?>
                                                <p class="team-six__sub-title">
                                                    <?php echo bixola_kses( $item['designation'] ); ?>
                                                </p>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </div>
                                <?php endforeach; ?>
                            </div>
                        </div>
                    </div>
                </section>

            <?php elseif ( $settings['design_style']  == 'layout-5' ): ?>

                <section class="bixola-el-section team-page">
                    <div class="container">
                        <div class="row">
                            <?php foreach ( $settings['team_list'] as $item ) :

                                if ( !empty($item['image']['url']) ) {
                                    $bixola_team_image_url = !empty($item['image']['id']) ? wp_get_attachment_image_url( $item['image']['id'], 'full') : $item['image']['url'];
                                    $bixola_team_image_alt = get_post_meta($item["image"]["id"], "_wp_attachment_image_alt", true);
                                }

                                // Link
                                if ('2' == $item['bixola_link_type']) {
                                    $link = get_permalink($item['bixola_page_link']);
                                    $target = '_self';
                                    $rel = 'nofollow';
                                } else {
                                    $link = !empty($item['bixola_link']['url']) ? $item['bixola_link']['url'] : '';
                                    $target = !empty($item['bixola_link']['is_external']) ? '_blank' : '';
                                    $rel = !empty($item['bixola_link']['nofollow']) ? 'nofollow' : '';
                                }

                            ?>
                            <div class="col-xl-<?php print esc_attr($settings['column_desktop']); ?> col-lg-<?php print esc_attr($settings['column_laptop']); ?> col-md-<?php print esc_attr($settings['column_tablet']); ?> col-sm-<?php print esc_attr($settings['column_mobile']); ?>">
                                <div class="team-one__single">
                                    <div class="team-one__img-box">
                                        <?php if( !empty( $bixola_team_image_url ) ) : ?>
                                            <div class="team-one__img">
                                                <img src="<?php echo esc_url($bixola_team_image_url); ?>" alt="<?php echo esc_attr($bixola_team_image_alt); ?>">
                                            </div>
                                        <?php endif; ?>
                                        <div class="team-one__content">
                                            <h3 class="team-one__name">
                                                <a target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>" href="<?php echo esc_url($link); ?>">
                                                    <?php echo bixola_kses( $item['name'] ); ?>
                                                </a>
                                            </h3>
                                            <?php if( !empty($item['designation']) ) : ?>
                                                <p class="team-one__sub-title">
                                                    <?php echo bixola_kses( $item['designation'] ); ?>
                                                </p>
                                            <?php endif; ?>
                                        </div>
                                        <div class="team-one__social">
                                            <?php if( !empty($item['facebook_title'] ) ) : ?>
                                                <a href="<?php echo esc_url( $item['facebook_title'] ); ?>"><i class="fa-brands fa-facebook-f"></i></a>
                                            <?php endif; ?>

                                            <?php if( !empty($item['twitter_title'] ) ) : ?>
                                                <a href="<?php echo esc_url( $item['twitter_title'] ); ?>"><i class="fa-brands fa-x-twitter"></i></a>
                                            <?php endif; ?>

                                            <?php if( !empty($item['pinterest_title'] ) ) : ?>
                                                <a href="<?php echo esc_url( $item['pinterest_title'] ); ?>"><i class="fa-brands fa-pinterest-p"></i></a>
                                            <?php endif; ?>

                                            <?php if( !empty($item['instagram_title'] ) ) : ?>
                                                <a href="<?php echo esc_url( $item['instagram_title'] ); ?>"><i class="fa-brands fa-instagram"></i></a>
                                            <?php endif; ?>

                                            <?php if( !empty($item['linkedin_title'] ) ) : ?>
                                                <a href="<?php echo esc_url( $item['linkedin_title'] ); ?>"><i class="fa-brands fa-linkedin-in"></i></a>
                                            <?php endif; ?>

                                            <?php if( !empty($item['youtube_title'] ) ) : ?>
                                                <a href="<?php echo esc_url( $item['youtube_title'] ); ?>"><i class="fa-brands fa-youtube"></i></a>
                                            <?php endif; ?>

                                            <?php if( !empty($item['flickr_title'] ) ) : ?>
                                                <a href="<?php echo esc_url( $item['flickr_title'] ); ?>"><i class="fa-brands fa-flickr"></i></a>
                                            <?php endif; ?>

                                            <?php if( !empty($item['vimeo_title'] ) ) : ?>
                                                <a href="<?php echo esc_url( $item['vimeo_title'] ); ?>"><i class="fa-brands fa-vimeo-v"></i></a>
                                            <?php endif; ?>

                                            <?php if( !empty($item['behance_title'] ) ) : ?>
                                                <a href="<?php echo esc_url( $item['behance_title'] ); ?>"><i class="fa-brands fa-behance"></i></a>
                                            <?php endif; ?>

                                            <?php if( !empty($item['dribble_title'] ) ) : ?>
                                                <a href="<?php echo esc_url( $item['dribble_title'] ); ?>"><i class="fa-brands fa-dribbble"></i></a>
                                            <?php endif; ?>

                                            <?php if( !empty($item['phone_title'] ) ) : ?>
                                                <a href="tel:<?php echo esc_url( $item['phone_title'] ); ?>"><i class="fa-solid fa-phone"></i></a>
                                            <?php endif; ?>  

                                            <?php if( !empty($item['gitub_title'] ) ) : ?>
                                                <a href="<?php echo esc_url( $item['gitub_title'] ); ?>"><i class="fa-brands fa-github"></i></a>
                                            <?php endif; ?>

                                            <?php if( !empty($item['web_title'] ) ) : ?>
                                                <a href="<?php echo esc_url( $item['web_title'] ); ?>"><i class="fa-solid fa-globe"></i></a>
                                            <?php endif; ?>  

                                            <?php if( !empty($item['email_title'] ) ) : ?>
                                                <a href="mailto:<?php echo esc_url( $item['email_title'] ); ?>"><i class="fa-solid fa-envelope"></i></a>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </section>

            <?php elseif ( $settings['design_style']  == 'layout-6' ): ?>

                <section class="bixola-el-section team-carousel-page">
                    <div class="container">
                        <div class="team-carousel thm-owl__carousel owl-theme owl-carousel carousel-dot-style">
                            <?php foreach ( $settings['team_list'] as $item ) :

                                if ( !empty($item['image']['url']) ) {
                                    $bixola_team_image_url = !empty($item['image']['id']) ? wp_get_attachment_image_url( $item['image']['id'], 'full') : $item['image']['url'];
                                    $bixola_team_image_alt = get_post_meta($item["image"]["id"], "_wp_attachment_image_alt", true);
                                }

                                // Link
                                if ('2' == $item['bixola_link_type']) {
                                    $link = get_permalink($item['bixola_page_link']);
                                    $target = '_self';
                                    $rel = 'nofollow';
                                } else {
                                    $link = !empty($item['bixola_link']['url']) ? $item['bixola_link']['url'] : '';
                                    $target = !empty($item['bixola_link']['is_external']) ? '_blank' : '';
                                    $rel = !empty($item['bixola_link']['nofollow']) ? 'nofollow' : '';
                                }

                            ?>
                            <div class="item">
                                <div class="team-one__single">
                                    <div class="team-one__img-box">
                                        <?php if( !empty( $bixola_team_image_url ) ) : ?>
                                            <div class="team-one__img">
                                                <img src="<?php echo esc_url($bixola_team_image_url); ?>" alt="<?php echo esc_attr($bixola_team_image_alt); ?>">
                                            </div>
                                        <?php endif; ?>
                                        <div class="team-one__content">
                                            <h3 class="team-one__name">
                                                <a target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>" href="<?php echo esc_url($link); ?>">
                                                    <?php echo bixola_kses( $item['name'] ); ?>
                                                </a>
                                            </h3>
                                            <?php if( !empty($item['designation']) ) : ?>
                                                <p class="team-one__sub-title">
                                                    <?php echo bixola_kses( $item['designation'] ); ?>
                                                </p>
                                            <?php endif; ?>
                                        </div>
                                        <div class="team-one__social">
                                            <?php if( !empty($item['facebook_title'] ) ) : ?>
                                                <a href="<?php echo esc_url( $item['facebook_title'] ); ?>"><i class="fa-brands fa-facebook-f"></i></a>
                                            <?php endif; ?>

                                            <?php if( !empty($item['twitter_title'] ) ) : ?>
                                                <a href="<?php echo esc_url( $item['twitter_title'] ); ?>"><i class="fa-brands fa-x-twitter"></i></a>
                                            <?php endif; ?>

                                            <?php if( !empty($item['pinterest_title'] ) ) : ?>
                                                <a href="<?php echo esc_url( $item['pinterest_title'] ); ?>"><i class="fa-brands fa-pinterest-p"></i></a>
                                            <?php endif; ?>

                                            <?php if( !empty($item['instagram_title'] ) ) : ?>
                                                <a href="<?php echo esc_url( $item['instagram_title'] ); ?>"><i class="fa-brands fa-instagram"></i></a>
                                            <?php endif; ?>

                                            <?php if( !empty($item['linkedin_title'] ) ) : ?>
                                                <a href="<?php echo esc_url( $item['linkedin_title'] ); ?>"><i class="fa-brands fa-linkedin-in"></i></a>
                                            <?php endif; ?>

                                            <?php if( !empty($item['youtube_title'] ) ) : ?>
                                                <a href="<?php echo esc_url( $item['youtube_title'] ); ?>"><i class="fa-brands fa-youtube"></i></a>
                                            <?php endif; ?>

                                            <?php if( !empty($item['flickr_title'] ) ) : ?>
                                                <a href="<?php echo esc_url( $item['flickr_title'] ); ?>"><i class="fa-brands fa-flickr"></i></a>
                                            <?php endif; ?>

                                            <?php if( !empty($item['vimeo_title'] ) ) : ?>
                                                <a href="<?php echo esc_url( $item['vimeo_title'] ); ?>"><i class="fa-brands fa-vimeo-v"></i></a>
                                            <?php endif; ?>

                                            <?php if( !empty($item['behance_title'] ) ) : ?>
                                                <a href="<?php echo esc_url( $item['behance_title'] ); ?>"><i class="fa-brands fa-behance"></i></a>
                                            <?php endif; ?>

                                            <?php if( !empty($item['dribble_title'] ) ) : ?>
                                                <a href="<?php echo esc_url( $item['dribble_title'] ); ?>"><i class="fa-brands fa-dribbble"></i></a>
                                            <?php endif; ?>

                                            <?php if( !empty($item['phone_title'] ) ) : ?>
                                                <a href="tel:<?php echo esc_url( $item['phone_title'] ); ?>"><i class="fa-solid fa-phone"></i></a>
                                            <?php endif; ?>  

                                            <?php if( !empty($item['gitub_title'] ) ) : ?>
                                                <a href="<?php echo esc_url( $item['gitub_title'] ); ?>"><i class="fa-brands fa-github"></i></a>
                                            <?php endif; ?>

                                            <?php if( !empty($item['web_title'] ) ) : ?>
                                                <a href="<?php echo esc_url( $item['web_title'] ); ?>"><i class="fa-solid fa-globe"></i></a>
                                            <?php endif; ?>  

                                            <?php if( !empty($item['email_title'] ) ) : ?>
                                                <a href="mailto:<?php echo esc_url( $item['email_title'] ); ?>"><i class="fa-solid fa-envelope"></i></a>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </section>

            <?php endif; ?>

		<?php
	}
}

$widgets_manager->register( new Bixola_Team() );