<?php
namespace BixolaCore\Widgets;

use Elementor\Widget_Base;
use \Elementor\Control_Media;
use \Elementor\Controls_Manager;
use \Elementor\Group_Control_Border;
use \Elementor\Group_Control_Typography;
use \Elementor\Repeater;


if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Bixola Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class Bixola_Pricing extends Widget_Base {

	/**
	 * Retrieve the widget name.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'bixola_pricing';
	}

	/**
	 * Retrieve the widget title.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Pricing', 'bixola-core' );
	}

	/**
	 * Retrieve the widget icon.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'bixola-icon';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * Note that currently Elementor supports only one category.
	 * When multiple categories passed, Elementor uses the first one.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'bixola_core' ];
	}

	/**
	 * Retrieve the list of scripts the widget depended on.
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'bixola-core' ];
	}

	/**
	 * Register the widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function register_controls() {

        $this->start_controls_section(
            '_content_design_layout',
            [
                'label' => __('Design Layout', 'bixola-core'),
            ]
        );

        $this->add_control(
            'design_style',
            [
                'label' => esc_html__( 'Select Layout', 'text-domain' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'layout-1' => esc_html__( 'Layout 1', 'text-domain' ),
                    'layout-2' => esc_html__( 'Layout 2', 'text-domain' ),
                    'layout-3' => esc_html__( 'Layout 3', 'text-domain' ),
                    'layout-4' => esc_html__( 'Layout 4', 'text-domain' ),
                ],
                'default' => 'layout-1',
            ]
        );

        $this->add_control(
            'shape_switch',
            [
                'label' => esc_html__( 'Shape Switch', 'text-domain' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Show', 'text-domain' ),
                'label_off' => esc_html__( 'Hide', 'text-domain' ),
                'return_value' => 'yes',
                'default' => 'yes',
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-4' ],
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_content_title',
            [
                'label' => __('Title & Content', 'bixola-core'),
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-2', 'layout-4' ],
                ],
            ]
        );

        $this->add_control(
            'section_subheading',
            [
                'label' => esc_html__( 'Subheading', 'text-domain' ),
                'description' => bixola_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'pricing plans', 'text-domain' ),
                'label_block' => true,
                'condition' => [
                    'design_style' => 'layout-1',
                ],
            ]
        );

        $this->add_control(
            'section_title',
            [
                'label' => esc_html__( 'Title', 'text-domain' ),
                'description' => bixola_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'Offering Our Best Plan For Your Business Purpose', 'text-domain' ),
                'label_block' => true,
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-2', 'layout-4' ],
                ],
            ]
        );

        $this->add_control(
            'section_discount',
            [
                'label' => esc_html__( 'Discount', 'text-domain' ),
                'description' => bixola_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'Save 20%', 'text-domain' ),
                'label_block' => true,
                'condition' => [
                    'design_style' => [ 'layout-4' ],
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_content_pricing_tab',
            [
                'label' => __('Pricing Tab', 'bixola-core'),
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-2', 'layout-4' ],
                ],
            ]
        );

        $repeater = new Repeater();

        $repeater->add_control(
            'active_tab',
            [
                'label' => __('Is Active Tab?', 'seoq-core'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => __('Yes', 'seoq-core'),
                'label_off' => __('No', 'seoq-core'),
                'return_value' => 'yes',
                'default' => 'yes',
                'frontend_available' => true,
            ]
        );

        $repeater->add_control(
            'tab_title',
            [
                'label' => esc_html__( 'Tab Title', 'bixola-core' ),
                'type' => Controls_Manager::TEXT,
                'dynamic' => [
                    'active' => true
                ]
            ]
        );
        
        $repeater->add_control(
            'template',
            [
                'label' => __('Section Template', 'bixola-core'),
                'placeholder' => __('Select a section template for as tab content', 'bixola-core'),
  
                'type' => Controls_Manager::SELECT2,
                'options' => get_elementor_templates()
            ]
        );

        $this->add_control(
            'tabs',
            [
                'show_label' => false,
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'title_field' => '{{tab_title}}',
                'default' => [
                    [
                        'tab_title' => 'Monthly',
                    ],
                    [
                        'tab_title' => 'Yearly',
                    ]
                ]
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_content_slider',
            [
                'label' => esc_html__( 'Slider',  'text-domain'  ),
                'tab' => Controls_Manager::TAB_CONTENT,
                'condition' => [
                    'design_style' => 'layout-3',
                ],
            ]
        );

        $repeater = new Repeater();
        
        $repeater->add_control(
            'template_two',
            [
                'label' => __('Section Template', 'bixola-core'),
                'placeholder' => __('Select a section template for as tab content', 'bixola-core'),
  
                'type' => Controls_Manager::SELECT2,
                'options' => get_elementor_templates()
            ]
        );

        $this->add_control(
            'tabs_two',
            [
                'show_label' => false,
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
			'_style_design_layout',
			[
				'label' => __( 'Design Layout', 'bixola-core' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

        $this->add_responsive_control(
            'design_layout_padding',
            [
                'label' => __( 'Padding', 'bixola-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-section' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'design_layout_margin',
            [
                'label' => __( 'Margin', 'bixola-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-section' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'design_layout_background',
            [
                'label' => __( 'Background', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-section' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
			'_style_title',
			[
				'label' => __( 'Title & Content', 'bixola-core' ),
				'tab' => Controls_Manager::TAB_STYLE,
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-2', 'layout-4' ],
                ],
			]
		);

        $this->add_control(
            '_heading_subtitle',
            [
                'label' => esc_html__( 'Subtitle', 'bixola-core' ),
                'type' => Controls_Manager::HEADING,
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-2' ],
                ],
            ]
        );

        $this->add_control(
            '_heading_discount',
            [
                'label' => esc_html__( 'Discount', 'bixola-core' ),
                'type' => Controls_Manager::HEADING,
                'condition' => [
                    'design_style' => [ 'layout-4' ],
                ],
            ]
        );

        $this->add_responsive_control(
            'subtitle_spacing',
            [
                'label' => __( 'Bottom Spacing', 'bixola-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .section-title__tagline' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-2' ],
                ],
            ]
        );

        $this->add_control(
            'subtitle_border_color',
            [
                'label'     => esc_html__( 'Border Color', 'bixola-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .section-title__tagline:before' => 'background-color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-2' ],
                ],
            ]
        );

        $this->add_control(
            'subtitle_color',
            [
                'label'     => esc_html__( 'Color', 'bixola-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .section-title__tagline' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .pricing-two__save p' => 'color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-2', 'layout-4' ],
                ],
            ]
        );

        $this->add_control(
            'subtitle_background',
            [
                'label'     => esc_html__( 'Background', 'bixola-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .pricing-two__save p' => 'background-color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => [ 'layout-4' ],
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'label'    => esc_html__( 'Typography', 'bixola-core' ),
                'name'     => 'subtitle_typography',
                'selector' => '{{WRAPPER}} .section-title__tagline, .pricing-two__save p',
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-2', 'layout-4' ],
                ],
            ]
        );

        $this->add_control(
            '_heading_title',
            [
                'label' => esc_html__( 'Title', 'bixola-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'title_spacing',
            [
                'label' => __( 'Bottom Spacing', 'bixola-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .section-title__title' => 'margin-bottom: {{SIZE}}{{UNIT}}!important;',
                    '{{WRAPPER}} .pricing-two .section__title-two' => 'margin-bottom: {{SIZE}}{{UNIT}}!important;',
                ],
            ]
        );

        $this->add_control(
            'title_color',
            [
                'label'     => esc_html__( 'Color', 'bixola-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .section-title__title' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .pricing-two .section__title-two' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'label'    => esc_html__( 'Typography', 'bixola-core' ),
                'name'     => 'title_typography',
                'selector' => '{{WRAPPER}} .section-title__title, .pricing-two .section__title-two',
            ]
        );

        $this->add_control(
            '_heading_style_tab',
            [
                'label' => esc_html__( 'Tab', 'bixola-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->start_controls_tabs( '_tabs_tab' );
        
        $this->start_controls_tab(
            'tab_normal_tab',
            [
                'label' => esc_html__( 'Normal', 'text-domain' ),
            ]
        );

        $this->add_control(
            'tab_color',
            [
                'label'     => esc_html__( 'Color', 'bixola-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .pricing-one__main-tab-box .tab-buttons .tab-btn span' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .pricing-two__main-tab-box .tab-buttons .tab-btn span' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'tab_background',
            [
                'label'     => esc_html__( 'Background', 'bixola-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .pricing-one__main-tab-box .tab-buttons .tab-btn span' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .pricing-two__main-tab-box .tab-buttons .tab-btn span' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name' => 'tab_border',
                'selector' => '{{WRAPPER}} .pricing-one__main-tab-box .tab-buttons .tab-btn span, .pricing-two__main-tab-box .tab-buttons .tab-btn span',
            ]
        );
        
        $this->end_controls_tab();
        
        $this->start_controls_tab(
            'tab_active_tab',
            [
                'label' => esc_html__( 'Active', 'text-domain' ),
            ]
        );
        
        $this->add_control(
            'tab_color_active',
            [
                'label'     => esc_html__( 'Color', 'bixola-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .pricing-one__main-tab-box .tab-buttons .tab-btn.active-btn span' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .pricing-two__main-tab-box .tab-buttons .tab-btn.active-btn span' => 'color: {{VALUE}}',
                ],
            ]
        );
        
        $this->add_control(
            'tab_background_active',
            [
                'label'     => esc_html__( 'Background', 'bixola-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .pricing-one__main-tab-box .tab-buttons .tab-btn span:before' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .pricing-two__main-tab-box .tab-buttons .tab-btn span:before' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name' => 'tab_border_active',
                'selector' => '{{WRAPPER}} .pricing-one__main-tab-box .tab-buttons .tab-btn.active-btn span, .pricing-two__main-tab-box .tab-buttons .tab-btn.active-btn span',
            ]
        );
        
        $this->end_controls_tab();
        
        $this->end_controls_tabs();

        $this->end_controls_section();

        $this->start_controls_section(
            '_style_slider',
            [
                'label' => esc_html__( 'Slider',  'text-domain'  ),
                'tab' => Controls_Manager::TAB_STYLE,
                'condition' => [
                    'design_style' => 'layout-3',
                ],
            ]
        );
        
        $this->add_control(
            '_heading_style_navigation',
            [
                'label' => esc_html__( 'Navigation', 'text-domain' ),
                'type' => Controls_Manager::HEADING,
            ]
        );

        $this->start_controls_tabs( '_tabs_navigation' );
        
        $this->start_controls_tab(
            'navigation_normal_tab',
            [
                'label' => esc_html__( 'Normal', 'text-domain' ),
            ]
        );
        
        $this->add_control(
            'navigation_background',
            [
                'label' => esc_html__( 'Background', 'text-domain' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .carousel-dot-style.owl-carousel .owl-dots .owl-dot' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        
        $this->end_controls_tab();
        
        $this->start_controls_tab(
            'navigation_active_tab',
            [
                'label' => esc_html__( 'Active', 'text-domain' ),
            ]
        );
        
        $this->add_control(
            'navigation_background_active',
            [
                'label' => esc_html__( 'Background', 'text-domain' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .carousel-dot-style.owl-carousel .owl-dot.active' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        
        $this->end_controls_tab();
        
        $this->end_controls_tabs();
        
        $this->end_controls_section();

	}

    private static function get_currency_symbol($symbol_name)
    {
        $symbols = [
            'baht' => '&#3647;',
            'bdt' => '&#2547;',
            'dollar' => '&#36;',
            'euro' => '&#128;',
            'franc' => '&#8355;',
            'guilder' => '&fnof;',
            'indian_rupee' => '&#8377;',
            'pound' => '&#163;',
            'peso' => '&#8369;',
            'peseta' => '&#8359',
            'lira' => '&#8356;',
            'ruble' => '&#8381;',
            'shekel' => '&#8362;',
            'rupee' => '&#8360;',
            'real' => 'R$',
            'krona' => 'kr',
            'won' => '&#8361;',
            'yen' => '&#165;',
        ];

        return isset($symbols[$symbol_name]) ? $symbols[$symbol_name] : '';
    }


	/**
	 * Render the widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		?>

            <?php if ( $settings['design_style']  == 'layout-1' ): ?>
            
                <section class="bixola-el-section pricing-one">
                    <?php if ( !empty( $settings['shape_switch'] ) ) : ?>
                        <div class="pricing-one__shape-1 zoominout">
                            <img src="<?php print get_template_directory_uri(); ?>/assets/images/shapes/pricing-one-shape-1.png" alt="">
                        </div>
                        <div class="pricing-one__shape-2 img-bounce">
                            <img src="<?php print get_template_directory_uri(); ?>/assets/images/shapes/pricing-one-shape-2.png" alt="">
                        </div>
                    <?php endif; ?>
                    <div class="container">
                        <div class="section-title text-center">
                            <?php if ( !empty( $settings['section_subheading'] ) ) : ?>
                                <div class="section-title__tagline-box">
                                    <p class="section-title__tagline">
                                        <?php print twinkle_kses($settings['section_subheading']); ?>
                                    </p>
                                </div>
                            <?php endif; ?>
                            <?php if ( !empty( $settings['section_title'] ) ) : ?>
                                <h2 class="section-title__title">
                                    <?php print twinkle_kses($settings['section_title']); ?>
                                </h2>
                            <?php endif; ?>
                        </div>
                        <div class="pricing-one__main-tab-box tabs-box">
                            <ul class="tab-buttons list-unstyled">
                                <?php foreach ($settings['tabs'] as $key => $tab):
                                    $active = ($key == 0) ? 'active-btn' : '';
                                ?>
                                <li data-tab="#<?php echo esc_attr($key); ?>" class="tab-btn tab-btn-one <?php echo esc_attr($active); ?>">
                                    <span><?php echo twinkle_kses($tab['tab_title']); ?></span>
                                </li>
                                <?php endforeach; ?>
                            </ul>
                            <div class="tabs-content">
                                <?php foreach ($settings['tabs'] as $key => $tab):
                                    $active = ($key == 0) ? 'active-tab' : '';
                                ?>
                                    <div class="tab <?php echo esc_attr($active); ?>" id="<?php echo esc_attr($key); ?>">
                                        <div class="pricing-one__inner">
                                            <?php echo \Elementor\Plugin::instance()->frontend->get_builder_content($tab['template'], true); ?>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        </div>
                    </div>
                </section>
            
            <?php elseif ( $settings['design_style']  == 'layout-2' ): ?>

                <section class="bixola-el-section pricing-page">
                    <div class="container">
                        <?php if ( !empty( $settings['section_title'] ) ) : ?>
                            <h3 class="pricing-page-title">
                                <?php print twinkle_kses($settings['section_title']); ?>
                            </h2>
                        <?php endif; ?>
                        <div class="pricing-one__main-tab-box tabs-box">
                            <ul class="tab-buttons list-unstyled">
                                <?php foreach ($settings['tabs'] as $key => $tab):
                                    $active = ($key == 0) ? 'active-btn' : '';
                                ?>
                                <li data-tab="#<?php echo esc_attr($key); ?>" class="tab-btn tab-btn-one <?php echo esc_attr($active); ?>">
                                    <span><?php echo twinkle_kses($tab['tab_title']); ?></span>
                                </li>
                                <?php endforeach; ?>
                            </ul>
                            <div class="tabs-content">
                                <?php foreach ($settings['tabs'] as $key => $tab):
                                    $active = ($key == 0) ? 'active-tab' : '';
                                ?>
                                <div class="tab <?php echo esc_attr($active); ?>" id="<?php echo esc_attr($key); ?>">
                                    <?php echo \Elementor\Plugin::instance()->frontend->get_builder_content($tab['template'], true); ?>
                                </div>
                                <?php endforeach; ?>
                            </div>
                        </div>
                    </div>
                </section>

            <?php elseif ( $settings['design_style']  == 'layout-3' ): ?>
            
                <section class="bixola-el-section pricing-carousel-page">
                    <div class="container">
                        <div class="pricing-carousel thm-owl__carousel owl-theme owl-carousel carousel-dot-style">
                            <?php foreach ($settings['tabs_two'] as $item) : ?>
                                <div class="item">
                                    <?php echo \Elementor\Plugin::instance()->frontend->get_builder_content($item['template_two'], true); ?>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </section>

            <?php elseif ( $settings['design_style']  == 'layout-4' ): ?>

                <section class="bixola-el-section pricing-two">
                    <?php if ( !empty( $settings['shape_switch'] ) ) : ?>
                        <div class="pricing-two__shape-2 zoominout">
                            <img src="<?php print get_template_directory_uri(); ?>/assets/images/shapes/pricing-two-shape-2.png" alt="">
                        </div>
                        <div class="pricing-two__shape-3 rotate-me">
                            <img src="<?php print get_template_directory_uri(); ?>/assets/images/shapes/pricing-two-shape-3.png" alt="">
                        </div>
                        <div class="pricing-two__shape-4 rotate-me">
                            <img src="<?php print get_template_directory_uri(); ?>/assets/images/shapes/pricing-two-shape-4.png" alt="">
                        </div>
                        <div class="pricing-two__shape-5 img-bounce">
                            <img src="<?php print get_template_directory_uri(); ?>/assets/images/shapes/pricing-two-shape-5.png" alt="">
                        </div>
                    <?php endif; ?>
                    <div class="container">
                        <?php if ( !empty( $settings['section_title'] ) ) : ?>
                            <h3 class="section__title-two">
                                <?php print twinkle_kses($settings['section_title']); ?>
                            </h2>
                        <?php endif; ?>
                        <?php if ( !empty( $settings['section_discount'] ) ) : ?>
                            <div class="pricing-two__save-box">
                                <div class="pricing-two__save">
                                    <div class="pricing-two__shape-1">
                                        <img src="<?php print get_template_directory_uri(); ?>/assets/images/shapes/pricing-two-shape-1.png" alt="">
                                    </div>
                                    <p><?php print twinkle_kses($settings['section_discount']); ?></p>
                                </div>
                            </div>
                        <?php endif; ?>
                        <div class="pricing-two__main-tab-box tabs-box">
                            <ul class="tab-buttons list-unstyled">
                                <?php foreach ($settings['tabs'] as $key => $tab):
                                    $active = ($key == 0) ? 'active-btn' : '';
                                ?>
                                <li data-tab="#tab-<?php echo esc_attr($key); ?>" class="tab-btn tab-btn-one <?php echo esc_attr($active); ?>">
                                    <span><?php echo twinkle_kses($tab['tab_title']); ?></span>
                                </li>
                                <?php endforeach; ?>
                            </ul>
                            <div class="tabs-content">
                                <?php foreach ($settings['tabs'] as $key => $tab):
                                    $active = ($key == 0) ? 'active-tab' : '';
                                ?>
                                <div class="tab <?php echo esc_attr($active); ?>" id="tab-<?php echo esc_attr($key); ?>">
                                    <?php echo \Elementor\Plugin::instance()->frontend->get_builder_content($tab['template'], true); ?>
                                </div>
                                <?php endforeach; ?>
                            </div>
                        </div>
                    </div>
                </section>
            
            <?php endif; ?>

        <?php
	}

}

$widgets_manager->register( new Bixola_Pricing() );