<?php
namespace BixolaCore\Widgets;

use \Elementor\Widget_Base;
use \Elementor\Controls_Manager;
use \Elementor\Group_Control_Background;
use \Elementor\Group_Control_Typography;
use \Elementor\Group_Control_Border;
use \Elementor\Repeater;
use \Elementor\Utils;
use \Elementor\Control_Media;
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Xlab Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class Xlab_Hero extends Widget_Base {

	/**
	 * Retrieve the widget name.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'xlab_hero';
	}

	/**
	 * Retrieve the widget title.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Hero', 'bixola-core' );
	}

	/**
	 * Retrieve the widget icon.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'xlab-icon';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * Note that currently Elementor supports only one category.
	 * When multiple categories passed, Elementor uses the first one.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'xlab_core' ];
	}

	/**
	 * Retrieve the list of scripts the widget depended on.
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'bixola-core' ];
	}

	/**
	 * Register the widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function register_controls() {

        // layout Panel
        $this->start_controls_section(
            'xlab_layout',
            [
                'label' => esc_html__('Design Layout', 'bixola-core'),
            ]
        );
        $this->add_control(
            'xlab_design_style',
            [
                'label' => esc_html__('Select Layout', 'bixola-core'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'layout-1' => esc_html__('Layout 1', 'bixola-core'),
                    'layout-2' => esc_html__('Layout 2', 'bixola-core'),
                    'layout-3' => esc_html__('Layout 3', 'bixola-core'),
                ],
                'default' => 'layout-1',
            ]
        );

        $this->end_controls_section();

        // xlab_section_title
        $this->start_controls_section(
            'xlab_hero_content',
            [
                'label' => esc_html__('Hero Content', 'bixola-core'),
            ]
        );

        $this->add_control(
            'xlab_hero_image',
            [
                'label' => esc_html__( 'Hero Image', 'bixola-core' ),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
            ]
        );

        $this->add_control(
            'xlab_hero_image_shape',
            [
                'label' => esc_html__( 'Shape Image', 'bixola-core' ),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
                'condition' => [
                    'xlab_design_style' => [ 'layout-3' ],
                ],
            ]
        );

        $this->add_control(
            'xlab_title',
            [
                'label' => esc_html__('Title', 'bixola-core'),
                'description' => xlab_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXTAREA,
                'default' => __('Strategies for Tomorrow <span>Solutions</span> for Today', 'bixola-core'),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'xlab_description',
            [
                'label' => esc_html__('Description', 'bixola-core'),
                'description' => xlab_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXTAREA,
                'default' => esc_html__('Starting a corporate business typically involves several steps, such as develo a business', 'bixola-core'),
                'placeholder' => esc_html__('Type description here', 'bixola-core'),
                'condition' => [
                    'xlab_design_style' => [ 'layout-1', 'layout-2', 'layout-3' ],
                ],
            ]
        );
        
        $this->end_controls_section();

        // xlab_button
        $this->start_controls_section(
            'xlab_hero_button_content',
            [
                'label' => esc_html__('Button', 'bixola-core'),
            ]
        );

        $this->add_control(
            'xlab_btn_button_show',
            [
                'label' => esc_html__( 'Show Button', 'bixola-core' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Show', 'bixola-core' ),
                'label_off' => esc_html__( 'Hide', 'bixola-core' ),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'xlab_btn_text',
            [
                'label' => esc_html__('Button Text', 'bixola-core'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Let’s Talk ', 'bixola-core'),
                'title' => esc_html__('Enter button text', 'bixola-core'),
                'label_block' => true,
                'condition' => [
                    'xlab_btn_button_show' => 'yes'
                ],
            ]
        );

        $this->add_control(
            'xlab_btn_link_type',
            [
                'label' => esc_html__('Button Link Type', 'bixola-core'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '1' => 'Custom Link',
                    '2' => 'Internal Page',
                ],
                'default' => '1',
                'label_block' => true,
                'condition' => [
                    'xlab_btn_button_show' => 'yes'
                ],
            ]
        );

        $this->add_control(
            'xlab_btn_link',
            [
                'label' => esc_html__('Button link', 'bixola-core'),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => esc_html__('https://your-link.com', 'bixola-core'),
                'show_external' => false,
                'default' => [
                    'url' => '#',
                    'is_external' => true,
                    'nofollow' => true,
                    'custom_attributes' => '',
                ],
                'condition' => [
                    'xlab_btn_link_type' => '1',
                    'xlab_btn_button_show' => 'yes'
                ],
                'label_block' => true,
            ]
        );

        $this->add_control(
            'xlab_btn_page_link',
            [
                'label' => esc_html__('Select Button Page', 'bixola-core'),
                'type' => Controls_Manager::SELECT2,
                'label_block' => true,
                'options' => xlab_get_all_pages(),
                'condition' => [
                    'xlab_btn_link_type' => '2',
                    'xlab_btn_button_show' => 'yes'
                ]
            ]
        );

        // Secondary Button
        $this->add_control(
            'xlab_secondary_button_show',
            [
                'label' => esc_html__( 'Show Button', 'bixola-core' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Show', 'bixola-core' ),
                'label_off' => esc_html__( 'Hide', 'bixola-core' ),
                'return_value' => 'yes',
                'default' => 'yes',
                'condition' => [
                    'xlab_design_style' => [ 'layout-2', 'layout-3' ],
                ],
            ]
        );

        $this->add_control(
            'xlab_secondary_btn_text',
            [
                'label' => esc_html__('Button Text', 'bixola-core'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Contact Us', 'bixola-core'),
                'label_block' => true,
                'condition' => [
                    'xlab_design_style' => [ 'layout-3' ],
                    'xlab_secondary_button_show' => 'yes'
                ],
            ]
        );

        $this->add_control(
            'xlab_secondary_btn_link_type',
            [
                'label' => esc_html__('Button Link Type', 'bixola-core'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '1' => 'Custom Link',
                    '2' => 'Internal Page',
                ],
                'default' => '1',
                'label_block' => true,
                'condition' => [
                    'xlab_design_style' => [ 'layout-2', 'layout-3' ],
                    'xlab_secondary_button_show' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'xlab_secondary_btn_link',
            [
                'label' => esc_html__('Button link', 'bixola-core'),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => esc_html__('https://your-link.com', 'bixola-core'),
                'show_external' => false,
                'default' => [
                    'url' => '#',
                    'is_external' => true,
                    'nofollow' => true,
                    'custom_attributes' => '',
                ],
                'condition' => [
                    'xlab_design_style' => [ 'layout-2', 'layout-3' ],
                    'xlab_secondary_btn_link_type' => '1',
                    'xlab_secondary_button_show' => 'yes',
                ],
                'label_block' => true,
            ]
        );

        $this->add_control(
            'xlab_secondary_btn_page_link',
            [
                'label' => esc_html__('Select Button Page', 'bixola-core'),
                'type' => Controls_Manager::SELECT2,
                'label_block' => true,
                'options' => xlab_get_all_pages(),
                'condition' => [
                    'xlab_design_style' => [ 'layout-2', 'layout-3' ],
                    'xlab_secondary_btn_link_type' => '2',
                    'xlab_secondary_button_show' => 'yes',
                ]
            ]
        );

        $this->end_controls_section();

        // xlab_rating
        $this->start_controls_section(
            'xlab_hero_rating_content',
            [
                'label' => esc_html__('Rating', 'bixola-core'),
                'condition' => [
                    'xlab_design_style' => [ 'layout-3' ],
                ],
            ]
        );

        $this->add_control(
            'xlab_hero_rating_image_one',
            [
                'label' => esc_html__( 'Image One', 'bixola-core' ),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
                'condition' => [
                    'xlab_design_style' => [ 'layout-3' ],
                ],
            ]
        );

        $this->add_control(
            'xlab_hero_rating_image_two',
            [
                'label' => esc_html__( 'Image Two', 'bixola-core' ),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
                'condition' => [
                    'xlab_design_style' => [ 'layout-3' ],
                ],
            ]
        );

        $this->add_control(
            'xlab_hero_rating_image_three',
            [
                'label' => esc_html__( 'Image Three', 'bixola-core' ),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
                'condition' => [
                    'xlab_design_style' => [ 'layout-3' ],
                ],
            ]
        );

        $this->add_control(
            'xlab_hero_rating_number',
            [
                'label' => esc_html__('Rating Number', 'bixola-core'),
                'type' => Controls_Manager::TEXT,
                'default' => __('3,000+', 'bixola-core'),
                'label_block' => true,
                'condition' => [
                    'xlab_design_style' => [ 'layout-3' ],
                ],
            ]
        );

        $this->add_control(
            'xlab_hero_rating_text',
            [
                'label' => esc_html__('Rating Text', 'bixola-core'),
                'type' => Controls_Manager::TEXT,
                'default' => __('Satisfied Customer', 'bixola-core'),
                'label_block' => true,
                'condition' => [
                    'xlab_design_style' => [ 'layout-3' ],
                ],
            ]
        );

        $this->end_controls_section();

        // Style Control
		$this->start_controls_section(
            '_style_design_layout',
            [
                'label' => esc_html__( 'Design Layout', 'bixola-core' ),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'design_layout_padding',
            [
                'label' => __( 'Padding', 'bixola-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .banner-1' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .banner-2' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .banner-3' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'design_layout_background',
            [
                'label' => esc_html__( 'Background', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .banner-1' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .banner-2' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .banner-3' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'design_layout_shape_1',
            [
                'label' => esc_html__( 'Shape 1', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .big-shape.big-shape-1 svg path' => 'fill: {{VALUE}}',
                    '{{WRAPPER}} .small-shape.small-shape-2 svg path' => 'fill: {{VALUE}}',
                    '{{WRAPPER}} .small-shape.small-shape-3 svg path' => 'fill: {{VALUE}}',
                ],
                'condition' => [
                    'xlab_design_style' => [ 'layout-1' ],
                ],
            ]
        );

        $this->add_control(
            'design_layout_shape_2',
            [
                'label' => esc_html__( 'Shape 2', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .big-shape.big-shape-2 svg path' => 'fill: {{VALUE}}',
                    '{{WRAPPER}} .small-shape.small-shape-4 svg path' => 'fill: {{VALUE}}',
                    '{{WRAPPER}} .banner-1__shapes .small-shape-1 svg path' => 'fill: {{VALUE}}',
                ],
                'condition' => [
                    'xlab_design_style' => [ 'layout-1' ],
                ],
            ]
        );

        $this->add_control(
            'design_layout_shape_3',
            [
                'label' => esc_html__( 'Shape 3', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .big-shape.big-shape-3 svg path' => 'fill: {{VALUE}}',
                    '{{WRAPPER}} .banner-1__shapes .big-shape-4 svg path' => 'fill: {{VALUE}}',
                ],
                'condition' => [
                    'xlab_design_style' => [ 'layout-1' ],
                ],
            ]
        );

        $this->add_control(
            'design_layout_shape',
            [
                'label' => esc_html__( 'Shape Color', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .banner-2__bottom-media__shape-2' => 'background-color: {{VALUE}}',
                ],
                'condition' => [
                    'xlab_design_style' => [ 'layout-2' ],
                ],
            ]
        );

        $this->end_controls_section();

		$this->start_controls_section(
            '_style_title',
            [
                'label' => esc_html__( 'Title & Content', 'bixola-core' ),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        // Title
        $this->add_control(
            '_heading_title',
            [
                'type' => Controls_Manager::HEADING,
                'label' => esc_html__( 'Title', 'bixola-core' ),
                'separator' => 'before'
            ]
        );

        $this->add_responsive_control(
            'title_spacing',
            [
                'label' => __( 'Bottom Spacing', 'bixola-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .banner-1__content h1' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .banner-2__top-left h1' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .banner-3__content h1' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'title_color',
            [
                'label' => __( 'Color', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .banner-1__content h1' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .banner-2__top-left h1' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .banner-3__content h1' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'title_highlight_color',
            [
                'label' => __( 'Highlight', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .banner-1__content h1 span' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .banner-2__top-left h1 span' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .banner-3__content h1 span' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'title_typography',
                'selector' => '{{WRAPPER}} .banner-1__content h1, .banner-2__top-left h1, .banner-3__content h1',
            ]
        );

        // Description
        $this->add_control(
            '_content_description',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Description', 'bixola-core' ),
                'separator' => 'before'
            ]
        );

        $this->add_responsive_control(
            'description_spacing',
            [
                'label' => __( 'Bottom Spacing', 'bixola-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .banner-1__content p' => 'margin-bottom: {{SIZE}}{{UNIT}}!important;',
                    '{{WRAPPER}} .banner-2__top-left p' => 'margin-bottom: {{SIZE}}{{UNIT}}!important;',
                    '{{WRAPPER}} .banner-3__content p' => 'margin-bottom: {{SIZE}}{{UNIT}}!important;',
                ],
            ]
        );

        $this->add_control(
            'description_color',
            [
                'label' => esc_html__( 'Color', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .banner-1__content p' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .banner-2__top-left p' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .banner-3__content p' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'description_typography',
                'selector' => '{{WRAPPER}} .banner-1__content p, .banner-2__top-left p, .banner-3__content p',
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->end_controls_section();

        $this->start_controls_section(
			'xlab_button_style',
			[
				'label' => __( 'Button', 'bixola-core' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

        $this->start_controls_tabs( 'tabs_button_style' );

        $this->start_controls_tab(
            'button_tab',
            [
                'label' => esc_html__( 'Normal', 'bixola-core' ),
            ]
        );

        $this->add_control(
            'button_color',
            [
                'label'     => esc_html__( 'Color', 'bixola-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .rs-btn'    => 'color: {{VALUE}}',
                    '{{WRAPPER}} .rs-btn.btn-transparent'    => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'button_background',
            [
                'label'     => esc_html__( 'Background', 'bixola-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .rs-btn' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .rs-btn.btn-transparent' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'label'    => esc_html__( 'Border', 'bixola-core' ),
                'name'     => 'button_border',
                'selector' => '{{WRAPPER}} .rs-btn, .rs-btn.btn-transparent',
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'button_hover_tab',
            [
                'label' => esc_html__( 'Hover', 'bixola-core' ),
            ]
        );

        $this->add_control(
            'button_color_hover',
            [
                'label'     => esc_html__( 'Color', 'bixola-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .rs-btn.btn-hover-white:hover' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .rs-btn:hover' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'button_background_hover',
            [
                'label'     => esc_html__( 'Background', 'bixola-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .rs-btn.btn-hover-white::before' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .rs-btn.btn-transparent::before' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'label'    => esc_html__( 'Border', 'bixola-core' ),
                'name'     => 'button_border_hover',
                'selector' => '{{WRAPPER}} .rs-btn:hover',
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->add_control(
            'button_border_radius',
            [
                'label'     => esc_html__( 'Border Radius', 'bixola-core' ),
                'type'      => Controls_Manager::SLIDER,
                'selectors' => [
                    '{{WRAPPER}} .rs-btn' => 'border-radius: {{SIZE}}px;',
                    '{{WRAPPER}} .rs-btn.btn-hover-white::before' => 'border-radius: {{SIZE}}px;',
                    '{{WRAPPER}} .rs-btn.btn-transparent::before' => 'border-radius: {{SIZE}}px;',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'label'    => esc_html__( 'Typography', 'bixola-core' ),
                'name'     => 'button_typography',
                'selector' => '{{WRAPPER}} .rs-btn, .rs-btn.btn-transparent',
            ]
        );

        $this->add_control(
            'button_padding',
            [
                'label'      => esc_html__( 'Padding', 'bixola-core' ),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .rs-btn' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .rs-btn.btn-transparent' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'button_margin',
            [
                'label'      => esc_html__( 'Margin', 'bixola-core' ),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .rs-btn' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .rs-btn.btn-transparent' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            '_heading_button_secondary',
            [
                'label' => esc_html__( 'Button Secondary', 'bixola-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
                'condition' => [
                    'xlab_design_style' => [ 'layout-3' ],
                ],
            ]
        );

        $this->start_controls_tabs( 'tabs_button_secondary_style' );

        $this->start_controls_tab(
            'button_secondary_tab',
            [
                'label' => esc_html__( 'Normal', 'bixola-core' ),
                'condition' => [
                    'xlab_design_style' => [ 'layout-3' ],
                ],
            ]
        );

        $this->add_control(
            'button_secondary_color',
            [
                'label'     => esc_html__( 'Color', 'bixola-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .rs-btn.btn-transparent'    => 'color: {{VALUE}}',
                ],
                'condition' => [
                    'xlab_design_style' => [ 'layout-3' ],
                ],
            ]
        );

        $this->add_control(
            'button_secondary_background',
            [
                'label'     => esc_html__( 'Background', 'bixola-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .rs-btn.btn-transparent' => 'background-color: {{VALUE}}',
                ],
                'condition' => [
                    'xlab_design_style' => [ 'layout-3' ],
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'label'    => esc_html__( 'Border', 'bixola-core' ),
                'name'     => 'button_secondary_border',
                'selector' => '{{WRAPPER}} .rs-btn.btn-transparent',
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'button_secondary_hover_tab',
            [
                'label' => esc_html__( 'Hover', 'bixola-core' ),
                'condition' => [
                    'xlab_design_style' => [ 'layout-3' ],
                ],
            ]
        );

        $this->add_control(
            'button_secondary_color_hover',
            [
                'label'     => esc_html__( 'Color', 'bixola-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .rs-btn.btn-transparent:hover' => 'color: {{VALUE}};',
                ],
                'condition' => [
                    'xlab_design_style' => [ 'layout-3' ],
                ],
            ]
        );

        $this->add_control(
            'button_secondary_background_hover',
            [
                'label'     => esc_html__( 'Background', 'bixola-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .rs-btn.btn-transparent::before' => 'background-color: {{VALUE}}',
                ],
                'condition' => [
                    'xlab_design_style' => [ 'layout-3' ],
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'label'    => esc_html__( 'Border', 'bixola-core' ),
                'name'     => 'button_secondary_border_hover',
                'selector' => '{{WRAPPER}} .rs-btn.btn-transparent:hover',
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->add_control(
            'button_secondary_border_radius',
            [
                'label'     => esc_html__( 'Border Radius', 'bixola-core' ),
                'type'      => Controls_Manager::SLIDER,
                'selectors' => [
                    '{{WRAPPER}} .rs-btn.btn-transparent' => 'border-radius: {{SIZE}}px;',
                ],
                'condition' => [
                    'xlab_design_style' => [ 'layout-3' ],
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'label'    => esc_html__( 'Typography', 'bixola-core' ),
                'name'     => 'button_secondary_typography',
                'selector' => '{{WRAPPER}} .rs-btn.btn-transparent',
                'condition' => [
                    'xlab_design_style' => [ 'layout-3' ],
                ],
            ]
        );

        $this->add_control(
            'button_secondary_padding',
            [
                'label'      => esc_html__( 'Padding', 'bixola-core' ),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .rs-btn.btn-transparent' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'condition' => [
                    'xlab_design_style' => [ 'layout-3' ],
                ],
            ]
        );

        $this->add_control(
            'button_secondary_margin',
            [
                'label'      => esc_html__( 'Margin', 'bixola-core' ),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .rs-btn.btn-transparent' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'condition' => [
                    'xlab_design_style' => [ 'layout-3' ],
                ],
            ]
        );

		$this->end_controls_section();

        $this->start_controls_section(
			'_style_rating',
			[
				'label' => __( 'Rating', 'bixola-core' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

        $this->add_control(
            '_heading_rating_icon',
            [
                'label' => esc_html__( 'Icon', 'bixola-core' ),
                'type' => Controls_Manager::HEADING,
            ]
        );

        $this->add_control(
            'rating_icon_color',
            [
                'label' => esc_html__( 'Color', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .banner-3__item-left span.custom-img i' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'rating_icon_background',
            [
                'label' => esc_html__( 'background', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .banner-3__item-left span.custom-img' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'rating_icon_border_color',
            [
                'label' => esc_html__( 'Border Color', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .banner-3__item-left span.custom-img' => 'border-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            '_heading_rating_number',
            [
                'label' => esc_html__( 'Number', 'bixola-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'rating_number_spacing',
            [
                'label' => __( 'Bottom Spacing', 'bixola-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .banner-3__item-right h3' => 'margin-bottom: {{SIZE}}{{UNIT}}!important;',
                ],
            ]
        );

        $this->add_control(
            'rating_number_color',
            [
                'label' => esc_html__( 'Color', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .banner-3__item-right h3' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'label'    => esc_html__( 'Typography', 'bixola-core' ),
                'name'     => 'rating_number_typography',
                'selector' => '{{WRAPPER}} .banner-3__item-right h3',
            ]
        );

        $this->add_control(
            '_heading_rating_text',
            [
                'label' => esc_html__( 'Text', 'bixola-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'rating_text_spacing',
            [
                'label' => __( 'Bottom Spacing', 'bixola-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .banner-3__item-right span' => 'margin-bottom: {{SIZE}}{{UNIT}}!important;',
                ],
            ]
        );

        $this->add_control(
            'rating_text_color',
            [
                'label' => esc_html__( 'Color', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .banner-3__item-right span' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'label'    => esc_html__( 'Typography', 'bixola-core' ),
                'name'     => 'rating_text_typography',
                'selector' => '{{WRAPPER}} .banner-3__item-right span',
            ]
        );

        $this->add_control(
            '_heading_rating_layout',
            [
                'label' => esc_html__( 'Layout', 'bixola-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'rating_layout_padding',
            [
                'label' => __( 'Padding', 'bixola-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .banner-3__item' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'rating_layout_border_radius',
            [
                'label' => __( 'Border Radius', 'bixola-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .banner-3__item' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'rating_layout_background',
            [
                'label' => esc_html__( 'Background', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .banner-3__item' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'label'    => esc_html__( 'Border', 'bixola-core' ),
                'name'     => 'rating_layout_border',
                'selector' => '{{WRAPPER}} .banner-3__item',
            ]
        );

        
		$this->end_controls_section();

	}

	/**
	 * Render the widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();
		?>

		<?php if ( $settings['xlab_design_style']  == 'layout-1' ): 

            if ( !empty($settings['xlab_hero_image']['url']) ) {
                $xlab_hero_image = !empty($settings['xlab_hero_image']['id']) ? wp_get_attachment_image_url( $settings['xlab_hero_image']['id'], 'full') : $settings['xlab_hero_image']['url'];
                $xlab_hero_image_alt = get_post_meta($settings["xlab_hero_image"]["id"], "_wp_attachment_image_alt", true);
            }
            
            // Link
            if ('2' == $settings['xlab_btn_link_type']) {
                $this->add_render_attribute('xlab-button-arg', 'href', get_permalink($settings['xlab_btn_page_link']));
                $this->add_render_attribute('xlab-button-arg', 'target', '_self');
                $this->add_render_attribute('xlab-button-arg', 'rel', 'nofollow');
                $this->add_render_attribute('xlab-button-arg', 'class', 'rs-btn btn-hover-white wow fadeIn animated');
                $this->add_render_attribute('xlab-button-arg', 'data-wow-delay', '.5s');
            } else {
                if ( ! empty( $settings['xlab_btn_link']['url'] ) ) {
                    $this->add_link_attributes( 'xlab-button-arg', $settings['xlab_btn_link'] );
                    $this->add_render_attribute('xlab-button-arg', 'class', 'rs-btn btn-hover-white wow fadeIn animated');
                }
            }
            
            ?>

            <section class="banner-1 banner-1__space overflow-hidden rs-theme-100">
                <div class="container">
                    <div class="banner-1__shapes">
                        <div class="big-shape big-shape-3" data-parallax='{"scale": 200, "smoothness": 15}'>
                            <svg xmlns="http://www.w3.org/2000/svg" width="910" height="912" viewBox="0 0 910 912" fill="none">
                                <path opacity="0.4" d="M0 0L910 910L0 911.39V0Z" fill="#00160A"/>
                            </svg>
                        </div>

                        <div class="small-shape small-shape-1" data-parallax='{"y": -200, "x": 300, "smoothness": 15}'>
                            <svg xmlns="http://www.w3.org/2000/svg" width="519" height="519" viewBox="0 0 519 519" fill="none">
                                <path d="M518.873 259.436L259.438 0L0.00179011 259.436L259.438 518.871L518.873 259.436Z" fill="#00D563"/>
                            </svg>
                        </div>

                        <div class="big-shape big-shape-2" data-parallax='{"y": 300, "smoothness": 15}'>
                            <svg xmlns="http://www.w3.org/2000/svg" width="1854" height="1854" viewBox="0 0 1854 1854" fill="none">
                                <path d="M1853.98 926.992L926.992 0L0.00043509 926.992L926.992 1853.98L1853.98 926.992Z" fill="#00D563"/>
                            </svg>
                        </div>

                        <div class="big-shape big-shape-1" data-parallax='{"y": -20, "scale": 1.3, "smoothness": 15}'>
                            <svg xmlns="http://www.w3.org/2000/svg" width="1854" height="1854" viewBox="0 0 1854 1854" fill="none">
                                <path d="M1853.98 926.992L926.992 0L0.00043509 926.992L926.992 1853.98L1853.98 926.992Z" fill="#001C0D"/>
                            </svg>
                        </div>

                        <div class="big-shape big-shape-4" data-parallax='{"y": 800, "smoothness": 15}'>
                            <svg xmlns="http://www.w3.org/2000/svg" width="910" height="912" viewBox="0 0 910 912" fill="none">
                                <path opacity="0.4" d="M910 0L-5.44725e-06 910L910 911.39V0Z" fill="#484949"/>
                            </svg>
                        </div>

                        <div class="small-shape small-shape-2">
                            <svg xmlns="http://www.w3.org/2000/svg" width="519" height="185" viewBox="0 0 519 185" fill="none">
                                <path d="M518.904 259.436L259.469 0L0.0330401 259.436L259.469 518.871L518.904 259.436Z" fill="#0D2216"/>
                            </svg>
                        </div>

                        <div class="small-shape small-shape-3">
                            <svg xmlns="http://www.w3.org/2000/svg" width="594" height="594" viewBox="0 0 594 594" fill="none">
                                <path d="M593.236 296.619L296.617 0L-0.0015627 296.619L296.617 593.238L593.236 296.619Z" fill="#0D2216"/>
                            </svg>
                        </div>

                        <div class="small-shape small-shape-4" data-parallax='{"y": 100, "smoothness": 15}'>
                            <svg xmlns="http://www.w3.org/2000/svg" width="594" height="594" viewBox="0 0 594 594" fill="none">
                                <path d="M593.236 296.619L296.617 0L-0.0015627 296.619L296.617 593.238L593.236 296.619Z" fill="#00D563"/>
                            </svg>
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-xl-6 col-lg-7 col-md-8 col-sm-9">
                            <div class="banner-1__content" data-parallax='{"scale": 1.5, "y":300, "smoothness": 15}'>
                                <?php if ( !empty($settings['xlab_title']) ) : ?>
                                    <h1 class="mb-25 wow fadeIn animated" data-wow-delay=".1s"><?php print twinkle_kses( $settings['xlab_title' ] ); ?></h1>
                                <?php endif; ?>
                                <?php if ( !empty($settings['xlab_description']) ) : ?>
                                <div class="description mb-30 wow fadeIn animated" data-wow-delay=".3s">
                                    <p class="mb-0"><?php echo twinkle_kses( $settings['xlab_description'] ); ?></p>
                                </div>
                                <?php endif; ?>
                                <?php if (!empty($settings['xlab_btn_text'])) : ?>
                                    <a <?php echo $this->get_render_attribute_string( 'xlab-button-arg' ); ?>>
                                        <?php echo $settings['xlab_btn_text']; ?>
                                    </a>
                                <?php endif; ?>
                            </div>
                        </div>
                        <div class="col-xl-6 col-lg-5 col-md-4 col-sm-3">
                            <div class="banner-1__media">
                                <img data-parallax='{"scale": 1.5, "y":100, "smoothness": 15}' src="<?php print esc_url($xlab_hero_image); ?>" alt="<?php print esc_attr($xlab_hero_image_alt); ?>">
                            </div>
                        </div>
                    </div>
                </div>
            </section>

        <?php elseif ( $settings['xlab_design_style']  == 'layout-2' ):
            $this->add_render_attribute('title_args', 'class', 'wow fadeInUp section-title__title');
            $this->add_render_attribute('title_args', 'data-wow-delay', '.3s');

            if ( !empty($settings['xlab_hero_image']['url']) ) {
                $xlab_hero_image = !empty($settings['xlab_hero_image']['id']) ? wp_get_attachment_image_url( $settings['xlab_hero_image']['id'], 'full') : $settings['xlab_hero_image']['url'];
                $xlab_hero_image_alt = get_post_meta($settings["xlab_hero_image"]["id"], "_wp_attachment_image_alt", true);
            }

            // Link
            if ('2' == $settings['xlab_btn_link_type']) {
                $this->add_render_attribute('xlab-button-arg', 'href', get_permalink($settings['xlab_btn_page_link']));
                $this->add_render_attribute('xlab-button-arg', 'target', '_self');
                $this->add_render_attribute('xlab-button-arg', 'rel', 'nofollow');
                $this->add_render_attribute('xlab-button-arg', 'class', 'rs-btn rs-btn btn-transparent wow fadeIn animated');
                $this->add_render_attribute('xlab-button-arg', 'data-wow-delay', '.4s');
            } else {
                if ( ! empty( $settings['xlab_btn_link']['url'] ) ) {
                    $this->add_link_attributes( 'xlab-button-arg', $settings['xlab_btn_link'] );
                    $this->add_render_attribute('xlab-button-arg', 'class', 'rs-btn rs-btn btn-transparent wow fadeIn animated');
                }
            }

            // Link
            if ('2' == $settings['xlab_secondary_btn_link_type']) {
                $this->add_render_attribute('xlab-secondary-button-arg', 'href', get_permalink($settings['xlab_secondary_btn_page_link']));
                $this->add_render_attribute('xlab-secondary-button-arg', 'target', '_self');
                $this->add_render_attribute('xlab-secondary-button-arg', 'rel', 'nofollow');
                $this->add_render_attribute('xlab-secondary-button-arg', 'class', 'banner-2-circle wow fadeIn animated');
                $this->add_render_attribute('xlab-secondary-button-arg', 'data-wow-delay', '.1s');
            } else {
                if ( ! empty( $settings['xlab_secondary_btn_link']['url'] ) ) {
                    $this->add_link_attributes( 'xlab-secondary-button-arg', $settings['xlab_secondary_btn_link'] );
                    $this->add_render_attribute('xlab-secondary-button-arg', 'class', 'banner-2-circle wow fadeIn animated');
                }
            }
        ?>

        <section class="banner-2 overflow-hidden">
            <div class="container">
                <div class="row">
                    <div class="col-12">
                        <div class="banner-2__top d-flex flex-column flex-sm-row justify-content-between align-items-center mb-40">
                            <div class="banner-2__top-left">
                                <?php if ( !empty($settings['xlab_title']) ) : ?>
                                    <h1 class="mb-20  wow fadeIn animated" data-wow-delay=".1s"><?php print twinkle_kses( $settings['xlab_title' ] ); ?></span></h1>
                                <?php endif; ?>
                                <?php if ( !empty($settings['xlab_description']) ) : ?>
                                    <p class=" wow fadeIn animated" data-wow-delay=".3s"><?php echo twinkle_kses( $settings['xlab_description'] ); ?></p>
                                <?php endif; ?>
                            </div>
                            
                            <?php if (!empty($settings['xlab_secondary_button_show'])) : ?>
                                <div class="banner-2__top-right">
                                    <a <?php echo $this->get_render_attribute_string( 'xlab-secondary-button-arg' ); ?>>
                                        <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32" fill="none">
                                            <path d="M25.5039 8V21C25.5039 21.3978 25.3458 21.7794 25.0645 22.0607C24.7832 22.342 24.4017 22.5 24.0039 22.5C23.606 22.5 23.2245 22.342 22.9432 22.0607C22.6619 21.7794 22.5039 21.3978 22.5039 21V11.625L9.0651 25.0613C8.78331 25.343 8.40112 25.5014 8.0026 25.5014C7.60409 25.5014 7.22189 25.343 6.9401 25.0613C6.65831 24.7795 6.5 24.3973 6.5 23.9988C6.5 23.6002 6.65831 23.218 6.9401 22.9363L20.3789 9.5H11.0039C10.606 9.5 10.2245 9.34196 9.94319 9.06066C9.66189 8.77936 9.50385 8.39782 9.50385 8C9.50385 7.60218 9.66189 7.22064 9.94319 6.93934C10.2245 6.65804 10.606 6.5 11.0039 6.5H24.0039C24.4017 6.5 24.7832 6.65804 25.0645 6.93934C25.3458 7.22064 25.5039 7.60218 25.5039 8Z" fill="#00D563"/>
                                        </svg>
                                    </a>
                                </div>
                            <?php endif; ?>
                        </div>

                        <div class="banner-2__bottom">
                            <div class="banner-2__bottom-media">
                                <img data-parallax='{"scale": 1.1, "smoothness": 15}' src="<?php print esc_url($xlab_hero_image); ?>" alt="<?php print esc_url($xlab_hero_image_alt); ?>" class="img-fluid">
                            </div>

                            <div class="banner-2__bottom-media__shape-1 wow fadeIn animated" data-wow-delay=".4s" data-background="<?php echo get_template_directory_uri(); ?>/assets/img/banner/banner-2__bottom-media__shape-1.png"></div>
                            <div class="banner-2__bottom-media__shape-2 wow fadeIn animated" data-wow-delay=".4s"></div>

                            <?php if (!empty($settings['xlab_btn_text'])) : ?>
                                <div class="banner-2__bottom-button" data-background="<?php echo get_template_directory_uri(); ?>/assets/img/banner/banner-2__bottom-media__shape-2.png">
                                    <a <?php echo $this->get_render_attribute_string( 'xlab-button-arg' ); ?>><?php echo $settings['xlab_btn_text']; ?></a>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        </section>

        <?php elseif ( $settings['xlab_design_style']  == 'layout-3' ):

            if ( !empty($settings['xlab_hero_image_shape']['url']) ) {
                $xlab_hero_image_shape = !empty($settings['xlab_hero_image_shape']['id']) ? wp_get_attachment_image_url( $settings['xlab_hero_image_shape']['id'], 'full') : $settings['xlab_hero_image_shape']['url'];
                $xlab_hero_image_shape_alt = get_post_meta($settings["xlab_hero_image_shape"]["id"], "_wp_attachment_image_alt", true);
            }

            if ( !empty($settings['xlab_hero_image']['url']) ) {
                $xlab_hero_image = !empty($settings['xlab_hero_image']['id']) ? wp_get_attachment_image_url( $settings['xlab_hero_image']['id'], 'full') : $settings['xlab_hero_image']['url'];
                $xlab_hero_image_alt = get_post_meta($settings["xlab_hero_image"]["id"], "_wp_attachment_image_alt", true);
            }

            if ( !empty($settings['xlab_hero_rating_image_one']['url']) ) {
                $xlab_hero_rating_image_one = !empty($settings['xlab_hero_rating_image_one']['id']) ? wp_get_attachment_image_url( $settings['xlab_hero_rating_image_one']['id'], 'full') : $settings['xlab_hero_rating_image_one']['url'];
                $xlab_hero_rating_image_one_alt = get_post_meta($settings["xlab_hero_rating_image_one"]["id"], "_wp_attachment_image_alt", true);
            }

            if ( !empty($settings['xlab_hero_rating_image_two']['url']) ) {
                $xlab_hero_rating_image_two = !empty($settings['xlab_hero_rating_image_two']['id']) ? wp_get_attachment_image_url( $settings['xlab_hero_rating_image_two']['id'], 'full') : $settings['xlab_hero_rating_image_two']['url'];
                $xlab_hero_rating_image_two_alt = get_post_meta($settings["xlab_hero_rating_image_two"]["id"], "_wp_attachment_image_alt", true);
            }

            if ( !empty($settings['xlab_hero_rating_image_three']['url']) ) {
                $xlab_hero_rating_image_three = !empty($settings['xlab_hero_rating_image_three']['id']) ? wp_get_attachment_image_url( $settings['xlab_hero_rating_image_three']['id'], 'full') : $settings['xlab_hero_rating_image_three']['url'];
                $xlab_hero_rating_image_three_alt = get_post_meta($settings["xlab_hero_rating_image_three"]["id"], "_wp_attachment_image_alt", true);
            }

            // Link
            if ('2' == $settings['xlab_btn_link_type']) {
                $this->add_render_attribute('xlab-button-arg', 'href', get_permalink($settings['xlab_btn_page_link']));
                $this->add_render_attribute('xlab-button-arg', 'target', '_self');
                $this->add_render_attribute('xlab-button-arg', 'rel', 'nofollow');
                $this->add_render_attribute('xlab-button-arg', 'class', 'rs-btn');
            } else {
                if ( ! empty( $settings['xlab_btn_link']['url'] ) ) {
                    $this->add_link_attributes( 'xlab-button-arg', $settings['xlab_btn_link'] );
                    $this->add_render_attribute('xlab-button-arg', 'class', 'rs-btn');
                }
            }

            // Link
            if ('2' == $settings['xlab_secondary_btn_link_type']) {
                $this->add_render_attribute('xlab-secondary-button-arg', 'href', get_permalink($settings['xlab_secondary_btn_page_link']));
                $this->add_render_attribute('xlab-secondary-button-arg', 'target', '_self');
                $this->add_render_attribute('xlab-secondary-button-arg', 'rel', 'nofollow');
                $this->add_render_attribute('xlab-secondary-button-arg', 'class', 'rs-btn btn-transparent');
                $this->add_render_attribute('xlab-secondary-button-arg', 'data-wow-delay', '.1s');
            } else {
                if ( ! empty( $settings['xlab_secondary_btn_link']['url'] ) ) {
                    $this->add_link_attributes( 'xlab-secondary-button-arg', $settings['xlab_secondary_btn_link'] );
                    $this->add_render_attribute('xlab-secondary-button-arg', 'class', 'rs-btn btn-transparent');
                }
            }
        ?>

        <div class="banner-3 background-image space overflow-hidden rs-gray-100" data-background="<?php print esc_url($xlab_hero_image_shape); ?>">
            <div class="container">
                <div class="row align-items-center">
                    <div class="col-xl-5">
                        <div class="banner-3__content">
                            <?php if ( !empty($settings['xlab_title']) ) : ?>
                                <h1 class="mb-30 mb-xs-20 fadeIn animated" data-wow-delay=".1s"><?php print twinkle_kses( $settings['xlab_title' ] ); ?></h1>
                            <?php endif; ?>
                            <?php if ( !empty($settings['xlab_description']) ) : ?>
                                <p class="mb-0 fadeIn animated" data-wow-delay=".3s"><?php echo twinkle_kses( $settings['xlab_description'] ); ?></p>
                            <?php endif; ?>
                            
                            <div class="rs-btn__wrapper d-flex flex-wrap mt-40 mt-md-35 mt-sm-30 mt-xs-25 fadeIn animated" data-wow-delay=".6s">
                                <?php if (!empty($settings['xlab_btn_text'])) : ?>
                                    <a <?php echo $this->get_render_attribute_string( 'xlab-button-arg' ); ?>>
                                        <?php echo $settings['xlab_btn_text']; ?>
                                    </a>
                                <?php endif; ?>
                                <?php if (!empty($settings['xlab_secondary_btn_text'])) : ?>
                                    <a <?php echo $this->get_render_attribute_string( 'xlab-secondary-button-arg' ); ?>><?php echo esc_html($settings['xlab_secondary_btn_text']); ?></a>
                                <?php endif; ?>
                            </div>

                            <div class="banner-3__item d-inline-flex flex-sm-row flex-column align-items-center justify-content-sm-start mt-50 mt-md-35 mt-sm-30 mt-xs-25 fadeIn animated" data-wow-delay=".9s">
                                <div class="banner-3__item-left d-flex align-items-center flex-row">
                                    <img src="<?php echo esc_url($xlab_hero_rating_image_one); ?>" alt="<?php echo esc_attr($xlab_hero_rating_image_one_alt); ?>">
                                    <img src="<?php echo esc_url($xlab_hero_rating_image_two); ?>" alt="<?php echo esc_attr($xlab_hero_rating_image_two_alt); ?>">
                                    <img src="<?php echo esc_url($xlab_hero_rating_image_three); ?>" alt="<?php echo esc_attr($xlab_hero_rating_image_three_alt); ?>">
                                    <span class="custom-img"><i class="fa-solid fa-plus"></i></span>
                                </div>
                                <div class="banner-3__item-right">
                                    <h3 class="mb-0"><?php print esc_html($settings['xlab_hero_rating_number']); ?></h3>
                                    <span><?php print esc_html($settings['xlab_hero_rating_text']); ?></span>
                                </div>
                            </div>
                        </div>
                    </div>

                    <div class="col-xl-7">
                        <div class="banner-3__media">
                            <div class="banner-3__media__shape" data-background="<?php echo get_template_directory_uri(); ?>/assets/img/banner-3/banner-media__shape.svg">
                                <div class="img">
                                    <img src="<?php print esc_url($xlab_hero_image); ?>" alt="<?php print esc_url($xlab_hero_image_alt); ?>">
                                </div>
                            </div>

                            <img class="banner-image-mobile d-block d-xl-none img-fluid" src="<?php echo get_template_directory_uri(); ?>/assets/img/banner-3/banner-img-mobile.jpg" alt="">
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <?php endif; ?>

		<?php
	}
}

$widgets_manager->register( new Xlab_Hero() );