<?php
namespace BixolaCore\Widgets;

use \Elementor\Widget_Base;
use \Elementor\Controls_Manager;
use \Elementor\Group_Control_Background;
use \Elementor\Group_Control_Image_Size;
use \Elementor\Group_Control_Border;
use \Elementor\Group_Control_Typography;
use \Elementor\Repeater;
use \Elementor\Utils;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Bixola Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class Bixola_Brand_Slider extends Widget_Base {

	/**
	 * Retrieve the widget name.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'bixola_brand_slider';
	}

	/**
	 * Retrieve the widget title.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Brand Slider', 'bixola-core' );
	}

	/**
	 * Retrieve the widget icon.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'bixola-icon';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * Note that currently Elementor supports only one category.
	 * When multiple categories passed, Elementor uses the first one.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'bixola_core' ];
	}

	/**
	 * Retrieve the list of scripts the widget depended on.
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'bixola-core' ];
	}

	/**
	 * Register the widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function register_controls() {
        
        $this->start_controls_section(
            '_content_design_layout',
            [
                'label' => esc_html__('Design Layout', 'bixola-core'),
            ]
        );

        $this->add_control(
            'design_style',
            [
                'label' => esc_html__('Select Layout', 'bixola-core'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'layout-1' => esc_html__('Layout 1', 'bixola-core'),
                    'layout-2' => esc_html__('Layout 2', 'bixola-core'),
                    'layout-3' => esc_html__('Layout 3', 'bixola-core'),
                ],
                'default' => 'layout-1',
            ]
        );

        $this->add_control(
            'section_title',
            [
                'label' => esc_html__( 'Title', 'bixola-core' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'Our Clients', 'bixola-core' ),
                'label_block' => true,
                'condition' => [
                    'design_style' => 'layout-2',
                ],
            ]
        );

        $this->end_controls_section();

		$this->start_controls_section(
            '_content_brand_slider',
            [
                'label' => __( 'Brand Slider', 'bixola-core' ),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $repeater = new Repeater();

        $repeater->add_control(
            'brand_slider_image',
            [
                'label' => esc_html__('Upload Bg Image', 'bixola-core'),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ]
            ]
        );

        $this->add_control(
            'brand_slider_list',
            [
                'show_label' => false,
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'title_field' => esc_html__( 'Brand Item', 'bixola-core' ),
                'default' => [
                    [
                        'brand_slider_image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                    ],
                    [
                        'brand_slider_image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                    ],
                ]
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_style_design_layout',
            [
                'label' => esc_html__( 'Design Layout',  'text-domain'  ),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );
        
        $this->add_responsive_control(
            'design_layout_margin',
            [
                'label' => esc_html__( 'Margin', 'text-domain' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .brand-one__inner' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .brand-two__inner' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        
        $this->add_responsive_control(
            'design_layout_padding',
            [
                'label' => esc_html__( 'Padding', 'text-domain' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .brand-one__inner' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .brand-two__inner' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'design_layout_background',
            [
                'label' => esc_html__( 'Background', 'text-domain' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .brand-one' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .brand-two' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            '_heading_style_section_title',
            [
                'label' => esc_html__( 'Title', 'text-domain' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
                'condition' => [
                    'design_style' => 'layout-2',
                ],
            ]
        );
        
        $this->add_responsive_control(
            'section_title_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'text-domain' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .brand-two__inner .section__title-two' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'design_style' => 'layout-2',
                ],
            ]
        );
        
        $this->add_control(
            'section_title_color',
            [
                'label' => __( 'Color', 'text-domain' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .brand-two__inner .section__title-two' => 'color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => 'layout-2',
                ],
            ]
        );
        
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'section_title_typography',
                'selector' => '{{WRAPPER}} .brand-two__inner .section__title-two',
                'condition' => [
                    'design_style' => 'layout-2',
                ],
            ]
        );
        
        $this->end_controls_section();
		
	}

	/**
	 * Render the widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		$this->add_render_attribute('title_args', 'class', 'brand-one__title');

		?>

        <?php if ( $settings['design_style']  == 'layout-1' ) : ?>

            <section class="brand-one">
                <div class="container">
                    <div class="brand-one__inner">
                        <div class="brand-one__carousel thm-owl__carousel owl-theme owl-carousel">
                            <?php foreach ($settings['brand_slider_list'] as $item) :
                                if ( !empty($item['brand_slider_image']['url']) ) {
                                    $brand_slider_image_url = !empty($item['brand_slider_image']['id']) ? wp_get_attachment_image_url( $item['brand_slider_image']['id'], 'full') : $item['brand_slider_image']['url'];
                                    $brand_slider_image_alt = get_post_meta($item["brand_slider_image"]["id"], "_wp_attachment_image_alt", true);
                                }
                            ?>
                            <div class="brand-one__single">
                                <div class="brand-one__img">
                                    <img src="<?php echo esc_url($brand_slider_image_url); ?>" alt="<?php echo esc_attr($brand_slider_image_alt); ?>">
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                        <!-- If we need navigation buttons -->
                    </div>
                </div>
            </section>

        <?php elseif ( $settings['design_style']  == 'layout-2' ) : ?>

            <section class="brand-two">
                <div class="container">
                    <div class="brand-two__inner">
                        <?php if ( !empty ($settings['section_title']) ) : ?>
                            <h3 class="section__title-two">
                                <?php print twinkle_kses($settings['section_title']); ?>
                            </h3>
                        <?php endif; ?>
                        <div class="brand-two__carousel thm-owl__carousel owl-theme owl-carousel">
                            <?php foreach ($settings['brand_slider_list'] as $item) :
                                if ( !empty($item['brand_slider_image']['url']) ) {
                                    $brand_slider_image_url = !empty($item['brand_slider_image']['id']) ? wp_get_attachment_image_url( $item['brand_slider_image']['id'], $settings['thumbnail_size']) : $item['brand_slider_image']['url'];
                                    $brand_slider_image_alt = get_post_meta($item["brand_slider_image"]["id"], "_wp_attachment_image_alt", true);
                                }
                            ?>
                            <div class="brand-two__single">
                                <div class="brand-two__img">
                                    <img src="<?php echo esc_url($brand_slider_image_url); ?>" alt="<?php echo esc_attr($brand_slider_image_alt); ?>">
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </div>
            </section>

        <?php elseif ( $settings['design_style']  == 'layout-3' ) : ?>

            <section class="brand-one brand-three">
                <div class="container">
                    <div class="brand-one__inner">
                        <div class="brand-one__carousel thm-owl__carousel owl-theme owl-carousel">
                            <?php foreach ($settings['brand_slider_list'] as $item) :
                                if ( !empty($item['brand_slider_image']['url']) ) {
                                    $brand_slider_image_url = !empty($item['brand_slider_image']['id']) ? wp_get_attachment_image_url( $item['brand_slider_image']['id'], $settings['thumbnail_size']) : $item['brand_slider_image']['url'];
                                    $brand_slider_image_alt = get_post_meta($item["brand_slider_image"]["id"], "_wp_attachment_image_alt", true);
                                }
                            ?>
                            <div class="brand-one__single">
                                <div class="brand-one__img">
                                    <img src="<?php echo esc_url($brand_slider_image_url); ?>" alt="<?php echo esc_attr($brand_slider_image_alt); ?>">
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </div>
            </section>

        <?php endif;
	}
}
$widgets_manager->register( new Bixola_Brand_Slider() );  