<?php
namespace BixolaCore\Widgets;

use \Elementor\Widget_Base;
use \Elementor\Controls_Manager;
use \Elementor\Group_Control_Background;
use \Elementor\Group_Control_Image_Size;
use \Elementor\Group_Control_Border;
use \Elementor\Control_Box_Shadow;
use \Elementor\Group_Control_Typography;
use \Elementor\Utils;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Bixola Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class Bixola_Blog_Post extends Widget_Base {

	/**
	 * Retrieve the widget name.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'bixola_blog_post';
	}

	/**
	 * Retrieve the widget title.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Blog Post', 'bixola-core' );
	}

	/**
	 * Retrieve the widget icon.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'bixola-icon';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * Note that currently Elementor supports only one category.
	 * When multiple categories passed, Elementor uses the first one.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'bixola_core' ];
	}

	/**
	 * Retrieve the list of scripts the widget depended on.
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'bixola-core' ];
	}

	/**
	 * Register the widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function register_controls() {

        $this->start_controls_section(
            '_content_design_layout',
            [
                'label' => esc_html__( 'Design Layout', 'bixola-core' ),
            ]
        );

        $this->add_control(
            'design_style',
            [
                'label' => esc_html__( 'Style', 'bixola-core' ),
                'type' => Controls_Manager::SELECT,
                'default' => 'layout-1',
                'options' => [
                    'layout-1' => esc_html__( 'Layout 1', 'bixola-core' ),
                    'layout-2'  => esc_html__( 'Layout 2', 'bixola-core' ),
                    'layout-3'  => esc_html__( 'Layout 3', 'bixola-core' ),
                ],
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
            '_content_title',
            [
                'label' => esc_html__( 'Title & Content',  'text-domain'  ),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'section_subheading',
            [
                'label' => esc_html__( 'Subheading', 'text-domain' ),
                'description' => bixola_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'Latest Blog', 'text-domain' ),
                'label_block' => true,
            ]
        );
        
        $this->add_control(
            'section_title',
            [
                'label' => esc_html__( 'Title', 'text-domain' ),
                'description' => bixola_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'Learn about our latest <br> news from blog.', 'text-domain' ),
                'label_block' => true,
            ]
        );
        
        $this->end_controls_section();

		$this->start_controls_section(
            'post_query',
            [
                'label' => esc_html__('Blog Query', 'bixola-core'),
            ]
        );

        $post_type = 'post';
        $taxonomy = 'category';

        $this->add_control(
            'posts_per_page',
            [
                'label' => esc_html__('Posts Per Page', 'bixola-core'),
                'description' => esc_html__('Leave blank or enter -1 for all.', 'bixola-core'),
                'type' => Controls_Manager::NUMBER,
                'default' => '3',
            ]
        );

        $this->add_control(
            'category',
            [
                'label' => esc_html__('Include Categories', 'bixola-core'),
                'description' => esc_html__('Select a category to include or leave blank for all.', 'bixola-core'),
                'type' => Controls_Manager::SELECT2,
                'multiple' => true,
                'options' => bixola_get_categories($taxonomy),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'exclude_category',
            [
                'label' => esc_html__('Exclude Categories', 'bixola-core'),
                'description' => esc_html__('Select a category to exclude', 'bixola-core'),
                'type' => Controls_Manager::SELECT2,
                'multiple' => true,
                'options' => bixola_get_categories($taxonomy),
                'label_block' => true
            ]
        );

        $this->add_control(
            'post__not_in',
            [
                'label' => esc_html__('Exclude Item', 'bixola-core'),
                'type' => Controls_Manager::SELECT2,
                'options' => bixola_get_all_types_post($post_type),
                'multiple' => true,
                'label_block' => true
            ]
        );

        $this->add_control(
            'offset',
            [
                'label' => esc_html__('Offset', 'bixola-core'),
                'type' => Controls_Manager::NUMBER,
                'default' => '0',
            ]
        );

        $this->add_control(
            'orderby',
            [
                'label' => esc_html__('Order By', 'bixola-core'),
                'type' => Controls_Manager::SELECT,
                'options' => array(
			        'ID' => 'Post ID',
			        'author' => 'Post Author',
			        'title' => 'Title',
			        'date' => 'Date',
			        'modified' => 'Last Modified Date',
			        'parent' => 'Parent Id',
			        'rand' => 'Random',
			        'comment_count' => 'Comment Count',
			        'menu_order' => 'Menu Order',
			    ),
                'default' => 'date',
            ]
        );

        $this->add_control(
            'order',
            [
                'label' => esc_html__('Order', 'bixola-core'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'asc' 	=> esc_html__( 'Ascending', 'bixola-core' ),
                    'desc' 	=> esc_html__( 'Descending', 'bixola-core' )
                ],
                'default' => 'desc',

            ]
        );
        $this->add_control(
            'ignore_sticky_posts',
            [
                'label' => esc_html__( 'Ignore Sticky Posts', 'bixola-core' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Yes', 'bixola-core' ),
                'label_off' => esc_html__( 'No', 'bixola-core' ),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'blog_title_word',
            [
                'label' => esc_html__('Title Word Count', 'bixola-core'),
                'description' => esc_html__('Set how many word you want to displa!', 'bixola-core'),
                'type' => Controls_Manager::NUMBER,
                'default' => '6',
            ]
        );

        $this->add_control(
            'post_content',
            [
                'label' => __('Content', 'bixola-core'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => __('Show', 'bixola-core'),
                'label_off' => __('Hide', 'bixola-core'),
                'return_value' => 'yes',
                'default' => '',
            ]
        );

        $this->add_control(
            'post_content_limit',
            [
                'label' => __('Content Limit', 'bixola-core'),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'default' => '14',
                'dynamic' => [
                    'active' => true,
                ],
                'condition' => [
                    'post_content' => 'yes'
                ]
            ]
        );

        $this->add_control(
            'btn_button_show',
            [
                'label' => esc_html__( 'Show Button', 'bixola-core' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Show', 'bixola-core' ),
                'label_off' => esc_html__( 'Hide', 'bixola-core' ),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'btn_text',
            [
                'label' => esc_html__('Button Text', 'bixola-core'),
                'type' => Controls_Manager::TEXT,
                'default' => __('Read More', 'bixola-core'),
                'label_block' => true,
                'condition' => [
                    'btn_button_show' => 'yes'
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_style_design_layout',
            [
                'label' => __( 'Design Layout', 'bixola-core' ),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'design_layout_margin',
            [
                'label' => __( 'Margin', 'bixola-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-section' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'design_layout_padding',
            [
                'label' => __( 'Padding', 'bixola-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-section' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'design_layout_background',
            [
                'label' => __( 'Background', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-section' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'design_layout_image_hover',
            [
                'label' => __( 'Image Hover', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .news-one__img:before' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .news-two__img:after' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .news-three__img:after' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_style_title',
            [
                'label' => esc_html__( 'Title & Content',  'text-domain'  ),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );
        
        $this->add_control(
            '_heading_style_section_subheading',
            [
                'label' => esc_html__( 'Subheading', 'text-domain' ),
                'type' => Controls_Manager::HEADING,
            ]
        );
        
        $this->add_responsive_control(
            'section_subheading_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'text-domain' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-section-subheading' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        
        $this->add_control(
            'section_subheading_color',
            [
                'label' => __( 'Color', 'text-domain' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-section-subheading' => 'color: {{VALUE}}',
                ],
            ]
        );
        
        $this->add_control(
            'section_subheading_border_color',
            [
                'label' => __( 'Border Color', 'text-domain' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-section-subheading:before' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'section_subheading_typography',
                'selector' => '{{WRAPPER}} .bixola-el-section-subheading',
            ]
        );

        $this->add_control(
            '_heading_style_section_title',
            [
                'label' => esc_html__( 'Title', 'text-domain' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );
        
        $this->add_responsive_control(
            'section_title_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'text-domain' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-section-title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        
        $this->add_control(
            'section_title_color',
            [
                'label' => __( 'Color', 'text-domain' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-section-title' => 'color: {{VALUE}}',
                ],
            ]
        );
        
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'section_title_typography',
                'selector' => '{{WRAPPER}} .bixola-el-section-title',
            ]
        );
        
        $this->end_controls_section();

		$this->start_controls_section(
			'_style_blog',
			[
				'label' => __( 'Blog Query', 'bixola-core' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

        $this->add_control(
            '_heading_style_blog_title',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Title', 'bixola-core' ),
            ]
        );

        $this->add_responsive_control(
            'blog_title_spacing',
            [
                'label' => __( 'Bottom Spacing', 'bixola-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .news-one__title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .news-two__title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .news-three__title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->start_controls_tabs( '_tabs_blog_title' );
        
        $this->start_controls_tab(
            'blog_title_normal_tab',
            [
                'label' => esc_html__( 'Normal', 'bixola-core' ),
            ]
        );
        
        $this->add_control(
            'blog_title_color',
            [
                'label' => __( 'Color', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .news-one__title a' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .news-two__title a' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .news-three__title a' => 'color: {{VALUE}}',
                ],
            ]
        );
        
        $this->end_controls_tab();
        
        $this->start_controls_tab(
            'blog_title_hover_tab',
            [
                'label' => esc_html__( 'Hover', 'bixola-core' ),
            ]
        );
        
        $this->add_control(
            'blog_title_color_hover',
            [
                'label' => __( 'Color', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .news-one__title a:hover' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .news-two__title a:hover' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .news-three__title a:hover' => 'color: {{VALUE}}',
                ],
            ]
        );
        
        $this->end_controls_tab();
        
        $this->end_controls_tabs();

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'blog_title_typography',
                'selector' => '{{WRAPPER}} .news-one__title, .news-two__title, .news-three__title',
            ]
        );

        $this->add_control(
            '_heading_style_blog_description',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Description', 'bixola-core' ),
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'blog_description_spacing',
            [
                'label' => __( 'Bottom Spacing', 'bixola-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .news-one__text' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .news-three__text' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'blog_description_color',
            [
                'label' => __( 'Text Color', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .news-one__text' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .news-three__text' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'blog_description_typography',
                'selector' => '{{WRAPPER}} .news-one__text, .news-three__text',
            ]
        );

        $this->add_control(
            '_heading_style_blog_date',
            [
                'label' => esc_html__( 'Date', 'text-domain' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
                'condition' => [
                    'design_style' => 'layout-3',
                ],
            ]
        );

        $this->add_control(
            'date_color',
            [
                'label' => esc_html__( 'Color', 'text-domain' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .news-three__date p' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .news-three__date span' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'date_background',
            [
                'label' => esc_html__( 'Background', 'text-domain' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .news-three__date' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            '_heading_style_blog_category',
            [
                'label' => esc_html__( 'Category', 'text-domain' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
                'condition' => [
                    'design_style' => 'layout-1',
                ],
            ]
        );

        $this->add_responsive_control(
            'blog_category_bottom_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'text-domain' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .news-one__sub-title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'design_style' => 'layout-1',
                ],
            ]
        );

        $this->add_control(
            'blog_category_color',
            [
                'label' => esc_html__( 'Color', 'text-domain' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .news-one__sub-title' => 'color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => 'layout-1',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'blog_category_typography',
                'selector' => '{{WRAPPER}} .news-one__sub-title',
                'condition' => [
                    'design_style' => 'layout-1',
                ],
            ]
        );

        $this->add_control(
            '_heading_style_meta',
            [
                'label' => esc_html__( 'Meta', 'text-domain' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3' ],
                ],
            ]
        );

        $this->add_control(
            'meta_color',
            [
                'label' => esc_html__( 'Color', 'text-domain' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .news-one__person-text p' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .news-three__meta li:last-child p' => 'color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3' ],
                ],
            ]
        );

        $this->add_control(
            'meta_color_two',
            [
                'label' => esc_html__( 'Color Two', 'text-domain' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .news-one__date p' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .news-three__meta li p' => 'color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3' ],
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'meta_typography',
                'selector' => '{{WRAPPER}} .news-one__person-text p, .news-one__date p, .news-three__meta li p',
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3' ],
                ],
            ]
        );

        $this->add_control(
            '_heading_style_link',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Link', 'bixola-core' ),
                'separator' => 'before'
            ]
        );

        $this->add_responsive_control(
            'link_spacing',
            [
                'label' => __( 'Top Spacing', 'bixola-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .thm-btn-two' => 'margin-top: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .news-three__btn' => 'margin-top: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->start_controls_tabs( '_link_style_tabs' );
        
        $this->start_controls_tab(
            'link_normal_tab',
            [
                'label' => esc_html__( 'Normal', 'bixola-core' ),
            ]
        );
        
        $this->add_control(
            'link_color',
            [
                'label' => __( 'Color', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .thm-btn-two' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .news-three__btn a' => 'color: {{VALUE}}',
                ],
            ]
        );
        
        $this->add_control(
            'link_background',
            [
                'label' => __( 'Background', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .thm-btn-two' => 'background-color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => [ 'layout-2' ],
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'label'    => esc_html__( 'Border', 'bixola-core' ),
                'name'     => 'link_border',
                'selector' => '{{WRAPPER}} .thm-btn-two',
                'condition' => [
                    'design_style' => [ 'layout-2' ],
                ],
            ]
        );
        
        $this->end_controls_tab();
        
        $this->start_controls_tab(
            'link_hover_tab',
            [
                'label' => esc_html__( 'Hover', 'bixola-core' ),
            ]
        );
        
        $this->add_control(
            'link_color_hover',
            [
                'label' => __( 'Color', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .news-two__btn:hover' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .news-three__btn a:hover' => 'color: {{VALUE}}',
                ],
            ]
        );
        
        $this->add_control(
            'link_background_hover',
            [
                'label' => __( 'Background', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .news-two__btn::before' => 'background-color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => [ 'layout-2' ],
                ],
            ]
        );
        
        $this->end_controls_tab();
        
        $this->end_controls_tabs();

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'link_typography',
                'selector' => '{{WRAPPER}} .news-two__btn, .news-three__btn a',
            ]
        );

        $this->add_responsive_control(
            'link_padding',
            [
                'label' => __( 'Padding', 'bixola-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .news-two__btn' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'condition' => [
                    'design_style' => [ 'layout-2' ],
                ],
            ]
        );

        $this->add_responsive_control(
            'link_border_radius',
            [
                'label' => __( 'Border Radius', 'bixola-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .news-two__btn' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .news-two__btn::before' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'condition' => [
                    'design_style' => [ 'layout-2' ],
                ],
            ]
        );

        $this->add_control(
            '_blog_layout_style',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Layout', 'bixola-core' ),
                'separator' => 'before'
            ]
        );

        $this->add_control(
            'blog_layout_border_color',
            [
                'label' => esc_html__( 'Border Color', 'text-domain' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .news-one__content-top' => 'border-color: {{VALUE}}',
                    '{{WRAPPER}} .news-three__content' => 'border-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'blog_layout_background_color',
            [
                'label' => __( 'Background', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .news-one__content-top' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .news-one__person-and-date' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .news-three__content' => 'background-color: {{VALUE}}',
                ],
            ]
        );

		$this->end_controls_section();
        
	}

	/**
	 * Render the widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		if (get_query_var('paged')) {
            $paged = get_query_var('paged');
        } else if (get_query_var('page')) {
            $paged = get_query_var('page');
        } else {
            $paged = 1;
        }

        // include_categories
        $category_list = '';
        if (!empty($settings['category'])) {
            $category_list = implode(", ", $settings['category']);
        }
        $category_list_value = explode(" ", $category_list);

        // exclude_categories
        $exclude_categories = '';
        if(!empty($settings['exclude_category'])){
            $exclude_categories = implode(", ", $settings['exclude_category']);
        }
        $exclude_category_list_value = explode(" ", $exclude_categories);

        $post__not_in = '';
        if (!empty($settings['post__not_in'])) {
            $post__not_in = $settings['post__not_in'];
            $args['post__not_in'] = $post__not_in;
        }
        $posts_per_page = (!empty($settings['posts_per_page'])) ? $settings['posts_per_page'] : '-1';
        $orderby = (!empty($settings['orderby'])) ? $settings['orderby'] : 'post_date';
        $order = (!empty($settings['order'])) ? $settings['order'] : 'desc';
        $offset_value = (!empty($settings['offset'])) ? $settings['offset'] : '0';
        $ignore_sticky_posts = (! empty( $settings['ignore_sticky_posts'] ) && 'yes' == $settings['ignore_sticky_posts']) ? true : false ;


        // number
        $off = (!empty($offset_value)) ? $offset_value : 0;
        $offset = $off + (($paged - 1) * $posts_per_page);
        $p_ids = array();

        // build up the array
        if (!empty($settings['post__not_in'])) {
            foreach ($settings['post__not_in'] as $p_idsn) {
                $p_ids[] = $p_idsn;
            }
        }

        $args = array(
            'post_type' => 'post',
            'post_status' => 'publish',
            'posts_per_page' => $posts_per_page,
            'orderby' => $orderby,
            'order' => $order,
            'offset' => $offset,
            'paged' => $paged,
            'post__not_in' => $p_ids,
            'ignore_sticky_posts' => $ignore_sticky_posts
        );

        // exclude_categories
        if ( !empty($settings['exclude_category'])) {

            // Exclude the correct cats from tax_query
            $args['tax_query'] = array(
                array(
                    'taxonomy'	=> 'category',
                    'field'	 	=> 'slug',
                    'terms'		=> $exclude_category_list_value,
                    'operator'	=> 'NOT IN'
                )
            );

            // Include the correct cats in tax_query
            if ( !empty($settings['category'])) {
                $args['tax_query']['relation'] = 'AND';
                $args['tax_query'][] = array(
                    'taxonomy'	=> 'category',
                    'field'		=> 'slug',
                    'terms'		=> $category_list_value,
                    'operator'	=> 'IN'
                );
            }

        } else {
            // Include the cats from $cat_slugs in tax_query
            if (!empty($settings['category'])) {
                $args['tax_query'][] = [
                    'taxonomy' => 'category',
                    'field' => 'slug',
                    'terms' => $category_list_value,
                ];
            }
        }

        $filter_list = $settings['category'];

        // The Query
        $query = new \WP_Query($args);
        
        $this->add_render_attribute('section_title_args', 'class', 'section-title__title');

        if ( !empty($settings['blog_bg_img_1']['url']) ) {
            $blog_bg_img_1 = !empty($settings['blog_bg_img_1']['id']) ? wp_get_attachment_image_url( $settings['blog_bg_img_1']['id'], 'full' ) : $settings['blog_bg_img_1']['url'];
        }

        ?>

            <?php if ( $settings['design_style']  == 'layout-1' ): ?>

                <section class="bixola-el-section news-one">
                    <div class="news-one__shape-1 img-bounce">
                        <img src="<?php print get_template_directory_uri(); ?>/assets/images/shapes/news-one-shape-1.png" alt="<?php print esc_html( 'shape', 'bixola-core' ); ?>">
                    </div>
                    <div class="news-one__shape-2 float-bob-y">
                        <img src="<?php print get_template_directory_uri(); ?>/assets/images/shapes/news-one-shape-2.png" alt="<?php print esc_html( 'shape', 'bixola-core' ); ?>">
                    </div>
                    <div class="news-one__shape-3 rotate-me">
                        <img src="<?php print get_template_directory_uri(); ?>/assets/images/shapes/news-one-shape-3.png" alt="<?php print esc_html( 'shape', 'bixola-core' ); ?>">
                    </div>
                    <div class="container">
                        <div class="section-title text-center">
                            <?php if ( !empty ($settings['section_subheading']) ) : ?>
                                <div class="section-title__tagline-box">
                                    <p class="bixola-el-section-subheading section-title__tagline">
                                        <?php print twinkle_kses($settings['section_subheading']); ?>
                                    </p>
                                </div>
                            <?php endif; ?>
                            <?php if ( !empty ($settings['section_title']) ) : ?>
                                <h2 class="bixola-el-section-title section-title__title">
                                    <?php print twinkle_kses($settings['section_title']); ?>
                                </h2>
                            <?php endif; ?>
                        </div>
                        <div class="row">
                            <?php if ($query->have_posts()) : ?>
                                <?php while ($query->have_posts()) : 
                                    $query->the_post();
                                    global $post;
                                    $categories = get_the_category($post->ID);
                                ?>
                                <div class="col-xl-4 col-lg-4 wow fadeInLeft" data-wow-delay="100ms">
                                    <div class="news-one__single">
                                        <div class="news-one__img-box">
                                            <div class="news-one__img">
                                                <?php the_post_thumbnail('full');?>
                                            </div>
                                        </div>
                                        <div class="news-one__content">
                                            <div class="news-one__content-top">
                                                <?php  if ( ! empty( $categories ) ) {
                                                    echo '<p class="news-one__sub-title">' . esc_html( $categories[0]->name ) . '</p>';
                                                    }
                                                ?>
                                                <h3 class="news-one__title">
                                                    <a href="<?php the_permalink(); ?>">
                                                        <?php echo wp_trim_words(get_the_title(), $settings['blog_title_word'], ''); ?>
                                                    </a>
                                                </h3>
                                                <?php if (!empty($settings['post_content'])):
                                                    $post_content_limit = (!empty($settings['post_content_limit'])) ? $settings['post_content_limit'] : '';
                                                    ?>
                                                    <p class="news-one__text">
                                                        <?php print wp_trim_words(get_the_excerpt(get_the_ID()), $post_content_limit, ''); ?>
                                                    </p>
                                                <?php endif; ?>
                                            </div>
                                            <div class="news-one__person-and-date">
                                                <div class="news-one__person">
                                                    <div class="news-one__person-img">
                                                        <img src="<?php echo esc_url( get_avatar_url( get_the_author_meta( 'ID' ) ) ); ?>" alt="<?php echo esc_attr( get_the_author() ); ?>">
                                                    </div>
                                                    <div class="news-one__person-text">
                                                        <p><?php print esc_html( 'by', 'bixola-core' ); ?> <?php print get_the_author(); ?></p>
                                                    </div>
                                                </div>
                                                <div class="news-one__date">
                                                    <p>
                                                        <span class="icon-icon-calendar"></span>
                                                        <?php the_time( get_option('date_format') ); ?>
                                                    </p>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                <?php endwhile; wp_reset_query(); ?>
                            <?php endif; ?>
                        </div>
                    </div>
                </section>

            <?php elseif ( $settings['design_style']  == 'layout-2' ): ?>

                <section class="bixola-el-section news-two">
                    <div class="container">
                        <div class="section-title text-center">
                            <?php if ( !empty ($settings['section_subheading']) ) : ?>
                                <div class="section-title__tagline-box">
                                    <p class="bixola-el-section-subheading section-title__tagline">
                                        <?php print twinkle_kses($settings['section_subheading']); ?>
                                    </p>
                                </div>
                            <?php endif; ?>
                            <?php if ( !empty ($settings['section_title']) ) : ?>
                                <h2 class="bixola-el-section-title section-title__title">
                                    <?php print twinkle_kses($settings['section_title']); ?>
                                </h2>
                            <?php endif; ?>
                        </div>
                        <div class="row">
                            <?php if ($query->have_posts()) : ?>
                                <?php while ($query->have_posts()) : 
                                    $query->the_post();
                                    global $post;
                                    $categories = get_the_category($post->ID);
                                ?>
                            <div class="col-xl-4 col-lg-4 wow fadeInLeft" data-wow-delay="100ms">
                                <div class="news-two__single">
                                    <div class="news-two__img-box">
                                        <div class="news-two__img">
                                            <?php if (!empty( the_post_thumbnail() )): ?>
                                                <?php the_post_thumbnail( $post->ID );?>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                    <div class="news-two__content">
                                        <h3 class="news-two__title">
                                            <a href="<?php the_permalink(); ?>">
                                                <?php echo wp_trim_words(get_the_title(), $settings['blog_title_word'], ''); ?>
                                            </a>
                                        </h3>
                                        <?php if (!empty($settings['post_content'])):
                                            $post_content_limit = (!empty($settings['post_content_limit'])) ? $settings['post_content_limit'] : '';
                                            ?>
                                            <p class="news-two__text">
                                                <?php print wp_trim_words(get_the_excerpt(get_the_ID()), $post_content_limit, ''); ?>
                                            </p>
                                        <?php endif; ?>
                                        <?php if (!empty($settings['btn_button_show'])): ?>
                                            <div class="news-two__btn-box">
                                                <a href="<?php the_permalink(); ?>" class="news-two__btn thm-btn-two">
                                                    <?php print esc_html($settings['btn_text']);?>
                                                </a>
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </div>
                            <?php endwhile; wp_reset_query(); ?>
                        <?php endif; ?>
                        </div>
                    </div>
                </section>

            <?php elseif ( $settings['design_style']  == 'layout-3' ): ?>

                <section class="bixola-el-section news-three">
                    <div class="news-three__shape-1 img-bounce">
                        <img src="<?php print get_template_directory_uri(); ?>/assets/images/shapes/news-three-shape-1.png" alt="<?php print esc_attr( 'shape', 'bixola-core' ); ?>">
                    </div>
                    <div class="news-three__shape-2 float-bob-y">
                        <img src="<?php print get_template_directory_uri(); ?>/assets/images/shapes/news-three-shape-2.png" alt="<?php print esc_attr( 'shape', 'bixola-core' ); ?>">
                    </div>
                    <div class="container">
                        <div class="section-title-three text-center">
                            <?php if ( !empty ( $settings['section_subheading'] ) ) : ?>
                                <div class="section-title-three__tagline-box">
                                    <p class="bixola-el-section-subheading section-title-three__tagline">
                                        <?php print twinkle_kses($settings['section_subheading']); ?>
                                    </p>
                                </div>
                            <?php endif; ?>
                            <?php if ( !empty ( $settings['section_title'] ) ) : ?>
                                <h2 class="bixola-el-section-title section-title-three__title">
                                    <?php print twinkle_kses($settings['section_title']); ?>
                                </h2>
                            <?php endif; ?>
                        </div>
                        <div class="row">
                            <?php if ($query->have_posts()) : ?>
                                <?php while ($query->have_posts()) : 
                                    $query->the_post();
                                    global $post;
                                    $categories = get_the_category($post->ID);
                                ?>
                                <div class="col-xl-4 col-lg-4">
                                    <div class="news-three__single">
                                        <div class="news-three__img-box">
                                            <div class="news-three__img">
                                                <?php the_post_thumbnail( $post->ID );?>
                                            </div>
                                            <div class="news-three__date">
                                                <p><?php echo get_the_date('d'); ?></p>
                                                <span><?php echo get_the_date('M y'); ?></span>
                                            </div>
                                        </div>
                                        <div class="news-three__content">
                                            <ul class="news-three__meta list-unstyled">
                                                <li>
                                                    <p><span class="icon-user"></span><?php print get_the_author(); ?></p>
                                                </li>
                                                <li>
                                                    <p><span class="icon-chat"></span><?php comments_number(); ?></p>
                                                </li>
                                            </ul>
                                            <h3 class="news-three__title">
                                                <a href="<?php the_permalink(); ?>">
                                                    <?php echo wp_trim_words(get_the_title(), $settings['blog_title_word'], ''); ?>
                                                </a>
                                            </h3>
                                            <?php if (!empty($settings['post_content'])):
                                                $post_content_limit = (!empty($settings['post_content_limit'])) ? $settings['post_content_limit'] : '';
                                                ?>
                                                <p class="news-three__text">
                                                    <?php print wp_trim_words(get_the_excerpt(get_the_ID()), $post_content_limit, ''); ?>
                                                </p>
                                            <?php endif; ?>
                                            <?php if (!empty($settings['btn_button_show'])): ?>
                                                <div class="news-three__btn">
                                                    <a href="<?php the_permalink(); ?>">
                                                        <?php print esc_html($settings['btn_text']);?><span class="icon-right-arrow1"></span>
                                                    </a>
                                                </div>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </div>
                                <?php endwhile; wp_reset_query(); ?>
                            <?php endif; ?>
                        </div>
                    </div>
                </section>

            <?php endif; ?>
            
       <?php
	}

}

$widgets_manager->register( new Bixola_Blog_Post() );