<?php
namespace BixolaCore\Widgets;

use \Elementor\Widget_Base;
use \Elementor\Controls_Manager;
use \Elementor\Group_Control_Background;
use \Elementor\Group_Control_Image_Size;
use \Elementor\Group_Control_Typography;
use \Elementor\Group_Control_Border;
use \Elementor\Utils;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Bixola Core
 * @since 1.0.0
 */
class Bixola_Testimonial extends Widget_Base {

	/**
	 * Retrieve the widget name.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'bixola_testimonial';
	}

	/**
	 * Retrieve the widget title.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Testimonial', 'bixola-core' );
	}

	/**
	 * Retrieve the widget icon.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'bixola-icon';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * Note that currently Elementor supports only one category.
	 * When multiple categories passed, Elementor uses the first one.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'bixola_core' ];
	}

	/**
	 * Retrieve the list of scripts the widget depended on.
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'bixola-core' ];
	}

	/**
	 * Register the widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function register_controls() {

        $this->start_controls_section(
            'testimonial_content',
            [
                'label' => esc_html__( 'Testimonial', 'bixola-core' ),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'design_style',
            [
                'label' => esc_html__( 'Style', 'bixola-core' ),
                'type' => Controls_Manager::HIDDEN,
                'default' => 'layout-1',
                'options' => [
                    'layout-1' => esc_html__( 'Layout 1', 'bixola-core' ),
                    'layout-2'  => esc_html__( 'Layout 2', 'bixola-core' ),
                ],
            ]
        );

        $this->add_control(
            'testimonial_image',
            [
                'label' => esc_html__( 'Testimonial Image', 'bixola-core' ),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );

        $this->add_control(
            'testimonial_name', [
                'label' => esc_html__( 'Name', 'bixola-core' ),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__( 'Rashal Khan' , 'bixola-core' ),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'testimonial_designation', [
                'label' => esc_html__( 'Position', 'bixola-core' ),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__( 'Founder' , 'bixola-core' ),
                'label_block' => true,
            ]
        );
        $this->add_control(
            'testimonial_description',
            [
                'label' => esc_html__( 'Content', 'bixola-core' ),
                'type' => Controls_Manager::TEXTAREA,
                'rows' => 10,
                'default' => esc_html__( 'Hiring managers are busy people, so you need to make yourout the crowd as quickly as possible. In the first section. This should headline achievements', 'bixola-core' ),
            ]
        );

        $this->add_control(
            'testimonial_rating',
            [
                'label' => esc_html__( 'Rating', 'bixola-core' ),
                'type' => Controls_Manager::NUMBER,
                'min' => 1,
                'max' => 5,
                'step' => .5,
                'default' => 5,
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
			'_style_testimonial',
			[
				'label' => esc_html__( 'Testimonial', 'bixola-core' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

        $this->add_control(
            '_heading_style_testimonial_name',
            [
                'type' => Controls_Manager::HEADING,
                'label' => esc_html__( 'Name', 'bixola-core' ),
            ]
        );

        $this->add_responsive_control(
            'testimonial_name_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'bixola-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .testimonial-two__client-info h3' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'testimonial_name_color',
            [
                'label' => esc_html__( 'Color', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .testimonial-two__client-info h3' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'testimonial_name_typography',
                'selector' => '{{WRAPPER}} .testimonial-two__client-info h3',
            ]
        );

        $this->add_control(
            '_heading_testimonial_designation',
            [
                'type' => Controls_Manager::HEADING,
                'label' => esc_html__( 'Designation', 'bixola-core' ),
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'testimonial_designation_color',
            [
                'label' => esc_html__( 'Color', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .testimonial-two__client-info p' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'testimonial_designation_typography',
                'selector' => '.testimonial-two__client-info p',
            ]
        );

        $this->add_control(
            '_content_testimonial_description',
            [
                'type' => Controls_Manager::HEADING,
                'label' => esc_html__( 'Description', 'bixola-core' ),
                'separator' => 'before'
            ]
        );

        $this->add_responsive_control(
            'testimonial_description_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'bixola-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .testimonial-two__text' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'testimonial_description_color',
            [
                'label' => esc_html__( 'Color', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .testimonial-two__text' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'testimonial_description_typography',
                'selector' => '{{WRAPPER}} .testimonial-two__text',
            ]
        );

        $this->add_control(
            '_heading_style_testimonial_rating',
            [
                'label' => esc_html__( 'Rating', 'text-domain' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'testimonial_rating_color',
            [
                'label' => esc_html__( 'Color', 'text-domain' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .testimonial-two__ratting span' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_responsive_control(
            'testimonial_rating_icon_size',
            [
                'label' => esc_html__( 'Icon Size', 'text-domain' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .testimonial-two__ratting span' => 'font-size: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            '_heading_style_testimonial',
            [
                'label' => esc_html__( 'Layout', 'text-domain' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'testimonial_background',
            [
                'label' => esc_html__( 'Background', 'text-domain' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .testimonial-two__single' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_responsive_control(
            'testimonial_padding',
            [
                'label' => esc_html__( 'Padding', 'text-domain' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .testimonial-two__single' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'testimonial_border_radius',
            [
                'label' => esc_html__( 'Border Radius', 'text-domain' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'selectors' => [
                    '{{WRAPPER}} .testimonial-two__single' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

		$this->end_controls_section();

	}

	/**
	 * Render the widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

        if ( !empty($settings['testimonial_image']['url']) ) {
            $testimonial_image = !empty($settings['testimonial_image']['id']) ? wp_get_attachment_image_url( $settings['testimonial_image']['id']) : $settings['testimonial_image']['url'];
            $testimonial_image_alt = get_post_meta($settings["testimonial_image"]["id"], "_wp_attachment_image_alt", true);
        }

		?>

        <?php if ( $settings['design_style']  == 'layout-1' ): ?>

            <div class="testimonial-two__single">
                <?php if ( !empty($testimonial_image) ) : ?>
                    <div class="testimonial-two__client-img">
                        <img src="<?php echo esc_url($testimonial_image); ?>" alt="<?php echo esc_url($testimonial_image_alt); ?>">
                    </div>
                <?php endif; ?>
                <div class="testimonial-two__client-info">
                    <?php if ( !empty($settings['testimonial_name']) ) : ?>
                        <h3><?php echo twinkle_kses($settings['testimonial_name']); ?></h3>
                    <?php endif; ?>
                    <?php if ( !empty($settings['testimonial_designation']) ) : ?>
                        <p><?php echo twinkle_kses($settings['testimonial_designation']); ?></p>
                    <?php endif; ?>
                </div>
                <?php if ( !empty($settings['testimonial_description']) ) : ?>
                    <p class="testimonial-two__text">
                        <?php echo twinkle_kses($settings['testimonial_description']); ?>
                    </p>
                <?php endif; ?>
                <div class="testimonial-two__ratting">
                    <?php 
                    $testimonial_rating = $settings['testimonial_rating'];

                    // Calculate full stars, half stars, and empty stars
                    $full_stars = floor($testimonial_rating);
                    $half_star = ($testimonial_rating - $full_stars >= 0.5) ? 1 : 0;
                    $empty_stars = 5 - ($full_stars + $half_star);
                    for ($i = 1; $i <= $full_stars; $i++): ?>
                        <li><span class="fas fa-star"></span></li>
                    <?php endfor; ?>

                    <?php if ($half_star): ?>
                        <li><span class="fas fa-star-half-alt"></span></li>
                    <?php endif; ?>

                    <?php for ($i = 1; $i <= $empty_stars; $i++): ?>
                        <li><span class="far fa-star"></span></li>
                    <?php endfor; ?>
                </div>
            </div>

        <?php elseif ( $settings['design_style']  == 'layout-2' ) : ?>  

                
        
        <?php endif; ?>

        <?php 
	}
}

$widgets_manager->register( new Bixola_Testimonial() );