<?php

namespace BixolaCore\Widgets;

use \Elementor\Widget_Base;
use \Elementor\Controls_Manager;
use \Elementor\Group_Control_Background;
use \Elementor\Group_Control_Image_Size;
use \Elementor\Group_Control_Border;
use \Elementor\Group_Control_Typography;
use \Elementor\Repeater;
use \Elementor\Utils;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Bixola Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class Bixola_Services extends Widget_Base {

    /**
     * Retrieve the widget name.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget name.
     */
    public function get_name() {
        return 'bixola_services';
    }

    /**
     * Retrieve the widget title.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget title.
     */
    public function get_title() {
        return __( 'Services', 'bixola-core' );
    }

    /**
     * Retrieve the widget icon.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget icon.
     */
    public function get_icon() {
        return 'bixola-icon';
    }

    /**
     * Retrieve the list of categories the widget belongs to.
     *
     * Used to determine where to display the widget in the editor.
     *
     * Note that currently Elementor supports only one category.
     * When multiple categories passed, Elementor uses the first one.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget categories.
     */
    public function get_categories() {
        return [ 'bixola_core' ];
    }

    /**
     * Retrieve the list of scripts the widget depended on.
     *
     * Used to set scripts dependencies required to run the widget.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget scripts dependencies.
     */
    public function get_script_depends() {
        return [ 'bixola-services' ];
    }

    /**
     * Register the widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     *
     * @access protected
     */
    protected function register_controls() {

        $this->start_controls_section(
            '_content_design_layout',
            [
                'label' => esc_html__('Design Layout', 'bixola-core'),
            ]
        );
        $this->add_control(
            'design_style',
            [
                'label' => esc_html__('Select Layout', 'bixola-core'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'layout-1' => esc_html__('Layout 1', 'bixola-core'),
                    'layout-2' => esc_html__('Layout 2', 'bixola-core'),
                    'layout-3' => esc_html__('Layout 3', 'bixola-core'),
                    'layout-4' => esc_html__('Layout 4', 'bixola-core'),
                    'layout-5' => esc_html__('Layout 5', 'bixola-core'),
                    'layout-6' => esc_html__('Layout 6', 'bixola-core'),
                ],
                'default' => 'layout-1',
            ]
        );

        $this->add_control(
            'section_shape_switch',
            [
                'label' => esc_html__( 'Shape Switch', 'text-domain' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Show', 'text-domain' ),
                'label_off' => esc_html__( 'Hide', 'text-domain' ),
                'return_value' => 'yes',
                'default' => 'yes',
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-2', 'layout-3' ],
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_content_title',
            [
                'label' => esc_html__( 'Title & Content',  'text-domain'  ),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );
        
        $this->add_control(
            'section_subheading',
            [
                'label' => esc_html__( 'Subheading', 'text-domain' ),
                'description' => bixola_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'FEATURED SERVICES', 'text-domain' ),
                'label_block' => true,
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3', 'layout-6' ],
                ],
            ]
        );

        $this->add_control(
            'section_title',
            [
                'label' => esc_html__( 'Title', 'text-domain' ),
                'description' => bixola_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'What services we provide to our <br> customers business', 'text-domain' ),
                'label_block' => true,
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
            '_content_service_list_one',
            [
                'label' => esc_html__('Service List', 'bixola-core'),
                'tab' => Controls_Manager::TAB_CONTENT,
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-6' ],
                ],
            ]
        );

        $repeater = new Repeater();

        $repeater->add_control(
            'service_shape_switch',
            [
                'label' => esc_html__( 'Shape Switch', 'text-domain' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Show', 'text-domain' ),
                'label_off' => esc_html__( 'Hide', 'text-domain' ),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $repeater->add_control(
			'service_image_icon_switcher',
			[
				'label' => esc_html__( 'Image Type', 'bixola-core' ),
				'type' => Controls_Manager::SELECT,
				'default' => 'icon',
				'options' => [
					'image' => esc_html__( 'Image', 'bixola-core' ),
					'icon' => esc_html__( 'Icon', 'bixola-core' ),
				],
			]
		);

        $repeater->add_control(
            'service_list_image',
            [
                'label' => esc_html__('Upload Image', 'bixola-core'),
                'type' => Controls_Manager::MEDIA,
                'condition' => [
                    'service_image_icon_switcher' => 'image',
                ],
            ]
        );

        $repeater->add_control(
            'service_list_icon',
            [
                'label' => esc_html__('Icon', 'bixola-core'),
                'type' => Controls_Manager::ICONS,
                'fa4compatibility' => 'icon',
                'label_block' => true,
                'condition' => [
                    'service_image_icon_switcher' => 'icon',
                ],
            ]
        );

        $repeater->add_control(
            'service_image',
            [
                'label' => esc_html__( 'Service Image', 'text-domain' ),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
            ]
        );

		$repeater->add_control(
			'service_list_title',
			[
				'label' => esc_html__( 'Title', 'bixola-core' ),
				'type' => Controls_Manager::TEXT,
				'default' => esc_html__( 'Strategy Sync Pro' , 'bixola-core' ),
				'label_block' => true,
			]
		);

        $repeater->add_control(
            'service_list_description',
            [
                'label' => esc_html__('Description', 'bixola-core'),
                'type' => Controls_Manager::TEXTAREA,
                'label_block' => true,
            ]
        );

        $repeater->add_control(
			'_heading_style',
			[
				'label' => esc_html__( 'Button', 'bixola-core' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

        $repeater->add_control(
            'service_button_text',
            [
                'label' => esc_html__('Button Text', 'bixola-core'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Learn More', 'bixola-core'),
                'title' => esc_html__('Enter button text', 'bixola-core'),
            ]
        );

        $repeater->add_control(
            'service_link_type',
            [
                'label' => esc_html__( 'Service Link Type', 'bixola-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '1' => 'Custom Link',
                    '2' => 'Internal Page',
                ],
                'default' => '1',
            ]
        );
        $repeater->add_control(
            'service_link',
            [
                'label' => esc_html__( 'Service Link link', 'bixola-core' ),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => esc_html__( 'https://your-link.com', 'bixola-core' ),
                'show_external' => true,
                'default' => [
                    'url' => '#',
                    'is_external' => false,
                    'nofollow' => false,
                ],
                'condition' => [
                    'service_link_type' => '1',
                ]
            ]
        );
        $repeater->add_control(
            'service_page_link',
            [
                'label' => esc_html__( 'Select Service Link Page', 'bixola-core' ),
                'type' => Controls_Manager::SELECT2,
                'label_block' => true,
                'options' => bixola_get_all_pages(),
                'condition' => [
                    'service_link_type' => '2',
                ]
            ]
        );

        $repeater->add_control(
            'animation_delay',
            [
                'label' => esc_html__( 'Animation Delay', 'bixola-core' ),
                'type' => Controls_Manager::SELECT,
                'default' => '100ms',
                'options' => [
                    '100ms' => esc_html__( '100ms', 'bixola-core' ),
                    '200ms' => esc_html__( '200ms', 'bixola-core' ),
                    '300ms' => esc_html__( '300ms', 'bixola-core' ),
                    '400ms' => esc_html__( '400ms', 'bixola-core' ),
                    '500ms' => esc_html__( '500ms', 'bixola-core' ),
                    '600ms' => esc_html__( '600ms', 'bixola-core' ),
                    '700ms' => esc_html__( '700ms', 'bixola-core' ),
                    '800ms' => esc_html__( '800ms', 'bixola-core' ),
                    '900ms' => esc_html__( '900ms', 'bixola-core' ),
                ],
            ]
        );

        $this->add_control(
			'service_list_one',
			[
				'label' => esc_html__( 'Repeater List', 'bixola-core' ),
				'type' => Controls_Manager::REPEATER,
				'fields' => $repeater->get_controls(),
				'default' => [
					[   
                        'service_list_image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                        'service_image_icon_switcher' => 'icon',
						'service_list_icon' => 'icon-icon-business-audit',
						'service_list_title' => __( 'Business Audit', 'bixola-core' ),
						'service_list_description' => __( "Lorem Ipsum is simply dummy text of the printing industry. The industry's standard dummy", 'bixola-core' ),
                        'animation_delay' => '100ms',
                    ],
					[   
                        'service_list_image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                        'service_image_icon_switcher' => 'icon',
						'service_list_icon' => 'icon-icon-tax-strategy',
						'service_list_title' => __( 'Tax Strategy', 'bixola-core' ),
						'service_list_description' => __( "Lorem Ipsum is simply dummy text of the printing industry. The industry's standard dummy", 'bixola-core' ),
					    'animation_delay' => '200ms',
                    ],
					[   
                        'service_list_image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                        'service_image_icon_switcher' => 'icon',
						'service_list_icon' => 'icon-financial-advice',
						'service_list_title' => __( 'Financial Advices', 'bixola-core' ),
						'service_list_description' => __( "Lorem Ipsum is simply dummy text of the printing industry. The industry's standard dummy", 'bixola-core' ),
                        'animation_delay' => '300ms',
                    ],
					[   
                        'service_list_image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                        'service_image_icon_switcher' => 'icon',
						'service_list_icon' => 'icon-icon-insurance-strategy',
						'service_list_title' => __( 'Insurance Strategy', 'bixola-core' ),
						'service_list_description' => __( "Lorem Ipsum is simply dummy text of the printing industry. The industry's standard dummy", 'bixola-core' ),
                        'animation_delay' => '400ms',
                    ],
					[   
                        'service_list_image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                        'service_image_icon_switcher' => 'icon',
						'service_list_icon' => 'icon-icon-start-ups',
						'service_list_title' => __( 'Start Ups', 'bixola-core' ),
						'service_list_description' => __( "Lorem Ipsum is simply dummy text of the printing industry. The industry's standard dummy", 'bixola-core' ),
                        'animation_delay' => '500ms',
                    ],
					[   
                        'service_list_image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                        'service_image_icon_switcher' => 'icon',
						'service_list_icon' => 'icon-icon-manage-investment',
						'service_list_title' => __( 'Manage Investment', 'bixola-core' ),
						'service_list_description' => __( "Lorem Ipsum is simply dummy text of the printing industry. The industry's standard dummy", 'bixola-core' ),
                        'animation_delay' => '600ms',
                    ],
					
				],
				'title_field' => '{{{ service_list_title }}}',
			]
		);

        $this->end_controls_section();

        $this->start_controls_section(
            '_content_service_list_two',
            [
                'label' => esc_html__('Service List', 'bixola-core'),
                'tab' => Controls_Manager::TAB_CONTENT,
                'condition' => [
                    'design_style' => ['layout-2'],
                ],
            ]
        );

        $repeater = new Repeater();

        $repeater->add_control(
            'service_image',
            [
                'label' => esc_html__( 'Service Image', 'text-domain' ),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
            ]
        );

		$repeater->add_control(
			'service_list_title',
			[
				'label' => esc_html__( 'Title', 'bixola-core' ),
				'type' => Controls_Manager::TEXT,
				'default' => esc_html__( 'Strategy Sync Pro' , 'bixola-core' ),
				'label_block' => true,
			]
		);

        $repeater->add_control(
            'service_list_description',
            [
                'label' => esc_html__('Description', 'bixola-core'),
                'type' => Controls_Manager::TEXTAREA,
                'label_block' => true,
            ]
        );

        $repeater->add_control(
			'_heading_content_button',
			[
				'label' => esc_html__( 'Button', 'bixola-core' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

        $repeater->add_control(
            'service_button_text',
            [
                'label' => esc_html__('Button Text', 'bixola-core'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Learn More', 'bixola-core'),
                'title' => esc_html__('Enter button text', 'bixola-core'),
            ]
        );

        $repeater->add_control(
            'service_link_type',
            [
                'label' => esc_html__( 'Service Link Type', 'bixola-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '1' => 'Custom Link',
                    '2' => 'Internal Page',
                ],
                'default' => '1',
            ]
        );

        $repeater->add_control(
            'service_link',
            [
                'label' => esc_html__( 'Service Link link', 'bixola-core' ),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => esc_html__( 'https://your-link.com', 'bixola-core' ),
                'show_external' => true,
                'default' => [
                    'url' => '#',
                    'is_external' => false,
                    'nofollow' => false,
                ],
                'condition' => [
                    'service_link_type' => '1',
                ]
            ]
        );
        $repeater->add_control(
            'service_page_link',
            [
                'label' => esc_html__( 'Select Service Link Page', 'bixola-core' ),
                'type' => Controls_Manager::SELECT2,
                'label_block' => true,
                'options' => bixola_get_all_pages(),
                'condition' => [
                    'service_link_type' => '2',
                ]
            ]
        );

        $this->add_control(
			'service_list_two',
			[
				'label' => esc_html__( 'Repeater List', 'bixola-core' ),
				'type' => Controls_Manager::REPEATER,
				'fields' => $repeater->get_controls(),
				'default' => [
					[   
                        'service_list_image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
						'service_list_title' => __( 'Business Audit', 'bixola-core' ),
						'service_list_description' => __( "Lorem Ipsum is simply dummy text of the printing and typesetting industry. Lorem Ipsum has been the industry's standard dummy", 'bixola-core' ),
					],
					[   
                        'service_list_image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
						'service_list_title' => __( 'Tax Strategy', 'bixola-core' ),
						'service_list_description' => __( "Lorem Ipsum is simply dummy text of the printing and typesetting industry. Lorem Ipsum has been the industry's standard dummy", 'bixola-core' ),
					],
					[   
                        'service_list_image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
						'service_list_title' => __( 'Financial Advices', 'bixola-core' ),
						'service_list_description' => __( "Lorem Ipsum is simply dummy text of the printing and typesetting industry. Lorem Ipsum has been the industry's standard dummy", 'bixola-core' ),
					],
					[   
                        'service_list_image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
						'service_list_title' => __( 'Insurance Strategy', 'bixola-core' ),
						'service_list_description' => __( "Lorem Ipsum is simply dummy text of the printing and typesetting industry. Lorem Ipsum has been the industry's standard dummy", 'bixola-core' ),
					],
					[   
                        'service_list_image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
						'service_list_title' => __( 'Start Ups', 'bixola-core' ),
						'service_list_description' => __( "Lorem Ipsum is simply dummy text of the printing and typesetting industry. Lorem Ipsum has been the industry's standard dummy", 'bixola-core' ),
					],
					[   
                        'service_list_image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
						'service_list_title' => __( 'Manage Investment', 'bixola-core' ),
						'service_list_description' => __( "Lorem Ipsum is simply dummy text of the printing and typesetting industry. Lorem Ipsum has been the industry's standard dummy", 'bixola-core' ),
					],
					
				],
				'title_field' => '{{{ service_list_title }}}',
			]
		);

        $this->end_controls_section();

        $this->start_controls_section(
            '_content_service_list_three',
            [
                'label' => esc_html__('Service List', 'bixola-core'),
                'tab' => Controls_Manager::TAB_CONTENT,
                'condition' => [
                    'design_style' => [ 'layout-3', 'layout-4', 'layout-5' ],
                ],
            ]
        );

        $repeater = new Repeater();

        $repeater->add_control(
			'service_image_icon_switcher',
			[
				'label' => esc_html__( 'Image Type', 'bixola-core' ),
				'type' => Controls_Manager::SELECT,
				'default' => 'icon',
				'options' => [
					'image' => esc_html__( 'Image', 'bixola-core' ),
					'icon' => esc_html__( 'Icon', 'bixola-core' ),
				],
			]
		);

        $repeater->add_control(
            'service_list_image',
            [
                'label' => esc_html__('Upload Image', 'bixola-core'),
                'type' => Controls_Manager::MEDIA,
                'condition' => [
                    'service_image_icon_switcher' => 'image',
                ],
            ]
        );

        $repeater->add_control(
            'service_list_icon',
            [
                'label' => esc_html__('Icon', 'bixola-core'),
                'type' => Controls_Manager::ICONS,
                'fa4compatibility' => 'icon',
                'label_block' => true,
                'condition' => [
                    'service_image_icon_switcher' => 'icon',
                ],
            ]
        );

		$repeater->add_control(
			'service_list_title',
			[
				'label' => esc_html__( 'Title', 'bixola-core' ),
				'type' => Controls_Manager::TEXT,
				'default' => esc_html__( 'Strategy Sync Pro' , 'bixola-core' ),
				'label_block' => true,
			]
		);

        $repeater->add_control(
            'service_list_description',
            [
                'label' => esc_html__('Description', 'bixola-core'),
                'type' => Controls_Manager::TEXTAREA,
                'label_block' => true,
            ]
        );

        $repeater->add_control(
			'_heading_style_button',
			[
				'label' => esc_html__( 'Button', 'bixola-core' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

        $repeater->add_control(
            'service_button_text',
            [
                'label' => esc_html__('Button Text', 'bixola-core'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Learn More', 'bixola-core'),
                'title' => esc_html__('Enter button text', 'bixola-core'),
            ]
        );

        $repeater->add_control(
            'service_link_type',
            [
                'label' => esc_html__( 'Service Link Type', 'bixola-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '1' => 'Custom Link',
                    '2' => 'Internal Page',
                ],
                'default' => '1',
            ]
        );
        $repeater->add_control(
            'service_link',
            [
                'label' => esc_html__( 'Service Link link', 'bixola-core' ),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => esc_html__( 'https://your-link.com', 'bixola-core' ),
                'show_external' => true,
                'default' => [
                    'url' => '#',
                    'is_external' => false,
                    'nofollow' => false,
                ],
                'condition' => [
                    'service_link_type' => '1',
                ]
            ]
        );
        $repeater->add_control(
            'service_page_link',
            [
                'label' => esc_html__( 'Select Service Link Page', 'bixola-core' ),
                'type' => Controls_Manager::SELECT2,
                'label_block' => true,
                'options' => bixola_get_all_pages(),
                'condition' => [
                    'service_link_type' => '2',
                ]
            ]
        );

        $repeater->add_control(
            'animation_delay',
            [
                'label' => esc_html__( 'Animation Delay', 'bixola-core' ),
                'type' => Controls_Manager::SELECT,
                'default' => '100ms',
                'options' => [
                    '100ms' => esc_html__( '100ms', 'bixola-core' ),
                    '200ms' => esc_html__( '200ms', 'bixola-core' ),
                    '300ms' => esc_html__( '300ms', 'bixola-core' ),
                    '400ms' => esc_html__( '400ms', 'bixola-core' ),
                    '500ms' => esc_html__( '500ms', 'bixola-core' ),
                    '600ms' => esc_html__( '600ms', 'bixola-core' ),
                    '700ms' => esc_html__( '700ms', 'bixola-core' ),
                    '800ms' => esc_html__( '800ms', 'bixola-core' ),
                    '900ms' => esc_html__( '900ms', 'bixola-core' ),
                ],
            ]
        );

        $this->add_control(
			'service_list_three',
			[
				'label' => esc_html__( 'Repeater List', 'bixola-core' ),
				'type' => Controls_Manager::REPEATER,
				'fields' => $repeater->get_controls(),
				'default' => [
					[   
                        'service_list_image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                        'service_image_icon_switcher' => 'icon',
						'service_list_icon' => 'icon-icon-business-audit',
						'service_list_title' => __( 'Business Audit', 'bixola-core' ),
						'service_list_description' => __( "Lorem Ipsum is simply dummy text of the printing industry. The industry's standard dummy", 'bixola-core' ),
					    'animation_delay' => '100ms',
                    ],
					[   
                        'service_list_image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                        'service_image_icon_switcher' => 'icon',
						'service_list_icon' => 'icon-icon-tax-strategy',
						'service_list_title' => __( 'Tax Strategy', 'bixola-core' ),
						'service_list_description' => __( "Lorem Ipsum is simply dummy text of the printing industry. The industry's standard dummy", 'bixola-core' ),
					    'animation_delay' => '200ms',
                    ],
					[   
                        'service_list_image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                        'service_image_icon_switcher' => 'icon',
						'service_list_icon' => 'icon-financial-advice',
						'service_list_title' => __( 'Financial Advices', 'bixola-core' ),
						'service_list_description' => __( "Lorem Ipsum is simply dummy text of the printing industry. The industry's standard dummy", 'bixola-core' ),
                        'animation_delay' => '300ms',
                    ],
					[   
                        'service_list_image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                        'service_image_icon_switcher' => 'icon',
						'service_list_icon' => 'icon-icon-insurance-strategy',
						'service_list_title' => __( 'Insurance Strategy', 'bixola-core' ),
						'service_list_description' => __( "Lorem Ipsum is simply dummy text of the printing industry. The industry's standard dummy", 'bixola-core' ),
					    'animation_delay' => '400ms',
                    ],
					[   
                        'service_list_image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                        'service_image_icon_switcher' => 'icon',
						'service_list_icon' => 'icon-icon-start-ups',
						'service_list_title' => __( 'Start Ups', 'bixola-core' ),
						'service_list_description' => __( "Lorem Ipsum is simply dummy text of the printing industry. The industry's standard dummy", 'bixola-core' ),
					    'animation_delay' => '500ms',
                    ],
					[   
                        'service_list_image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                        'service_image_icon_switcher' => 'icon',
						'service_list_icon' => 'icon-icon-manage-investment',
						'service_list_title' => __( 'Manage Investment', 'bixola-core' ),
						'service_list_description' => __( "Lorem Ipsum is simply dummy text of the printing industry. The industry's standard dummy", 'bixola-core' ),
					    'animation_delay' => '600ms',
                    ],
					
				],
				'title_field' => '{{{ service_list_title }}}',
			]
		);

        $this->end_controls_section();

        $this->start_controls_section(
            '_content_button',
            [
                'label' => esc_html__( 'Button', 'text-domain' ),
                'tab' => Controls_Manager::TAB_CONTENT,
                'condition' => [
                    'design_style' => 'layout-3',
                ],
            ]
        );

        $this->add_control(
            '_heading_content_primary_button',
            [
                'label' => esc_html__( 'Primary', 'text-domain' ),
                'type' => Controls_Manager::HEADING,
            ]
        );
        
        $this->add_control(
            'section_primary_button_label',
            [
                'label' => esc_html__( 'Label', 'text-domain' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'You Can Also', 'text-domain' ),
                'label_block' => true,
            ]
        );
        
        $this->add_control(
            'section_primary_button_text',
            [
                'label' => esc_html__( 'Button Text', 'text-domain' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'See All Business', 'text-domain' ),
                'label_block' => true,
            ]
        );
        
        $this->add_control(
            'section_primary_button_link_type',
            [
                'label' => esc_html__( 'Button Link Type', 'text-domain' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '1' => 'Custom Link',
                    '2' => 'Internal Page',
                ],
                'default' => '1',
                'label_block' => true,
            ]
        );
        
        $this->add_control(
            'section_primary_button_link',
            [
                'label' => esc_html__( 'Button link', 'text-domain' ),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => esc_html__('https://your-link.com', 'text-domain'),
                'show_external' => false,
                'default' => [
                    'url' => '#',
                    'is_external' => true,
                    'nofollow' => true,
                    'custom_attributes' => '',
                ],
                'condition' => [
                    'section_primary_button_link_type' => '1',
                ],
                'label_block' => true,
            ]
        );

        $this->add_control(
            'section_primary_button_page_link',
            [
                'label' => esc_html__( 'Select Button Page', 'text-domain' ),
                'type' => Controls_Manager::SELECT2,
                'label_block' => true,
                'options' => bixola_get_all_pages(),
                'condition' => [
                    'section_primary_button_link_type' => '2',
                ]
            ]
        );

        $this->add_control(
            '_heading_content_secondary_button',
            [
                'label' => esc_html__( 'Secondary', 'text-domain' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );
        
        $this->add_control(
            'section_secondary_button_text',
            [
                'label' => esc_html__( 'Button Text', 'text-domain' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'Services', 'text-domain' ),
                'label_block' => true,
            ]
        );
        
        $this->add_control(
            'section_secondary_button_link_type',
            [
                'label' => esc_html__( 'Button Link Type', 'text-domain' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '1' => 'Custom Link',
                    '2' => 'Internal Page',
                ],
                'default' => '1',
                'label_block' => true,
            ]
        );
        
        $this->add_control(
            'section_secondary_button_link',
            [
                'label' => esc_html__( 'Button link', 'text-domain' ),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => esc_html__('https://your-link.com', 'text-domain'),
                'show_external' => false,
                'default' => [
                    'url' => '#',
                    'is_external' => true,
                    'nofollow' => true,
                    'custom_attributes' => '',
                ],
                'condition' => [
                    'section_secondary_button_link_type' => '1',
                ],
                'label_block' => true,
            ]
        );

        $this->add_control(
            'section_secondary_button_page_link',
            [
                'label' => esc_html__( 'Select Button Page', 'text-domain' ),
                'type' => Controls_Manager::SELECT2,
                'label_block' => true,
                'options' => bixola_get_all_pages(),
                'condition' => [
                    'section_secondary_button_link_type' => '2',
                ]
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
            '_content_column',
            [
                'label' => esc_html__( 'Column',  'bixola-core'  ),
                'tab' => Controls_Manager::TAB_CONTENT,
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3', 'layout-4', 'layout-6' ],
                ],
            ]
        );
        
        $this->add_control(
            'column_desktop',
            [
                'label' => esc_html__( 'Columns for Desktop', 'bixola-core' ),
                'description' => esc_html__( 'Screen width equal to or greater than 1200px', 'bixola-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    12 => esc_html__( '1 Columns', 'bixola-core' ),
                    6 => esc_html__( '2 Columns', 'bixola-core' ),
                    4 => esc_html__( '3 Columns', 'bixola-core' ),
                    3 => esc_html__( '4 Columns', 'bixola-core' ),
                    5 => esc_html__( '5 Columns (For Carousel Item)', 'bixola-core' ),
                    2 => esc_html__( '6 Columns', 'bixola-core' ),
                    1 => esc_html__( '12 Columns', 'bixola-core' ),
                ],
                'separator' => 'before',
                'default' => 4,
                'style_transfer' => true,
            ]
        );

        $this->add_control(
            'column_laptop',
            [
                'label' => esc_html__( 'Columns for Large', 'bixola-core' ),
                'description' => esc_html__( 'Screen width equal to or greater than 992px', 'bixola-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    12 => esc_html__( '1 Columns', 'bixola-core' ),
                    6 => esc_html__( '2 Columns', 'bixola-core' ),
                    4 => esc_html__( '3 Columns', 'bixola-core' ),
                    3 => esc_html__( '4 Columns', 'bixola-core' ),
                    5 => esc_html__( '5 Columns (For Carousel Item)', 'bixola-core' ),
                    2 => esc_html__( '6 Columns', 'bixola-core' ),
                    1 => esc_html__( '12 Columns', 'bixola-core' ),
                ],
                'separator' => 'before',
                'default' => 4,
                'style_transfer' => true,
            ]
        );

        $this->add_control(
            'column_tablet',
            [
                'label' => esc_html__( 'Columns for Tablet', 'bixola-core' ),
                'description' => esc_html__( 'Screen width equal to or greater than 768px', 'bixola-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    12 => esc_html__( '1 Columns', 'bixola-core' ),
                    6 => esc_html__( '2 Columns', 'bixola-core' ),
                    4 => esc_html__( '3 Columns', 'bixola-core' ),
                    3 => esc_html__( '4 Columns', 'bixola-core' ),
                    5 => esc_html__( '5 Columns (For Carousel Item)', 'bixola-core' ),
                    2 => esc_html__( '6 Columns', 'bixola-core' ),
                    1 => esc_html__( '12 Columns', 'bixola-core' ),
                ],
                'separator' => 'before',
                'default' => 6,
                'style_transfer' => true,
            ]
        );

        $this->add_control(
            'column_mobile',
            [
                'label' => esc_html__( 'Columns for Mobile', 'bixola-core' ),
                'description' => esc_html__( 'Screen width less than 767px', 'bixola-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    12 => esc_html__( '1 Columns', 'bixola-core' ),
                    6 => esc_html__( '2 Columns', 'bixola-core' ),
                    4 => esc_html__( '3 Columns', 'bixola-core' ),
                    3 => esc_html__( '4 Columns', 'bixola-core' ),
                    5 => esc_html__( '5 Columns (For Carousel Item)', 'bixola-core' ),
                    2 => esc_html__( '6 Columns', 'bixola-core' ),
                    1 => esc_html__( '12 Columns', 'bixola-core' ),
                ],
                'separator' => 'before',
                'default' => 12,
                'style_transfer' => true,
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
			'_style_design_style',
			[
				'label' => __( 'Design Layout', 'bixola-core' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

        $this->add_responsive_control(
            'design_style_margin',
            [
                'label' => __( 'Margin', 'bixola-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-section' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'design_style_padding',
            [
                'label' => __( 'Padding', 'bixola-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-section' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'design_layout_background',
            [
                'label' => esc_html__( 'Background', 'text-domain' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-section' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'bixola_services_list_style',
            [
                'label' => __( 'Services List', 'bixola-core' ),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            '_icon_title',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Icon', 'bixola-core' ),
                'condition' => [
                    'design_style' => [ 'layout-1' ],
                ],
            ]
        );

        $this->add_responsive_control(
            'icon_size',
            [
                'label' => __( 'Icon Size', 'bixola-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'default' => [
					'unit' => 'px',
					'size' => 60,
				],
                'selectors' => [
                    '{{WRAPPER}} .latest-service__item-icon svg' => 'width: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .our-services__item-icon svg' => 'width: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .service-item__icon svg' => 'width: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .service-item__icon img' => 'width: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1' ],
                ],
            ]
        );

        $this->start_controls_tabs( 'tabs_button_style', [
            'condition' => [
                'design_style' => [ 'layout-1' ],
            ],
        ]
        );

        $this->start_controls_tab(
            'icon_tab',
            [
                'label' => esc_html__( 'Normal', 'bixola-core' ),
            ],
        );

        $this->add_control(
            'icon_color',
            [
                'label' => __( 'Text', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .our-services__item-icon svg' => 'fill: {{VALUE}}',
                    '{{WRAPPER}} .service-item__icon svg path' => 'fill: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1' ],
                ],
            ]
        );

        $this->add_control(
            'icon_background_color',
            [
                'label' => __( 'Background', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .latest-service__item-icon' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .our-services__item-icon' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .service-item__icon' => 'background-color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1' ],
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'icon_hover_tab',
            [
                'label' => esc_html__( 'Hover', 'bixola-core' ),
                'condition' => [
                    'design_style' => [ 'layout-1' ],
                ],
            ]
        );

        $this->add_control(
            'icon_color_hover',
            [
                'label' => __( 'Text', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .service-one__single .service-one__icon:hover svg' => 'fill: {{VALUE}}',
                    '{{WRAPPER}} .service-two__single .icon-box:hover svg' => 'fill: {{VALUE}}',
                    '{{WRAPPER}} .services-three__single:hover .services-three__single-icon' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .service-three__single .service-three__single-img:hover svg' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .service-item:hover .service-item__icon svg path' => 'fill: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'icon_background_color_hover',
            [
                'label' => __( 'Background', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .services-one__single-img .icon::before' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .service-two__single-img .icon-box:before' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .service-item:hover .service-item__icon' => 'background-color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1' ],
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->add_control(
            '_heading_title',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Title', 'bixola-core' ),
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'title_spacing',
            [
                'label' => __( 'Bottom Spacing', 'bixola-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .latest-service__item-text a h4' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .our-services__item a h4' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .service-item .title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1' ],
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'title_typography',
                'selector' => '{{WRAPPER}} .latest-service__item-text a h4',
                'selector' => '{{WRAPPER}} .latest-service-2__item-text a h4',
                'selector' => '{{WRAPPER}} .our-services__item a h4',
                'selector' => '{{WRAPPER}} .service-item .title',
            ]
        );

        
        $this->start_controls_tabs( 'service_title_style');

        $this->start_controls_tab(
            'title_tab_normal',
            [
                'label' => esc_html__( 'Normal', 'bixola-core' ),
            ],
        );

        $this->add_control(
            'title_color_normal',
            [
                'label' => __( 'Text', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .latest-service__item-text a h4' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .latest-service-2__item-text a h4' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .our-services__item a h4' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .service-item .title' => 'color: {{VALUE}}',
                ],
            ]
        );

         $this->end_controls_tab();

        $this->start_controls_tab(
            'title_tab_hover',
            [
                'label' => esc_html__( 'Hover', 'bixola-core' ),
            ],
        );

        $this->add_control(
            'title_color_hover',
            [
                'label' => __( 'Text', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .latest-service__item-text a:hover h4' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .latest-service-2__item-text a:hover h4' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .our-services__item a:hover h4' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .service-item .title:hover' => 'color: {{VALUE}}',
                ],
            ]
        );

         $this->end_controls_tab();

        $this->end_controls_tabs();

        // Description
        $this->add_control(
            '_content_description',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Description', 'bixola-core' ),
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'description_spacing',
            [
                'label' => __( 'Bottom Spacing', 'bixola-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .latest-service-2__item-text h5' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .our-services__item p' => 'padding-bottom: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .description' => 'padding-bottom: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1' ],
                ],
            ]
        );

        $this->add_control(
            'services_list_description_color',
            [
                'label' => __( 'Text', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .latest-service__item-text p' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .latest-service-2__item-text h5' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .our-services__item p' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .description p' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'description_typography',
                'selector' => '{{WRAPPER}} .latest-service__item-text p',
                'selector' => '{{WRAPPER}} .latest-service-2__item-text h5',
                'selector' => '{{WRAPPER}} .our-services__item p',
                'selector' => '{{WRAPPER}} .description p',
            ]
        );

        // Link
        $this->add_control(
            '_content_link',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Link', 'bixola-core' ),
                'separator' => 'before',
                'condition' => [
                    'design_style' => [ 'layout-1' ],
                ],
            ]
        );

        $this->start_controls_tabs(
            'services_list_link_tabs', [
                'condition' => [
                    'design_style' => [ 'layout-1' ],
                ],
            ]
        );
        
        $this->start_controls_tab(
            'services_list_link_normal_tab',
            [
                'label' => esc_html__( 'Normal', 'bixola-core' ),
            ]
        );

        $this->add_control(
            'services_list_link_color',
            [
                'label' => __( 'Text', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .our-services__item a.readmore' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .our-services__item a.readmore svg path' => 'fill: {{VALUE}}',
                    '{{WRAPPER}} .service-item a.readmore' => 'color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1' ],
                ],
            ]
        );
        
        $this->end_controls_tab();

        $this->start_controls_tab(
            'services_list_link_hover_tab',
            [
                'label' => esc_html__( 'Hover', 'bixola-core' ),
            ]
        );

        $this->add_control(
            'services_list_link_color_hover',
            [
                'label' => __( 'Text', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .our-services__item a.readmore:hover' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .our-services__item a.readmore:hover svg path' => 'fill: {{VALUE}}',
                    '{{WRAPPER}} .service-item a.readmore:hover' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_tab();
        
        $this->end_controls_tabs();

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'link_typography',
                'selector' => '{{WRAPPER}} .our-services__item a.readmore',
                'condition' => [
                    'design_style' => [ 'layout-1' ],
                ],
            ],
        );

        $this->end_controls_section();
    }

    /**
     * Render the widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     *
     * @access protected
     */
    protected function render() {
        $settings = $this->get_settings_for_display();

        ?>

        <?php if ( $settings['design_style']  == 'layout-1' ) : ?>

            <section class="bixola-el-section services-one">
                <?php if ( !empty( $settings['section_shape_switch'] ) ) : ?>
                    <div class="services-one__shape-3 zoominout">
                        <img src="<?php print get_template_directory_uri(); ?>/assets/images/shapes/services-one-shape-3.png" alt="">
                    </div>
                    <div class="services-one__shape-4 shape-mover">
                        <img src="<?php print get_template_directory_uri(); ?>/assets/images/shapes/services-one-shape-4.png" alt="">
                    </div>
                <?php endif; ?>
                <div class="container">
                    <div class="section-title text-center">
                        <?php if ( !empty( $settings['section_subheading'] ) ) : ?>
                            <div class="section-title__tagline-box">
                                <p class="bixola-el-section-subheading section-title__tagline">
                                    <?php print twinkle_kses($settings['section_subheading']); ?>
                                </p>
                            </div>
                        <?php endif; ?>
                        <?php if ( !empty( $settings['section_title'] ) ) : ?>
                            <h2 class="bixola-el-section-title section-title__title">
                                <?php print twinkle_kses($settings['section_title']); ?>
                            </h2>
                        <?php endif; ?>
                    </div>
                    <div class="row">
                        <?php foreach($settings['service_list_one'] as $index => $item) :

                            if ( !empty($item['service_image']['url']) ) {
                                $service_image = !empty($item['service_image']['id']) ? wp_get_attachment_image_url( $item['service_image']['id'], 'full') : $item['service_image']['url'];
                                $service_image_alt = get_post_meta($item["service_image"]["id"], "_wp_attachment_image_alt", true);
                            }
                                
                            // Link
                            if ('2' == $item['service_link_type']) {
                                $link = get_permalink($item['service_page_link']);
                                $target = '_self';
                                $rel = 'nofollow';
                            } else {
                                $link = !empty($item['service_link']['url']) ? $item['service_link']['url'] : '';
                                $target = !empty($item['service_link']['is_external']) ? '_blank' : '';
                                $rel = !empty($item['service_link']['nofollow']) ? 'nofollow' : '';
                            }
                        ?>
                        <div class="col-xl-<?php print esc_attr($settings['column_desktop']); ?> col-lg-<?php print esc_attr($settings['column_laptop']); ?> col-md-<?php print esc_attr($settings['column_tablet']); ?> col-sm-<?php print esc_attr($settings['column_mobile']); ?> wow fadeInLeft" data-wow-delay="<?php print esc_attr($item['animation_delay']); ?>">
                            <div class="services-one__single">
                                <div class="services-one__single-hover-bg" style="background-image: url(<?php print esc_url($service_image); ?>);"></div>
                                <div class="services-one__icon">
                                    <?php if (!empty($item['service_list_image']['url'])): ?>
                                        <div class="icon">
                                            <img src="<?php echo $item['service_list_image']['url']; ?>" alt="<?php echo get_post_meta(attachment_url_to_postid($item['service_list_image']['url']), '_wp_attachment_image_alt', true); ?>">
                                        </div>
                                    <?php endif; ?>
                                    <?php if ( !empty($item['service_list_icon']['value']) ) : ?>
                                        <?php bixola_render_icon($item, 'service_list_icon'); ?>
                                    <?php endif; ?>
                                    <?php if ( !empty( $item['service_shape_switch'] ) ) : ?>
                                        <div class="services-one__shape-1">
                                            <img src="<?php print get_template_directory_uri(); ?>/assets/images/shapes/services-one-shape-1.png" alt="">
                                        </div>
                                        <div class="services-one__shape-2">
                                            <img src="<?php print get_template_directory_uri(); ?>/assets/images/shapes/services-one-shape-2.png" alt="">
                                        </div>
                                    <?php endif; ?>
                                </div>
                                <div class="services-one__content">
                                    <h3 class="services-one__title">
                                        <a href="<?php echo esc_url($link); ?>" target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>">
                                            <?php echo twinkle_kses($item['service_list_title']); ?>
                                        </a>
                                    </h3>
                                    <?php if($item['service_list_description']): ?>
                                        <p class="services-one__text">
                                            <?php echo twinkle_kses($item['service_list_description']); ?>
                                        </p>
                                    <?php endif; ?>
                                    <div class="services-one__read-more">
                                        <a href="<?php echo esc_url($link); ?>" target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>">
                                            <?php echo esc_html($item['service_button_text']); ?> <span class="icon-right-arrow1"></span>
                                        </a>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            </section>

        <?php elseif ( $settings['design_style']  == 'layout-2' ): ?>

            <section class="bixola-el-section services-two">
                <?php if ( !empty( $settings['section_shape_switch'] ) ) : ?>
                    <div class="services-two__shape-1 zoominout">
                        <img src="<?php print get_template_directory_uri(); ?>/assets/images/shapes/services-two-shape-1.png" alt="">
                    </div>
                <?php endif; ?>
                <div class="container">
                    <?php if ( !empty( $settings['section_title'] ) ) : ?>
                        <h3 class="bixola-el-section-title section__title-two">
                            <?php print twinkle_kses($settings['section_title']); ?>
                        </h3>
                    <?php endif; ?>
                    <div class="services-two__bottom">
                        <div class="services-two__carousel owl-carousel owl-theme thm-owl__carousel">
                            <?php foreach($settings['service_list_two'] as $index => $item) :

                                if ( !empty($item['service_image']['url']) ) {
                                    $service_image = !empty($item['service_image']['id']) ? wp_get_attachment_image_url( $item['service_image']['id'], 'full') : $item['service_image']['url'];
                                    $service_image_alt = get_post_meta($item["service_image"]["id"], "_wp_attachment_image_alt", true);
                                }
                                    
                                // Link
                                if ('2' == $item['service_link_type']) {
                                    $link = get_permalink($item['service_page_link']);
                                    $target = '_self';
                                    $rel = 'nofollow';
                                } else {
                                    $link = !empty($item['service_link']['url']) ? $item['service_link']['url'] : '';
                                    $target = !empty($item['service_link']['is_external']) ? '_blank' : '';
                                    $rel = !empty($item['service_link']['nofollow']) ? 'nofollow' : '';
                                }
                            ?>
                            <div class="item">
                                <div class="services-two__single">
                                    <div class="services-two__img-box">
                                        <div class="services-two__img">
                                            <img src="<?php print esc_url($service_image); ?>" alt="<?php print esc_attr($service_image_alt); ?>">
                                        </div>
                                    </div>
                                    <div class="services-two__content">
                                        <h3 class="services-two__title">
                                            <a href="<?php echo esc_url($link); ?>" target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>">
                                                <?php echo twinkle_kses($item['service_list_title']); ?>
                                            </a>
                                        </h3>
                                        <?php if($item['service_list_description']): ?>
                                            <p class="services-two__text">
                                                <?php echo twinkle_kses($item['service_list_description']); ?>
                                            </p>
                                        <?php endif; ?>
                                        <div class="services-two__btn">
                                            <a href="<?php echo esc_url($link); ?>" target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>">
                                                <?php echo esc_html($item['service_button_text']); ?> <span class="icon-right-arrow1"></span>
                                            </a>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </div>
            </section>
             
        <?php elseif ( $settings['design_style']  == 'layout-3' ) : 
            
            if ('2' == $settings['section_primary_button_link_type']) {
                $this->add_render_attribute('primary-button-arg', 'href', get_permalink($settings['section_primary_button_page_link']));
                $this->add_render_attribute('primary-button-arg', 'target', '_self');
                $this->add_render_attribute('primary-button-arg', 'rel', 'nofollow');
                $this->add_render_attribute('primary-button-arg', 'class', 'all-services');
            } else {
                if ( ! empty( $settings['section_primary_button_link']['url'] ) ) {
                    $this->add_link_attributes( 'primary-button-arg', $settings['section_primary_button_link'] );
                    $this->add_render_attribute('primary-button-arg', 'class', 'all-services');
                }
            }
            
            if ('2' == $settings['section_secondary_button_link_type']) {
                $this->add_render_attribute('secondary-button-arg', 'href', get_permalink($settings['section_secondary_button_page_link']));
                $this->add_render_attribute('secondary-button-arg', 'target', '_self');
                $this->add_render_attribute('secondary-button-arg', 'rel', 'nofollow');
            } else {
                if ( ! empty( $settings['section_secondary_button_link']['url'] ) ) {
                    $this->add_link_attributes( 'secondary-button-arg', $settings['section_secondary_button_link'] );
                }
            }
            
            ?>

            <section class="bixola-el-section services-three">
                <?php if ( !empty( $settings['section_shape_switch'] ) ) : ?>
                    <div class="services-three__shape-1 float-bob-y">
                        <img src="<?php print get_template_directory_uri(); ?>/assets/images/shapes/services-three-shape-1.png" alt="">
                    </div>
                    <div class="services-three__shape-2 zoominout">
                        <img src="<?php print get_template_directory_uri(); ?>/assets/images/shapes/services-three-shape-2.png" alt="">
                    </div>
                <?php endif; ?>
                <div class="container">
                    <div class="section-title-three text-center">
                        <?php if ( !empty( $settings['section_subheading'] ) ) : ?>
                            <div class="section-title-three__tagline-box">
                                <p class="bixola-el-section-subheading section-title-three__tagline">
                                    <?php print twinkle_kses($settings['section_subheading']); ?>
                                </p>
                            </div>
                        <?php endif; ?>
                        <?php if ( !empty( $settings['section_title'] ) ) : ?>
                            <h2 class="bixola-el-section-title section-title-three__title">
                                <?php print twinkle_kses($settings['section_title']); ?>
                            </h2>
                        <?php endif; ?>
                    </div>
                    <div class="row">
                        <?php foreach($settings['service_list_three'] as $index => $item) :

                            if ( !empty($item['service_image']['url']) ) {
                                $service_image = !empty($item['service_image']['id']) ? wp_get_attachment_image_url( $item['service_image']['id'], 'full') : $item['service_image']['url'];
                                $service_image_alt = get_post_meta($item["service_image"]["id"], "_wp_attachment_image_alt", true);
                            }
                                
                            // Link
                            if ('2' == $item['service_link_type']) {
                                $link = get_permalink($item['service_page_link']);
                                $target = '_self';
                                $rel = 'nofollow';
                            } else {
                                $link = !empty($item['service_link']['url']) ? $item['service_link']['url'] : '';
                                $target = !empty($item['service_link']['is_external']) ? '_blank' : '';
                                $rel = !empty($item['service_link']['nofollow']) ? 'nofollow' : '';
                            }
                        ?>
                        <div class="col-xl-<?php print esc_attr($settings['column_desktop']); ?> col-lg-<?php print esc_attr($settings['column_laptop']); ?> col-md-<?php print esc_attr($settings['column_tablet']); ?> col-sm-<?php print esc_attr($settings['column_mobile']); ?> wow fadeInUp" data-wow-delay="<?php print esc_attr($item['animation_delay']); ?>">
                            <div class="services-three__single">
                                <h3 class="services-three__title">
                                    <a href="<?php echo esc_url($link); ?>" target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>">
                                        <?php echo twinkle_kses($item['service_list_title']); ?>
                                    </a>
                                </h3>
                                <?php if (!empty($item['service_list_image']['url'])): ?>
                                    <div class="services-three__icon">
                                        <img src="<?php echo $item['service_list_image']['url']; ?>" alt="<?php echo get_post_meta(attachment_url_to_postid($item['service_list_image']['url']), '_wp_attachment_image_alt', true); ?>">
                                    </div>
                                <?php endif; ?>
                                <?php if ( !empty($item['service_list_icon']['value']) ) : ?>
                                    <div class="services-three__icon">
                                        <?php bixola_render_icon($item, 'service_list_icon'); ?>
                                    </div>
                                <?php endif; ?>
                                <?php if($item['service_list_description']): ?>
                                    <p class="services-three__text">
                                        <?php echo twinkle_kses($item['service_list_description']); ?>
                                    </p>
                                <?php endif; ?>
                                <?php if ( !empty( $item['service_button_text'] ) ) : ?>
                                    <div class="services-three__btn">
                                        <a href="<?php echo esc_url($link); ?>" target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>">
                                            <?php echo esc_html($item['service_button_text']); ?> <span class="icon-right-arrow1"></span>
                                        </a>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                        <?php endforeach; ?>
                    </div>
                    <p class="services-three__bottom-text">
                        <?php print twinkle_kses($settings['section_primary_button_label']); ?>
                        <a <?php echo $this->get_render_attribute_string( 'primary-button-arg' ); ?>>
                            <?php print twinkle_kses($settings['section_primary_button_text']); ?> <span class="icon-right-arrow-11"></span>
                        </a>
                        <a <?php echo $this->get_render_attribute_string( 'secondary-button-arg' ); ?>>
                            <?php print twinkle_kses($settings['section_primary_button_text']); ?>
                        </a>
                    </p>
                </div>
            </section>

        <?php elseif ( $settings['design_style']  == 'layout-4' ) : ?>

            <section class="bixola-el-section services-page">
                <div class="container">
                    <div class="row">
                        <?php foreach($settings['service_list_three'] as $index => $item) :

                            if ( !empty($item['service_image']['url']) ) {
                                $service_image = !empty($item['service_image']['id']) ? wp_get_attachment_image_url( $item['service_image']['id'], 'full') : $item['service_image']['url'];
                                $service_image_alt = get_post_meta($item["service_image"]["id"], "_wp_attachment_image_alt", true);
                            }
                                
                            // Link
                            if ('2' == $item['service_link_type']) {
                                $link = get_permalink($item['service_page_link']);
                                $target = '_self';
                                $rel = 'nofollow';
                            } else {
                                $link = !empty($item['service_link']['url']) ? $item['service_link']['url'] : '';
                                $target = !empty($item['service_link']['is_external']) ? '_blank' : '';
                                $rel = !empty($item['service_link']['nofollow']) ? 'nofollow' : '';
                            }

                        ?>
                        <div class="col-xl-<?php print esc_attr($settings['column_desktop']); ?> col-lg-<?php print esc_attr($settings['column_laptop']); ?> col-md-<?php print esc_attr($settings['column_tablet']); ?> col-sm-<?php print esc_attr($settings['column_mobile']); ?>">
                            <div class="services-page__single">
                                <h3 class="services-page__title">
                                    <a href="<?php echo esc_url($link); ?>" target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>">
                                        <?php echo twinkle_kses($item['service_list_title']); ?>
                                    </a>
                                </h3>
                                <?php if (!empty($item['service_list_image']['url'])): ?>
                                    <div class="services-page__icon">
                                        <img src="<?php echo $item['service_list_image']['url']; ?>" alt="<?php echo get_post_meta(attachment_url_to_postid($item['service_list_image']['url']), '_wp_attachment_image_alt', true); ?>">
                                    </div>
                                <?php endif; ?>
                                <?php if ( !empty($item['service_list_icon']['value']) ) : ?>
                                    <div class="services-page__icon">
                                        <?php bixola_render_icon($item, 'service_list_icon'); ?>
                                    </div>
                                <?php endif; ?>
                                <?php if($item['service_list_description']): ?>
                                    <p class="services-page__text">
                                        <?php echo twinkle_kses($item['service_list_description']); ?>
                                    </p>
                                <?php endif; ?>
                                <?php if ( !empty( $item['service_button_text'] ) ) : ?>
                                    <div class="services-page__btn">
                                        <a href="<?php echo esc_url($link); ?>" target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>">
                                            <?php echo esc_html($item['service_button_text']); ?> <span class="icon-right-arrow1"></span>
                                        </a>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            </section>

        <?php elseif ( $settings['design_style']  == 'layout-5' ) : ?>

            <section class="bixola-el-section services-carousel-page">
                <div class="container">
                    <div class="services-carousel thm-owl__carousel owl-theme owl-carousel carousel-dot-style">
                        <?php foreach($settings['service_list_three'] as $index => $item) :

                            if ( !empty($item['service_image']['url']) ) {
                                $service_image = !empty($item['service_image']['id']) ? wp_get_attachment_image_url( $item['service_image']['id'], 'full') : $item['service_image']['url'];
                                $service_image_alt = get_post_meta($item["service_image"]["id"], "_wp_attachment_image_alt", true);
                            }
                                
                            // Link
                            if ('2' == $item['service_link_type']) {
                                $link = get_permalink($item['service_page_link']);
                                $target = '_self';
                                $rel = 'nofollow';
                            } else {
                                $link = !empty($item['service_link']['url']) ? $item['service_link']['url'] : '';
                                $target = !empty($item['service_link']['is_external']) ? '_blank' : '';
                                $rel = !empty($item['service_link']['nofollow']) ? 'nofollow' : '';
                            }

                        ?>
                        <div class="item">
                            <div class="services-page__single">
                                <h3 class="services-page__title">
                                    <a href="<?php echo esc_url($link); ?>" target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>">
                                        <?php echo twinkle_kses($item['service_list_title']); ?>
                                    </a>
                                </h3>
                                <?php if (!empty($item['service_list_image']['url'])): ?>
                                    <div class="services-page__icon">
                                        <img src="<?php echo $item['service_list_image']['url']; ?>" alt="<?php echo get_post_meta(attachment_url_to_postid($item['service_list_image']['url']), '_wp_attachment_image_alt', true); ?>">
                                    </div>
                                <?php endif; ?>
                                <?php if ( !empty($item['service_list_icon']['value']) ) : ?>
                                    <div class="services-page__icon">
                                        <?php bixola_render_icon($item, 'service_list_icon'); ?>
                                    </div>
                                <?php endif; ?>
                                <?php if($item['service_list_description']): ?>
                                    <p class="services-page__text">
                                        <?php echo twinkle_kses($item['service_list_description']); ?>
                                    </p>
                                <?php endif; ?>
                                <?php if ( !empty( $item['service_button_text'] ) ) : ?>
                                    <div class="services-page__btn">
                                        <a href="<?php echo esc_url($link); ?>" target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>">
                                            <?php echo esc_html($item['service_button_text']); ?> <span class="icon-right-arrow1"></span>
                                        </a>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            </section>

        <?php elseif ( $settings['design_style']  == 'layout-6' ) : ?>

            <section class="bixola-el-section feature-one">
                <div class="container">
                    <div class="row">
                        <div class="col-xl-4">
                            <div class="feature-one__left">
                                <div class="section-title text-left">
                                    <?php if ( !empty( $settings['section_subheading'] ) ) : ?>
                                        <div class="section-title__tagline-box">
                                            <p class="bixola-el-section-subheading section-title__tagline">
                                                <?php print twinkle_kses($settings['section_subheading']); ?>
                                            </p>
                                        </div>
                                    <?php endif; ?>
                                    <?php if ( !empty( $settings['section_title'] ) ) : ?>
                                        <h2 class="bixola-el-section-title section-title__title">
                                            <?php print twinkle_kses($settings['section_title']); ?>
                                        </h2>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>
                        <div class="col-xl-8">
                            <div class="feature-one__right">
                                <div class="row">
                                    <?php foreach($settings['service_list_one'] as $index => $item) :

                                        if ( !empty($item['service_image']['url']) ) {
                                            $service_image = !empty($item['service_image']['id']) ? wp_get_attachment_image_url( $item['service_image']['id'], 'full') : $item['service_image']['url'];
                                            $service_image_alt = get_post_meta($item["service_image"]["id"], "_wp_attachment_image_alt", true);
                                        }
                                            
                                        // Link
                                        if ('2' == $item['service_link_type']) {
                                            $link = get_permalink($item['service_page_link']);
                                            $target = '_self';
                                            $rel = 'nofollow';
                                        } else {
                                            $link = !empty($item['service_link']['url']) ? $item['service_link']['url'] : '';
                                            $target = !empty($item['service_link']['is_external']) ? '_blank' : '';
                                            $rel = !empty($item['service_link']['nofollow']) ? 'nofollow' : '';
                                        }

                                    ?>
                                    <div class="col-xl-<?php print esc_attr($settings['column_desktop']); ?> col-lg-<?php print esc_attr($settings['column_laptop']); ?> col-md-<?php print esc_attr($settings['column_tablet']); ?> col-sm-<?php print esc_attr($settings['column_mobile']); ?> wow fadeInLeft" data-wow-delay="<?php print esc_attr($item['animation_delay']); ?>">
                                        <div class="feature-one__single">
                                            <?php if ( !empty( $service_image ) ) : ?>
                                                <div class="feature-one__single-hover-bg" style="background-image: url(<?php print esc_url($service_image); ?>);"></div>
                                            <?php endif; ?>
                                            <?php if (!empty($item['service_list_image']['url'])): ?>
                                                <div class="feature-one__icon">
                                                    <img src="<?php echo $item['service_list_image']['url']; ?>" alt="<?php echo get_post_meta(attachment_url_to_postid($item['service_list_image']['url']), '_wp_attachment_image_alt', true); ?>">
                                                </div>
                                            <?php endif; ?>
                                            <?php if ( !empty($item['service_list_icon']['value']) ) : ?>
                                                <div class="feature-one__icon">
                                                    <?php bixola_render_icon($item, 'service_list_icon'); ?>
                                                </div>
                                            <?php endif; ?>
                                            <h3 class="feature-one__title">
                                                <a href="<?php echo esc_url($link); ?>" target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>">
                                                    <?php echo twinkle_kses($item['service_list_title']); ?>
                                                </a>
                                            </h3>
                                            <?php if($item['service_list_description']): ?>
                                                <p class="feature-one__text">
                                                    <?php echo twinkle_kses($item['service_list_description']); ?>
                                                </p>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                    <?php endforeach; ?>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </section>

        <?php endif; ?>

        <?php 
    }
}

$widgets_manager->register( new Bixola_Services() );