<?php

namespace BixolaCore\Widgets;
use \Elementor\Widget_Base;
use \Elementor\Control_Media;
use \Elementor\Controls_Manager;
use \Elementor\Group_Control_Typography;
use \Elementor\Repeater;
use \Elementor\Utils;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Bixola Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class Bixola_Service_List extends Widget_Base {

	/**
	 * Retrieve the widget name.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'bixola_service_list';
	}

	/**
	 * Retrieve the widget title.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Service List', 'bixola-core' );
	}

	/**
	 * Retrieve the widget icon.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'bixola-icon';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * Note that currently Elementor supports only one category.
	 * When multiple categories passed, Elementor uses the first one.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'bixola_core' ];
	}

	/**
	 * Retrieve the list of scripts the widget depended on.
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'bixola-core' ];
	}

	/**
	 * Register the widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function register_controls() {

        $this->start_controls_section(
            '_content_service_list',
            [
                'label' => esc_html__( 'Service List', 'bixola-core' ),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $repeater = new Repeater();

        $repeater->add_control(
            'service_list_title', [
                'label' => esc_html__('Text', 'bixola-core'),
                'description' => bixola_get_allowed_html_desc( 'basic' ),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
            ]
        );
        
        $repeater->add_control(
            'service_link_switcher',
            [
                'label' => esc_html__( 'Add Services link', 'bixola-core' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Yes', 'bixola-core' ),
                'label_off' => esc_html__( 'No', 'bixola-core' ),
                'return_value' => 'yes',
                'default' => 'yes',
                'separator' => 'before',
            ]
        );

        $repeater->add_control(
            'service_link_type',
            [
                'label' => esc_html__( 'Service Link Type', 'bixola-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '1' => 'Custom Link',
                    '2' => 'Internal Page',
                ],
                'default' => '1',
                'condition' => [
                    'service_link_switcher' => 'yes'
                ]
            ]
        );

        $repeater->add_control(
            'service_link',
            [
                'label' => esc_html__( 'Service Link link', 'bixola-core' ),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => esc_html__( 'https://your-link.com', 'bixola-core' ),
                'show_external' => true,
                'default' => [
                    'url' => '#',
                    'is_external' => false,
                    'nofollow' => false,
                ],
                'condition' => [
                    'service_link_type' => '1',
                    'service_link_switcher' => 'yes',
                ]
            ]
        );

        $repeater->add_control(
            'service_page_link',
            [
                'label' => esc_html__( 'Select Service Link Page', 'bixola-core' ),
                'type' => Controls_Manager::SELECT2,
                'label_block' => true,
                'options' => bixola_get_all_pages(),
                'condition' => [
                    'service_link_type' => '2',
                    'service_link_switcher' => 'yes',
                ]
            ]
        );

        $this->add_control(
            'service_list',
            [
                'label' => esc_html__('Info List', 'bixola-core'),
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'service_list_title' => esc_html__('Business Planning', 'bixola-core'),
                    ],
                    [
                        'service_list_title' => esc_html__('Financial Advices', 'bixola-core'),
                    ],
                    [
                        'service_list_title' => esc_html__('Tax Strategy', 'bixola-core'),
                    ],
                    [
                        'service_list_title' => esc_html__('Insurance Strategy', 'bixola-core'),
                    ],
                    [
                        'service_list_title' => esc_html__('Start Ups', 'bixola-core'),
                    ],
                ],
                'title_field' => '{{{ service_list_title }}}',
            ]
        );
        $this->end_controls_section();

        $this->start_controls_section(
            '_style_service_list',
            [
                'label' => __( 'Service List', 'bixola-core' ),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'title_bottom_spacing',
            [
                'label' => __( 'Bottom Spacing', 'bixola-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .services-details__services-list li+li' => 'margin-top: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            '_heading_style_service_list',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'List', 'bixola-core' ),
                'separator' => 'before'
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'list_typography',
                'selector' => '{{WRAPPER}} .services-details__services-list li a',
            ]
        );

        $this->start_controls_tabs( '_list_tabs' );
        
        $this->start_controls_tab(
            'list_normal_tab',
            [
                'label' => esc_html__( 'Normal', 'text-domain' ),
            ]
        );
        
        $this->add_control(
            'list_color',
            [
                'label' => __( 'Color', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .services-details__services-list li a' => 'color: {{VALUE}}',
                ],
            ]
        );
        
        $this->add_control(
            'list_background_color',
            [
                'label' => __( 'Background', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .services-details__services-list li a' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        
        $this->end_controls_tab();
        
        $this->start_controls_tab(
            'list_hover_tab',
            [
                'label' => esc_html__( 'Hover', 'text-domain' ),
            ]
        );
        
        $this->add_control(
            'list_color_hover',
            [
                'label' => __( 'Color', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .services-details__services-list li a:hover' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .services-details__services-list li.active a' => 'color: {{VALUE}}',
                ],
            ]
        );
        
        $this->add_control(
            'list_background_hover',
            [
                'label' => __( 'Background', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .services-details__services-list li a::before' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        
        $this->end_controls_tab();
        
        $this->end_controls_tabs();

        $this->end_controls_section();
	}

	/**
	 * Render the widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		?>

            <div class="services-details__services-box">
                <ul class="services-details__services-list list-unstyled">
                    <?php foreach ($settings['service_list'] as $item) : 
                        // Link
                        if ('2' == $item['service_link_type']) {
                            $link = get_permalink($item['service_page_link']);
                            $target = '_self';
                            $rel = 'nofollow';
                        } else {
                            $link = !empty($item['service_link']['url']) ? $item['service_link']['url'] : '';
                            $target = !empty($item['service_link']['is_external']) ? '_blank' : '';
                            $rel = !empty($item['service_link']['nofollow']) ? 'nofollow' : '';
                        }
                    ?>
                    <?php if ( !empty($item['service_list_title']) ) : ?>
                        <li>
                            <a target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>" href="<?php echo esc_url($link); ?>">
                                <?php echo bixola_kses($item['service_list_title' ]); ?> <span class="icon-right-arrow1">
                            </a>
                        </li>
                    <?php endif; ?>
                    <?php endforeach; ?>
                </ul>
            </div>

        <?php 
	}
}

$widgets_manager->register( new Bixola_Service_List() );