<?php
namespace BixolaCore\Widgets;

use Elementor\Widget_Base;
use \Elementor\Repeater;
use \Elementor\Control_Media;
use \Elementor\Utils;
Use \Elementor\Core\Schemes\Typography;
use \Elementor\Controls_Manager;
use \Elementor\Group_Control_Border;
use \Elementor\Group_Control_Typography;
use \Elementor\Group_Control_Image_Size;


if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Bixola Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class Bixola_Pricing_List extends Widget_Base {

	/**
	 * Retrieve the widget name.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'bixola_pricing_list';
	}

	/**
	 * Retrieve the widget title.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Pricing List', 'bixola-core' );
	}

	/**
	 * Retrieve the widget icon.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'bixola-icon';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * Note that currently Elementor supports only one category.
	 * When multiple categories passed, Elementor uses the first one.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'bixola_core' ];
	}

	/**
	 * Retrieve the list of scripts the widget depended on.
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'bixola-core' ];
	}

	/**
	 * Register the widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function register_controls() {
        
        $this->start_controls_section(
            '_content_design_layout',
            [
                'label' => esc_html__( 'Design Layout',  'text-domain'  ),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );
        
        $this->add_control(
            'design_style',
            [
                'label' => esc_html__( 'Select Layout', 'text-domain' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'layout-1' => esc_html__( 'Layout 1', 'text-domain' ),
                    'layout-2' => esc_html__( 'Layout 2', 'text-domain' ),
                ],
                'default' => 'layout-1',
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
            '_section_pricing',
            [
                'label' => __('Pricing', 'bixola-core'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'title',
            [
                'label' => __('Title', 'bixola-core'),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'default' => __('Silver Pack', 'bixola-core'),
                'dynamic' => [
                    'active' => true
                ]
            ]
        );

        $this->add_control(
            'trail',
            [
                'label' => __('Trail', 'bixola-core'),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'default' => __('Get 7 Days Free Trial', 'bixola-core'),
                'dynamic' => [
                    'active' => true
                ]
            ]
        );

        $this->add_control(
            'currency',
            [
                'label' => __('Currency', 'bixola-core'),
                'type' => Controls_Manager::SELECT,
                'label_block' => false,
                'options' => [
                    '' => __('None', 'bixola-core'),
                    'baht' => '&#3647; ' . _x('Baht', 'Currency Symbol', 'bixola-core'),
                    'bdt' => '&#2547; ' . _x('BD Taka', 'Currency Symbol', 'bixola-core'),
                    'dollar' => '&#36; ' . _x('Dollar', 'Currency Symbol', 'bixola-core'),
                    'euro' => '&#128; ' . _x('Euro', 'Currency Symbol', 'bixola-core'),
                    'franc' => '&#8355; ' . _x('Franc', 'Currency Symbol', 'bixola-core'),
                    'guilder' => '&fnof; ' . _x('Guilder', 'Currency Symbol', 'bixola-core'),
                    'krona' => 'kr ' . _x('Krona', 'Currency Symbol', 'bixola-core'),
                    'lira' => '&#8356; ' . _x('Lira', 'Currency Symbol', 'bixola-core'),
                    'peseta' => '&#8359 ' . _x('Peseta', 'Currency Symbol', 'bixola-core'),
                    'peso' => '&#8369; ' . _x('Peso', 'Currency Symbol', 'bixola-core'),
                    'pound' => '&#163; ' . _x('Pound Sterling', 'Currency Symbol', 'bixola-core'),
                    'real' => 'R$ ' . _x('Real', 'Currency Symbol', 'bixola-core'),
                    'ruble' => '&#8381; ' . _x('Ruble', 'Currency Symbol', 'bixola-core'),
                    'rupee' => '&#8360; ' . _x('Rupee', 'Currency Symbol', 'bixola-core'),
                    'indian_rupee' => '&#8377; ' . _x('Rupee (Indian)', 'Currency Symbol', 'bixola-core'),
                    'shekel' => '&#8362; ' . _x('Shekel', 'Currency Symbol', 'bixola-core'),
                    'won' => '&#8361; ' . _x('Won', 'Currency Symbol', 'bixola-core'),
                    'yen' => '&#165; ' . _x('Yen/Yuan', 'Currency Symbol', 'bixola-core'),
                    'custom' => __('Custom', 'bixola-core'),
                ],
                'default' => 'dollar',
            ]
        );

        $this->add_control(
            'currency_custom',
            [
                'label' => __('Custom Symbol', 'bixola-core'),
                'type' => Controls_Manager::TEXT,
                'condition' => [
                    'currency' => 'custom',
                ],
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );

        $this->add_control(
            'price',
            [
                'label' => __('Price', 'bixola-core'),
                'type' => Controls_Manager::TEXT,
                'default' => '11.99',
                'dynamic' => [
                    'active' => true
                ]
            ]
        );

        $this->add_control(
            'period',
            [
                'label' => __('Period', 'bixola-core'),
                'type' => Controls_Manager::TEXT,
                'default' => __('Per Hour', 'bixola-core'),
                'dynamic' => [
                    'active' => true
                ]
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_section_features',
            [
                'label' => __('Features', 'bixola-core'),
            ]
        );

        $repeater = new Repeater();

        $repeater->add_control(
            'single_features_icon',
            [
                'label'       => esc_html__( 'Icon', 'mentor-pricing-table' ),
                'type'        => Controls_Manager::ICONS,
                'default'     => [
                    'value'   => 'fas fa-check-circle',
                    'library' => 'solid',
                ],
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'single_features_icon_color',
            [
                'label'       => esc_html__( 'Icon Color', 'mentor-pricing-table' ),
                'type'        => Controls_Manager::COLOR,
            ]
        );

        $repeater->add_control(
            'features_text',
            [
                'label' => __('Text', 'bixola-core'),
                'type' => Controls_Manager::TEXT,
                'default' => __('Exciting Feature', 'bixola-core'),
                'dynamic' => [
                    'active' => true
                ],
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'single_features_color',
            [
                'label'     => esc_html__( 'Color', 'mentor-pricing-table' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} {{CURRENT_ITEM}} > span' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'features_list',
            [
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'show_label' => false,
                'default' => [
                    [
                        'features_text' => __('Trained Cleane', 'bixola-core'),
                    ],
                    [
                        'features_text' => __('Maintenance Cleaning', 'bixola-core'),
                    ],
                    [
                        'features_text' => __('Liability Insurance', 'bixola-core'),
                    ],
                    [
                        'features_text' => __('Planned Holiday', 'bixola-core'),
                    ],
                    [
                        'features_text' => __('Feedback Centre Access', 'bixola-core'),
                    ],
                ],
                'title_field' => '<# print((features_text)); #>',
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'ribbon_content',
            [
                'label' => esc_html__( 'Ribbon', 'mentor-pricing-table' ),
                'tab'   => Controls_Manager::TAB_CONTENT,
                'condition' => [
                    'design_style' => 'layout-1',
                ],
            ]
        );

        $this->add_control(
            'ribbon_switch',
            [
                'label' => esc_html__( 'Ribbon', 'bixola-core' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Yes', 'bixola-core' ),
                'label_off' => esc_html__( 'No', 'bixola-core' ),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'ribbon_text',
            [
                'label' => __('Text', 'bixola-core'),
                'type' => Controls_Manager::TEXT,
                'default' => __('Most Popular', 'bixola-core'),
                'dynamic' => [
                    'active' => true
                ],
                'condition' => [
                    'ribbon_switch' => 'yes',
                ],
            ]
        );

        $this->end_controls_section();

		// bixola_btn_button_group
        $this->start_controls_section(
            'bixola_btn_button_group',
            [
                'label' => esc_html__('Button', 'bixola-core'),
            ]
        );

        $this->add_control(
            'bixola_btn_text',
            [
                'label' => esc_html__('Button Text', 'bixola-core'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Get Started', 'bixola-core'),
                'title' => esc_html__('Enter button text', 'bixola-core'),
                'label_block' => true,
            ]
        );
        $this->add_control(
            'bixola_btn_link_type',
            [
                'label' => esc_html__('Button Link Type', 'bixola-core'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '1' => 'Custom Link',
                    '2' => 'Internal Page',
                ],
                'default' => '1',
                'label_block' => true,
            ]
        );

        $this->add_control(
            'bixola_btn_link',
            [
                'label' => esc_html__('Button link', 'bixola-core'),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => esc_html__('https://your-link.com', 'bixola-core'),
                'show_external' => false,
                'default' => [
                    'url' => '#',
                    'is_external' => true,
                    'nofollow' => true,
                    'custom_attributes' => '',
                ],
                'condition' => [
                    'bixola_btn_link_type' => '1',
                ],
                'label_block' => true,
            ]
        );
        $this->add_control(
            'bixola_btn_page_link',
            [
                'label' => esc_html__('Select Button Page', 'bixola-core'),
                'type' => Controls_Manager::SELECT2,
                'label_block' => true,
                'options' => bixola_get_all_pages(),
                'condition' => [
                    'bixola_btn_link_type' => '2',
                ]
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
			'_style_design_layout',
			[
				'label' => __( 'Design Layout', 'bixola-core' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

        $this->add_responsive_control(
            'design_layout_margin',
            [
                'label' => __( 'Margin', 'bixola-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .pricing-one__single' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .pricing-two__single' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'design_layout_padding',
            [
                'label' => __( 'Padding', 'bixola-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .pricing-one__single-inner' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .pricing-two__single' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'design_layout_background',
            [
                'label' => __( 'Background', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .pricing-one__single-inner' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .pricing-two__single' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name' => 'design_layout_border',
                'selector' => '{{WRAPPER}} .pricing-one__single-inner, .pricing-two__single',
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_style_pricing',
            [
                'label' => esc_html__( 'Pricing',  'text-domain'  ),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            '_heading_style_package',
            [
                'label' => esc_html__( 'Package Name', 'text-domain' ),
                'type' => Controls_Manager::HEADING,
            ]
        );

        $this->add_responsive_control(
            'package_name_bottom_spacing',
            [
                'label' => __( 'Bottom Spacing', 'bixola-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .pricing-one__price-title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .pricing-two__price-title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'package_name_color',
            [
                'label'     => esc_html__( 'Color', 'bixola-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .pricing-one__price-title'    => 'color: {{VALUE}}',
                    '{{WRAPPER}} .pricing-two__price-title'    => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'label'    => esc_html__( 'Typography', 'bixola-core' ),
                'name'     => 'package_name_typography',
                'selector' => '{{WRAPPER}} .pricing-one__price-title, .pricing-two__price-title',
            ]
        );

        $this->add_control(
            '_heading_style_price',
            [
                'label' => esc_html__( 'Price', 'text-domain' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'pricing_bottom_spacing',
            [
                'label' => __( 'Bottom Spacing', 'bixola-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .pricing-one__price' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .pricing-two__price' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'pricing_color',
            [
                'label'     => esc_html__( 'Color', 'bixola-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .pricing-one__price'    => 'color: {{VALUE}}',
                    '{{WRAPPER}} .pricing-two__price'    => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'label'    => esc_html__( 'Typography', 'bixola-core' ),
                'name'     => 'pricing_typography',
                'selector' => '{{WRAPPER}} .pricing-one__price, .pricing-two__price',
            ]
        );

        $this->add_control(
            '_heading_style_pricing_duration',
            [
                'label' => esc_html__( 'Duration', 'bixola-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'pricing_duration_left_spacing',
            [
                'label' => __( 'Left Spacing', 'bixola-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .pricing-one__price span' => 'margin-left: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'design_style' => 'layout-1',
                ],
            ]
        );

        $this->add_control(
            'pricing_duration_color',
            [
                'label'     => esc_html__( 'Color', 'bixola-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .pricing-one__price span'    => 'color: {{VALUE}}',
                    '{{WRAPPER}} .pricing-two__text'    => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'label'    => esc_html__( 'Typography', 'bixola-core' ),
                'name'     => 'pricing_duration_typography',
                'selector' => '{{WRAPPER}} .pricing-one__price span, .pricing-two__text',
            ]
        );

        $this->add_control(
            '_heading_style_pricing_trial',
            [
                'label' => esc_html__( 'Trial', 'bixola-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
                'condition' => [
                    'design_style' => 'layout-1',
                ],
            ]
        );

        $this->add_responsive_control(
            'pricing_trial_bottom_spacing',
            [
                'label' => __( 'Bottom Spacing', 'bixola-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .pricing-one__text' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'design_style' => 'layout-1',
                ],
            ]
        );

        $this->add_control(
            'pricing_trial_color',
            [
                'label'     => esc_html__( 'Color', 'bixola-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .pricing-one__text'    => 'color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => 'layout-1',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'label'    => esc_html__( 'Typography', 'bixola-core' ),
                'name'     => 'pricing_trial_typography',
                'selector' => '{{WRAPPER}} .pricing-one__text',
                'condition' => [
                    'design_style' => 'layout-1',
                ],
            ]
        );

        $this->add_control(
            'pricing_trail_border',
            [
                'label'     => esc_html__( 'Border Bottom', 'bixola-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .pricing-one__price-box' => 'border-bottom-color: {{VALUE}};',
                ],
                'condition' => [
                    'design_style' => 'layout-1',
                ],
            ]
        );

        $this->add_responsive_control(
            'pricing_trial_border_spacing',
            [
                'label' => __( 'Bottom Spacing', 'bixola-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .pricing-one__price-box' => 'padding-bottom: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'design_style' => 'layout-1',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
			'bixola_features_list_style',
			[
				'label' => __( 'Features List', 'bixola-core' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

        $this->add_responsive_control(
            'features_list_spacing',
            [
                'label' => __( 'Bottom Spacing', 'bixola-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .pricing-plan-one__single .pricing-feature-list ul li' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .pricing-two__points-list li' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'features_list_typography',
                'selector' => '{{WRAPPER}} .pricing-plan-one__single .pricing-feature-list ul li span, .pricing-two__points-list li .text p',
            ]
        );

        $this->add_control(
            'features_list_background',
            [
                'label' => esc_html__( 'Border', 'text-domain' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .pricing-two__points-list::before' => 'background-color: {{VALUE}}',
                ],
            ]
        );

		$this->end_controls_section();

        $this->start_controls_section(
			'bixola_ribbon_style',
			[
				'label' => __( 'Ribbon', 'bixola-core' ),
				'tab' => Controls_Manager::TAB_STYLE,
                'condition' => [
                    'design_style' => 'layout-1',
                ],
			]
		);

        $this->add_control(
            'ribbon_color',
            [
                'label'     => esc_html__( 'Color', 'bixola-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .pricing-one__most-popular p' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'ribbon_background',
            [
                'label'     => esc_html__( 'Background', 'bixola-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .pricing-one__most-popular' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'label'    => esc_html__( 'Typography', 'bixola-core' ),
                'name'     => 'ribbon_typography',
                'selector' => '{{WRAPPER}} .pricing-one__most-popular p',
            ]
        );

		$this->end_controls_section();

        $this->start_controls_section(
			'bixola_button_style',
			[
				'label' => __( 'Button', 'bixola-core' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

        $this->start_controls_tabs( 'tabs_button_style' );

        $this->start_controls_tab(
            'button_tab',
            [
                'label' => esc_html__( 'Normal', 'bixola-core' ),
            ]
        );

        $this->add_control(
            'button_color',
            [
                'label'     => esc_html__( 'Text', 'bixola-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .thm-btn' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .pricing-two__btn' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'button_background_color',
            [
                'label'     => esc_html__( 'Background', 'bixola-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .thm-btn' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .pricing-two__btn' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'label'    => esc_html__( 'Border', 'bixola-core' ),
                'name'     => 'button_border',
                'selector' => '{{WRAPPER}} .thm-btn, .pricing-two__btn',
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'button_hover_tab',
            [
                'label' => esc_html__( 'Hover', 'bixola-core' ),
            ]
        );

        $this->add_control(
            'button_color_hover',
            [
                'label'     => esc_html__( 'Text', 'bixola-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .thm-btn:hover' => 'color: {{VALUE}}!important;',
                    '{{WRAPPER}} .pricing-two__btn:hover' => 'color: {{VALUE}}!important;',
                ],
            ]
        );

        $this->add_control(
            'button_background_hover',
            [
                'label'     => esc_html__( 'Background', 'bixola-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .thm-btn:before' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .pricing-two__btn:before' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'label'    => esc_html__( 'Border', 'bixola-core' ),
                'name'     => 'button_border_hover',
                'selector' => '{{WRAPPER}} .thm-btn:hover, .pricing-two__btn:hover',
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->add_control(
            'button_border_radius',
            [
                'label'     => esc_html__( 'Border Radius', 'bixola-core' ),
                'type'      => Controls_Manager::SLIDER,
                'selectors' => [
                    '{{WRAPPER}} .thm-btn' => 'border-radius: {{SIZE}}px;',
                    '{{WRAPPER}} .thm-btn:before' => 'border-radius: {{SIZE}}px;',
                    '{{WRAPPER}} .pricing-two__btn' => 'border-radius: {{SIZE}}px;',
                    '{{WRAPPER}} .pricing-two__btn:before' => 'border-radius: {{SIZE}}px;',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'label'    => esc_html__( 'Typography', 'bixola-core' ),
                'name'     => 'button_typography',
                'selector' => '{{WRAPPER}} .thm-btn, .pricing-two__btn',
            ]
        );

		$this->end_controls_section();

	}

    private static function get_currency_symbol($symbol_name)
    {
        $symbols = [
            'baht' => '&#3647;',
            'bdt' => '&#2547;',
            'dollar' => '&#36;',
            'euro' => '&#128;',
            'franc' => '&#8355;',
            'guilder' => '&fnof;',
            'indian_rupee' => '&#8377;',
            'pound' => '&#163;',
            'peso' => '&#8369;',
            'peseta' => '&#8359',
            'lira' => '&#8356;',
            'ruble' => '&#8381;',
            'shekel' => '&#8362;',
            'rupee' => '&#8360;',
            'real' => 'R$',
            'krona' => 'kr',
            'won' => '&#8361;',
            'yen' => '&#165;',
        ];

        return isset($symbols[$symbol_name]) ? $symbols[$symbol_name] : '';
    }


	/**
	 * Render the widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

	        if ($settings['currency'] === 'custom') {
	            $currency = $settings['currency_custom'];
	        } else {
	            $currency = self::get_currency_symbol($settings['currency']);
	        }
	        

		?>

            <?php if ( $settings['design_style']  == 'layout-1' ) : 
                
                if ('2' == $settings['bixola_btn_link_type']) {
                    $this->add_render_attribute('bixola-button-arg', 'href', get_permalink($settings['bixola_btn_page_link']));
                    $this->add_render_attribute('bixola-button-arg', 'target', '_self');
                    $this->add_render_attribute('bixola-button-arg', 'rel', 'nofollow');
                    $this->add_render_attribute('bixola-button-arg', 'class', 'thm-btn pricing-one__btn');
                } else {
                    if ( ! empty( $settings['bixola_btn_link']['url'] ) ) {
                        $this->add_link_attributes( 'bixola-button-arg', $settings['bixola_btn_link'] );
                        $this->add_render_attribute('bixola-button-arg', 'class', 'thm-btn pricing-one__btn');
                    }
                }

            ?>
            
                <div class="pricing-one__single">
                    <div class="pricing-one__single-inner">
                        <?php if ($settings['ribbon_switch']) : ?>
                            <div class="pricing-one__most-popular">
                                <p><?php echo twinkle_kses($settings['ribbon_text']); ?></p>
                            </div>
                        <?php endif; ?>
                        <div class="pricing-one__price-box">
                            <?php if ( !empty( $settings['title'] )) : ?>
                                <p class="pricing-one__price-title">
                                    <?php echo twinkle_kses($settings['title']); ?>
                                </p>
                            <?php endif; ?>
                            <h3 class="pricing-one__price">
                                <?php echo esc_html($currency); ?><?php echo twinkle_kses($settings['price']); ?>
                                <?php if ($settings['period']) : ?>
                                    <span><?php echo twinkle_kses($settings['period']); ?></span>
                                <?php endif; ?>
                            </h3>
                            <?php if ($settings['trail']) : ?>
                                <p class="pricing-one__text">
                                    <?php echo twinkle_kses($settings['trail']); ?>
                                </p>
                            <?php endif; ?>
                        </div>
                        <ul class="list-unstyled pricing-one__points-list">
                            <?php foreach ($settings['features_list'] as $index => $item) : ?>
                                <li class="elementor-repeater-item-<?php echo esc_attr( $item['_id'] ); ?>">
                                    <div class="icon" style="color: <?php echo esc_attr( $item['single_features_icon_color'] ); ?>">
                                        <?php \Elementor\Icons_Manager::render_icon( $item['single_features_icon'], ['aria-hidden' => 'true'] );?>
                                    </div>
                                    <div class="text" style="color: <?php echo esc_attr( $item['single_features_color'] ); ?>">
                                        <p><?php echo twinkle_kses($item['features_text']); ?></p>
                                    </div>
                                </li>
                            <?php endforeach; ?>
                        </ul>
                        <div class="pricing-one__btn-box">
                            <a <?php echo $this->get_render_attribute_string( 'bixola-button-arg' ); ?>>
                                <?php echo $settings['bixola_btn_text']; ?>
                            </a>
                        </div>
                    </div>
                </div>
            
            <?php elseif ( $settings['design_style']  == 'layout-2' ) : 
                
                if ('2' == $settings['bixola_btn_link_type']) {
                    $this->add_render_attribute('bixola-button-arg', 'href', get_permalink($settings['bixola_btn_page_link']));
                    $this->add_render_attribute('bixola-button-arg', 'target', '_self');
                    $this->add_render_attribute('bixola-button-arg', 'rel', 'nofollow');
                    $this->add_render_attribute('bixola-button-arg', 'class', 'thm-btn-two pricing-two__btn');
                } else {
                    if ( ! empty( $settings['bixola_btn_link']['url'] ) ) {
                        $this->add_link_attributes( 'bixola-button-arg', $settings['bixola_btn_link'] );
                        $this->add_render_attribute('bixola-button-arg', 'class', 'thm-btn-two pricing-two__btn');
                    }
                }
                
            ?>

                <div class="pricing-two__single pricing-two__single-2">
                    <div class="pricing-two__single-left">
                        <div class="pricing-two__price-box">
                            <?php if ( !empty( $settings['title'] )) : ?>
                                <p class="pricing-two__price-title">
                                    <?php echo twinkle_kses($settings['title']); ?>
                                </p>
                            <?php endif; ?>
                            <h3 class="pricing-two__price">
                                <?php echo esc_html($currency); ?><?php echo twinkle_kses($settings['price']); ?>
                            </h3>
                            <?php if ($settings['period']) : ?>
                                <p class="pricing-two__text">
                                    <?php echo twinkle_kses($settings['period']); ?>
                                </p>
                            <?php endif; ?>
                        </div>
                        <div class="pricing-two__btn-box">
                            <a <?php echo $this->get_render_attribute_string( 'bixola-button-arg' ); ?>>
                                <?php echo $settings['bixola_btn_text']; ?>
                            </a>
                        </div>
                    </div>
                    <ul class="list-unstyled pricing-two__points-list">
                        <?php foreach ($settings['features_list'] as $index => $item) : ?>
                            <li class="elementor-repeater-item-<?php echo esc_attr( $item['_id'] ); ?>">
                                <div class="icon" style="color: <?php echo esc_attr( $item['single_features_icon_color'] ); ?>">
                                    <?php \Elementor\Icons_Manager::render_icon( $item['single_features_icon'], ['aria-hidden' => 'true'] );?>
                                </div>
                                <div class="text" style="color: <?php echo esc_attr( $item['single_features_color'] ); ?>">
                                    <p><?php echo twinkle_kses($item['features_text']); ?></p>
                                </div>
                            </li>
                        <?php endforeach; ?>
                    </ul>
                </div>
            
            <?php endif; ?>

        <?php
	}

}

$widgets_manager->register( new Bixola_Pricing_List() );