<?php
namespace BixolaCore\Widgets;

use \Elementor\Widget_Base;
use \Elementor\Controls_Manager;
use \Elementor\Group_Control_Border;
use \Elementor\Group_Control_Typography;
use \Elementor\Repeater;
use \Elementor\Utils;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Bixola Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class Bixola_Portfolio_Slider extends Widget_Base {

	/**
	 * Retrieve the widget name.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'bixola_portfolio_slider';
	}

	/**
	 * Retrieve the widget title.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Portfolio Slider', 'bixola-core' );
	}

	/**
	 * Retrieve the widget icon.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'bixola-icon';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * Note that currently Elementor supports only one category.
	 * When multiple categories passed, Elementor uses the first one.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'bixola_core' ];
	}

	/**
	 * Retrieve the list of scripts the widget depended on.
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'bixola-core' ];
	}

	/**
	 * Register the widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function register_controls() {

        $this->start_controls_section(
            '_content_slider',
            [
                'label' => esc_html__('Portfolio Slider', 'bixola-core'),
            ]
        );

        $this->add_control(
            'bixola_design_style',
            [
                'label' => esc_html__('Select Layout', 'bixola-core'),
                'type' => Controls_Manager::HIDDEN,
                'options' => [
                    'layout-1' => esc_html__('Layout 1', 'bixola-core'),
                ],
                'default' => 'layout-1',
            ]
        );

        $repeater = new Repeater();

        $repeater->add_control(
            'image',
            [
                'type' => Controls_Manager::MEDIA,
                'label' => __( 'Image', 'bixola-core' ),
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );                      

        $repeater->add_control(
            'title',
            [
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'label' => __( 'Title', 'bixola-core' ),
                'default' => __( 'Wealth Wise', 'bixola-core' ),
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );

        $repeater->add_control(
            'subtitle',
            [
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'show_label' => true,
                'label' => __( 'Subtitle', 'bixola-core' ),
                'default' => __( 'Business Audit', 'bixola-core' ),
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );

        $repeater->add_control(
            'bixola_link_switcher',
            [
                'label' => esc_html__( 'Show Link', 'bixola-core' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Yes', 'bixola-core' ),
                'label_off' => esc_html__( 'No', 'bixola-core' ),
                'return_value' => 'yes',
                'default' => 'yes',
                'separator' => 'before',
            ]
        );

        $repeater->add_control(
            'bixola_link_type',
            [
                'label' => esc_html__( 'Service Link Type', 'bixola-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '1' => 'Custom Link',
                    '2' => 'Internal Page',
                ],
                'default' => '1',
                'condition' => [
                    'bixola_link_switcher' => 'yes'
                ]
            ]
        );

        $repeater->add_control(
            'bixola_link',
            [
                'label' => esc_html__( 'Service Link link', 'bixola-core' ),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => esc_html__( 'https://your-link.com', 'bixola-core' ),
                'show_external' => true,
                'default' => [
                    'url' => '#',
                    'is_external' => false,
                    'nofollow' => false,
                ],
                'condition' => [
                    'bixola_link_type' => '1',
                    'bixola_link_switcher' => 'yes',
                ]
            ]
        );

        $repeater->add_control(
            'bixola_page_link',
            [
                'label' => esc_html__( 'Select Service Link Page', 'bixola-core' ),
                'type' => Controls_Manager::SELECT2,
                'label_block' => true,
                'options' => bixola_get_all_pages(),
                'condition' => [
                    'bixola_link_type' => '2',
                    'bixola_link_switcher' => 'yes',
                ]
            ]
        );

        // REPEATER
        $this->add_control(
            'portfolio_list',
            [
                'show_label' => false,
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'title_field' => '<# print(title || "Carousel Item"); #>',
                'default' => [
                    [
                        'image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                        'subtitle' => __( 'Business Audit', 'bixola-core' ),
                        'title' => __( 'Our Business Growth', 'bixola-core' ),
                    ],
                    [
                        'image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                        'subtitle' => __( 'Business Audit', 'bixola-core' ),
                        'title' => __( 'Our Womens Empowerment', 'bixola-core' ),
                    ],
                    [
                        'image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                        'subtitle' => __( 'Business Audit', 'bixola-core' ),
                        'title' => __( 'Our Business Audit', 'bixola-core' ),
                    ],
                    [
                        'image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                        'subtitle' => __( 'Business Audit', 'bixola-core' ),
                        'title' => __( 'Our Business Planning', 'bixola-core' ),
                    ],
                    [
                        'image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                        'subtitle' => __( 'Business Audit', 'bixola-core' ),
                        'title' => __( 'Our Financial Advices', 'bixola-core' ),
                    ],
                    [
                        'image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                        'subtitle' => __( 'Business Audit', 'bixola-core' ),
                        'title' => __( 'Our Tax Strategy', 'bixola-core' ),
                    ],
                    [
                        'image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                        'subtitle' => __( 'Business Audit', 'bixola-core' ),
                        'title' => __( 'Solutions Your Problem', 'bixola-core' ),
                    ],
                    [
                        'image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                        'subtitle' => __( 'Business Audit', 'bixola-core' ),
                        'title' => __( 'Identifying Problems', 'bixola-core' ),
                    ],
                    [
                        'image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                        'subtitle' => __( 'Business Audit', 'bixola-core' ),
                        'title' => __( 'Research Your Problem', 'bixola-core' ),
                    ],
                ]
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_style_design_layout',
            [
                'label' => esc_html__( 'Design Layout',  'text-domain'  ),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );
        
        $this->add_responsive_control(
            'design_layout_margin',
            [
                'label' => esc_html__( 'Margin', 'text-domain' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-section' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        
        $this->add_responsive_control(
            'design_layout_padding',
            [
                'label' => esc_html__( 'Padding', 'text-domain' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-section' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        
        $this->add_control(
            'design_layout_background',
            [
                'label' => esc_html__( 'Background', 'text-domain' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-section' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            '_heading_style_navigation',
            [
                'label' => esc_html__( 'Navigation', 'text-domain' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->start_controls_tabs( '_tabs_navigation' );
        
        $this->start_controls_tab(
            'navigation_normal_tab',
            [
                'label' => esc_html__( 'Normal', 'text-domain' ),
            ]
        );
        
        $this->add_control(
            'navigation_background',
            [
                'label' => esc_html__( 'Background', 'text-domain' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .carousel-dot-style.owl-carousel .owl-dots .owl-dot' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        
        $this->end_controls_tab();
        
        $this->start_controls_tab(
            'navigation_active_tab',
            [
                'label' => esc_html__( 'Active', 'text-domain' ),
            ]
        );
        
        $this->add_control(
            'navigation_background_active',
            [
                'label' => esc_html__( 'Background', 'text-domain' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .carousel-dot-style.owl-carousel .owl-dot.active' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        
        $this->end_controls_tab();
        
        $this->end_controls_tabs();
        
        $this->end_controls_section();

        $this->start_controls_section(
			'_style_portfolio_slider',
			[
				'label' => __( 'Portfolio Slider', 'bixola-core' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

        // Title
        $this->add_control(
            '_heading_style_portfolio_slider_title',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Title', 'bixola-core' ),
            ]
        );

        $this->add_responsive_control(
            'portfolio_slider_title_bottom_spacing',
            [
                'label' => __( 'Bottom Spacing', 'bixola-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .portfolio-three__title a' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

		$this->add_control(
            'portfolio_slider_title_color',
            [
                'label' => __( 'Color', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio-three__title a' => 'color: {{VALUE}}',
                ],
            ]
        );

		$this->add_control(
            'portfolio_slider_title_color_hover',
            [
                'label' => __( 'Color (Hover)', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio-three__single:hover .portfolio-three__title a' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'portfolio_slider_title_typography',
                'selector' => '{{WRAPPER}} .portfolio-three__title a',
            ]
        );

        // Name
        $this->add_control(
            '_heading_style_portfolio_slider_subtitle',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Subtitle', 'bixola-core' ),
                'separator' => 'before'
            ]
        );

		$this->add_control(
            'portfolio_slider_subtitle_color',
            [
                'label' => __( 'Color', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio-three__single:hover .portfolio-three__sub-title' => 'color: {{VALUE}}',
                ],
            ]
        );

		$this->add_control(
            'portfolio_slider_subtitle_color_hover',
            [
                'label' => __( 'Color (Hover)', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio-three__single:hover .portfolio-three__sub-title' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'portfolio_slider_subtitle_typography',
                'selector' => '{{WRAPPER}} .portfolio-three__single:hover .portfolio-three__sub-title',
            ]
        );

        $this->add_control(
            '_heading_style_portfolio_slider_layout',
            [
                'label' => esc_html__( 'Layout', 'bixola-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

		$this->add_control(
            'portfolio_slider_layout_image_hover',
            [
                'label' => __( 'Image Hover', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio-three__img:before' => 'background-color: {{VALUE}}!important;',
                ],
            ]
        );

		$this->add_control(
            'portfolio_slider_layout_background',
            [
                'label' => __( 'Background', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio-three__content' => 'background-color: {{VALUE}}!important;',
                ],
            ]
        );

		$this->add_control(
            'portfolio_slider_layout_background_hover',
            [
                'label' => __( 'Background (Hover)', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio-three__content::before' => 'background-color: {{VALUE}}!important;',
                ],
            ]
        );

        $this->add_responsive_control(
            'portfolio_slider_layout_border_radius',
            [
                'label' => __( 'Border Radius', 'bixola-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .portfolio-three__content' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'portfolio_slider_layout_padding',
            [
                'label' => __( 'Padding', 'bixola-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .portfolio-three__content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'portfolio_slider_layout_margin',
            [
                'label' => __( 'Margin', 'bixola-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .portfolio-three__content' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

		$this->end_controls_section();
	}

	/**
	 * Render the widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();
        $this->add_render_attribute('title_args', 'class', 'section-title__title');
        

		?>

            <?php if ( $settings['bixola_design_style']  == 'layout-1' ): ?>

                <section class="bixola-el-section portfolio-carousel-page">
                    <div class="container">
                        <div class="portfolio-carousel thm-owl__carousel owl-theme owl-carousel carousel-dot-style">
                            <?php foreach ( $settings['portfolio_list'] as $item ) :

                                if ( !empty($item['image']['url']) ) {
                                    $bixola_portfolio_image_url = !empty($item['image']['id']) ? wp_get_attachment_image_url( $item['image']['id'], 'full') : $item['image']['url'];
                                    $bixola_portfolio_image_alt = get_post_meta($item["image"]["id"], "_wp_attachment_image_alt", true);
                                }

                                // Link
                                if ('2' == $item['bixola_link_type']) {
                                    $link = get_permalink($item['bixola_page_link']);
                                    $target = '_self';
                                    $rel = 'nofollow';
                                } else {
                                    $link = !empty($item['bixola_link']['url']) ? $item['bixola_link']['url'] : '';
                                    $target = !empty($item['bixola_link']['is_external']) ? '_blank' : '';
                                    $rel = !empty($item['bixola_link']['nofollow']) ? 'nofollow' : '';
                                }

                            ?>
                            <div class="item">
                                <div class="portfolio-three__single">
                                    <div class="portfolio-three__img-box">
                                        <div class="portfolio-three__img">
                                            <img src="<?php echo esc_url($bixola_portfolio_image_url); ?>" alt="<?php echo esc_attr($bixola_portfolio_image_alt); ?>">
                                        </div>
                                    </div>
                                    <div class="portfolio-three__content">
                                        <p class="portfolio-three__sub-title">
                                            <?php echo bixola_kses( $item['subtitle'] ); ?>
                                        </p>
                                        <h3 class="portfolio-three__title">
                                            <a target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>" href="<?php echo esc_url($link); ?>">
                                                <?php echo bixola_kses( $item['title'] ); ?>
                                            </a>
                                        </h3>
                                    </div>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </section>

            <?php elseif ( $settings['bixola_design_style']  == 'layout-2' ): ?>

            <?php endif; ?>

		<?php
	}
}

$widgets_manager->register( new bixola_Portfolio_Slider() );