<?php
namespace BixolaCore\Widgets;

use \Elementor\Widget_Base;
use \Elementor\Controls_Manager;
use \Elementor\Group_Control_Background;
use \Elementor\Group_Control_Typography;
use \Elementor\Group_Control_Border;
use \Elementor\Repeater;
use \Elementor\Utils;
use \Elementor\Control_Media;
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Bixola Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class Bixola_Counter extends Widget_Base {

	/**
	 * Retrieve the widget name.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'bixola_counter';
	}

	/**
	 * Retrieve the widget title.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Counter', 'bixola-core' );
	}

	/**
	 * Retrieve the widget icon.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'bixola-icon';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * Note that currently Elementor supports only one category.
	 * When multiple categories passed, Elementor uses the first one.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'bixola_core' ];
	}

	/**
	 * Retrieve the list of scripts the widget depended on.
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'bixola-core' ];
	}

	/**
	 * Register the widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function register_controls() {

        $this->start_controls_section(
            '_content_fun_fact',
            [
                'label' => esc_html__('Fun Fact', 'bixola-core'),
                'tab'   => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'design_style',
            [
                'label' => esc_html__( 'Select Layout', 'bixola-core' ),
                'type' => Controls_Manager::HIDDEN,
                'options' => [
                    'layout-1' => esc_html__( 'Layout 1', 'bixola-core' ),
                    'layout-2' => esc_html__( 'Layout 2', 'bixola-core' ),
                ],
                'default' => 'layout-1',
            ]
        );

        $this->add_control(
            'shape_switch',
            [
                'label' => esc_html__( 'Shape ON/OFF', 'text-domain' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Show', 'text-domain' ),
                'label_off' => esc_html__( 'Hide', 'text-domain' ),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $repeater = new Repeater();

        $repeater->add_control(
            'fun_fact_icon_type',
            [
                'label' => esc_html__( 'Image Type', 'text-domain' ),
                'type' => Controls_Manager::SELECT,
                'default' => 'icon',
                'options' => [
                    'icon' => esc_html__( 'Icon', 'text-domain' ),
                    'image' => esc_html__( 'Image', 'text-domain' ),
                ],
            ]
        );
        
        $repeater->add_control(
            'fun_fact_image_icon',
            [
                'label' => esc_html__( 'Upload Image', 'text-domain' ),
                'type' => Controls_Manager::MEDIA,
                'condition' => [
                    'fun_fact_icon_type' => 'image',
                ],
            ]
        );
        
        $repeater->add_control(
            'fun_fact_icon',
            [
                'label' => esc_html__( 'Icon', 'text-domain' ),
                'type' => Controls_Manager::ICONS,
                'fa4compatibility' => 'icon',
                'label_block' => true,
                'default' => [
                    'value' => 'fas fa-star',
                    'library' => 'solid',
                ],
                'condition' => [
                    'fun_fact_icon_type' => 'icon',
                ],
            ]
        );

        $repeater->add_control(
            'fun_fact_number',
            [
                'label' => esc_html__( 'Number', 'bixola-core' ),
                'type' => Controls_Manager::TEXT,
            ]
        );

        $repeater->add_control(
            'fun_fact_title',
            [
                'label' => esc_html__( 'Title', 'bixola-core' ),
                'type' => Controls_Manager::TEXT,
            ]
        );
        
        $this->add_control(
            'counter_list',
            [
                'label' => esc_html__( ' List', 'bixola-core' ),
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'fun_fact_number' => __( '10', 'bixola-core' ),
                        'fun_fact_title' => __( 'Years of Experience', 'bixola-core' ),
                    ],
                    [
                        'fun_fact_number' => __( '89', 'bixola-core' ),
                        'fun_fact_title' => __( 'Team members', 'bixola-core' ),
                    ],
                    [
                        'fun_fact_number' => __( '789', 'bixola-core' ),
                        'fun_fact_title' => __( 'Successful Projects', 'bixola-core' ),
                    ],
                    [
                        'fun_fact_number' => __( '650', 'bixola-core' ),
                        'fun_fact_title' => __( 'Satisfied Clients', 'bixola-core' ),
                    ],
                ],
                'title_field' => '{{{ fun_fact_title }}}',
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_style_design_layout',
            [
                'label' => esc_html__( 'Design Layout',  'bixola-core'  ),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );
        
        $this->add_responsive_control(
            'design_layout_margin',
            [
                'label' => esc_html__( 'Margin', 'bixola-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'selectors' => [
                    '{{WRAPPER}} .counter-one' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'design_layout_padding',
            [
                'label' => esc_html__( 'Padding', 'bixola-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .counter-one__inner' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'design_layout_background',
            [
                'label' => esc_html__( 'Background', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .counter-one__inner' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
            '_style_fun_fact',
            [
                'label' => esc_html__( 'Fun Fact', 'bixola-core' ),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            '_heading_fun_fact_icon',
            [
                'label' => esc_html__( 'Icon', 'bixola-core' ),
                'type' => Controls_Manager::HEADING,
            ]
        );

        $this->add_responsive_control(
            'fun_fact_icon_size',
            [
                'label' => esc_html__( 'Icon Size', 'text-domain' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .counter-one__icon span' => 'font-size: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'fun_fact_color',
            [
                'label' => esc_html__( 'Color', 'text-domain' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .counter-one__icon span' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            '_heading_fun_fact_number',
            [
                'label' => esc_html__( 'Number', 'bixola-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );
        
        $this->add_control(
            'fun_fact_number_color',
            [
                'label' => esc_html__( 'Color', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .counter-one__count h3' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'fun_fact_number_typography',
                'selector' => '{{WRAPPER}} .counter-one__count h3',
            ]
        );

        $this->add_control(
            '_heading_fun_fact_text',
            [
                'label' => esc_html__( 'Text', 'bixola-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );
        
        $this->add_control(
            'fun_fact_text_color',
            [
                'label' => esc_html__( 'Color', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .counter-one__text' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'fun_fact_text_typography',
                'selector' => '{{WRAPPER}} .counter-one__text',
            ]
        );

		$this->end_controls_section();

	}

	/**
	 * Render the widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();
        
        if ( !empty($settings['section_background_image']['url']) ) {
            $section_background_image = !empty($settings['section_background_image']['id']) ? wp_get_attachment_image_url( $settings['section_background_image']['id'], 'full') : $settings['section_background_image']['url'];
            $section_background_image_alt = get_post_meta($settings["section_background_image"]["id"], "_wp_attachment_image_alt", true);
        }

		?>

        <?php if ( $settings['design_style']  == 'layout-1' ): ?>

            <section class="counter-one">
                <div class="container">
                    <div class="counter-one__inner">
                        <?php if ( !empty( $settings['shape_switch'] ) ) : ?>
                            <div class="counter-one__bg float-bob-y" style="background-image: url(<?php print get_template_directory_uri(); ?>/assets/images/backgrounds/counter-one-bg.png);"></div>
                        <?php endif; ?>
                        <ul class="counter-one__count-list list-unstyled">
                            <?php foreach( $settings['counter_list'] as $key => $item): ?>
                                <li>
                                    <?php if ( $item['fun_fact_icon_type']  == 'image' ): ?>
                                        <div class="counter-one__icon">
                                            <img src="<?php echo $item['fun_fact_image_icon']['url']; ?>" alt="<?php echo get_post_meta(attachment_url_to_postid($item['fun_fact_image_icon']['url']), '_wp_attachment_image_alt', true); ?>">
                                        </div>
                                    <?php elseif ( $item['fun_fact_icon_type']  == 'icon' ): ?>
                                        <div class="counter-one__icon">
                                            <?php bixola_render_icon($item, 'fun_fact_icon' ); ?>
                                        </div>
                                    <?php endif; ?>
                                    <div class="counter-one__count count-box">
                                        <h3 class="count-text" data-stop="<?php print esc_attr($item['fun_fact_number']); ?>" data-speed="1500">
                                            <?php print esc_html( '00', 'bixola-core' ); ?>
                                        </h3>
                                    </div>
                                    <p class="counter-one__text"><?php print twinkle_kses($item['fun_fact_title']); ?></p>
                                </li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                </div>
            </section>
        
        <?php elseif ( $settings['design_style']  == 'layout-2' ): ?>
        
        <?php endif; ?>

		<?php
	}
}

$widgets_manager->register( new Bixola_Counter() );