<?php

namespace BixolaCore\Widgets;

use \Elementor\Widget_Base;
use \Elementor\Controls_Manager;
use \Elementor\Group_Control_Background;
use \Elementor\Group_Control_Image_Size;
use \Elementor\Group_Control_Border;
use \Elementor\Group_Control_Typography;
use \Elementor\Repeater;
use \Elementor\Utils;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Bixola Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class Bixola_Case_Study extends Widget_Base {

    /**
     * Retrieve the widget name.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget name.
     */
    public function get_name() {
        return 'bixola_case_study';
    }

    /**
     * Retrieve the widget title.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget title.
     */
    public function get_title() {
        return __( 'Case Study', 'bixola-core' );
    }

    /**
     * Retrieve the widget icon.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget icon.
     */
    public function get_icon() {
        return 'bixola-icon';
    }

    /**
     * Retrieve the list of categories the widget belongs to.
     *
     * Used to determine where to display the widget in the editor.
     *
     * Note that currently Elementor supports only one category.
     * When multiple categories passed, Elementor uses the first one.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget categories.
     */
    public function get_categories() {
        return [ 'bixola_core' ];
    }

    /**
     * Retrieve the list of scripts the widget depended on.
     *
     * Used to set scripts dependencies required to run the widget.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget scripts dependencies.
     */
    public function get_script_depends() {
        return [ 'bixola-case-study' ];
    }

    /**
     * Register the widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     *
     * @access protected
     */
    protected function register_controls() {

        $this->start_controls_section(
            '_content_design_layout',
            [
                'label' => esc_html__( 'Design Layout',  'text-domain'  ),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );
        
        $this->add_control(
            'design_style',
            [
                'label' => esc_html__( 'Select Layout', 'text-domain' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'layout-1' => esc_html__( 'Layout 1', 'text-domain' ),
                    'layout-2' => esc_html__( 'Layout 2', 'text-domain' ),
                ],
                'default' => 'layout-1',
            ]
        );

        $this->add_control(
            'shape_switch',
            [
                'label' => esc_html__( 'Shape Switch', 'text-domain' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Show', 'text-domain' ),
                'label_off' => esc_html__( 'Hide', 'text-domain' ),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
            '_content_title',
            [
                'label' => esc_html__( 'Title & Content',  'text-domain'  ),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );
        
        $this->add_control(
            'section_subheading',
            [
                'label' => esc_html__( 'Subheading', 'text-domain' ),
                'description' => bixola_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'Latest CASE STUDIES', 'text-domain' ),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'section_title',
            [
                'label' => esc_html__( 'Title', 'text-domain' ),
                'description' => bixola_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'We Are Specialist For Many <br> Consulting Cases', 'text-domain' ),
                'label_block' => true,
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
            '_content_case_study',
            [
                'label' => esc_html__( 'Case Study List', 'bixola-core'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $repeater = new Repeater();

        $repeater->add_control(
            'case_study_image',
            [
                'label' => esc_html__('Upload Icon Image', 'bixola-core'),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],

            ]
        );

        $repeater->add_control(
            'case_study_subheading',
            [
                'label' => esc_html__('Subheading', 'bixola-core'),
                'description' => bixola_get_allowed_html_desc( 'basic' ),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'case_study_title',
            [
                'label' => esc_html__('Title', 'bixola-core'),
                'description' => bixola_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'case_study_link_switcher',
            [
                'label' => esc_html__( 'Add Case Study link', 'bixola-core' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Yes', 'bixola-core' ),
                'label_off' => esc_html__( 'No', 'bixola-core' ),
                'return_value' => 'yes',
                'default' => 'yes',
                'separator' => 'before',
            ]
        );

        $repeater->add_control(
            'case_study_link_type',
            [
                'label' => esc_html__( 'Case Study Link Type', 'bixola-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '1' => 'Custom Link',
                    '2' => 'Internal Page',
                ],
                'default' => '1',
                'condition' => [
                    'case_study_link_switcher' => 'yes'
                ]
            ]
        );

        $repeater->add_control(
            'case_study_link',
            [
                'label' => esc_html__( 'Case Study Link', 'bixola-core' ),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => esc_html__( 'https://your-link.com', 'bixola-core' ),
                'show_external' => true,
                'default' => [
                    'url' => '#',
                    'is_external' => false,
                    'nofollow' => false,
                ],
                'condition' => [
                    'case_study_link_type' => '1',
                    'case_study_link_switcher' => 'yes',
                ]
            ]
        );

        $repeater->add_control(
            'case_study_page_link',
            [
                'label' => esc_html__( 'Select Case Study Link Page', 'bixola-core' ),
                'type' => Controls_Manager::SELECT2,
                'label_block' => true,
                'options' => bixola_get_all_pages(),
                'condition' => [
                    'case_study_link_type' => '2',
                    'case_study_link_switcher' => 'yes',
                ]
            ]
        );

        $repeater->add_control(
            'animation_delay',
            [
                'label' => esc_html__( 'Animation delay', 'text-domain' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '100ms' => esc_html__( '100ms', 'text-domain' ),
                    '200ms' => esc_html__( '200ms', 'text-domain' ),
                    '300ms' => esc_html__( '300ms', 'text-domain' ),
                    '400ms' => esc_html__( '400ms', 'text-domain' ),
                    '500ms' => esc_html__( '500ms', 'text-domain' ),
                    '600ms' => esc_html__( '600ms', 'text-domain' ),
                    '700ms' => esc_html__( '700ms', 'text-domain' ),
                    '800ms' => esc_html__( '800ms', 'text-domain' ),
                    '900ms' => esc_html__( '900ms', 'text-domain' ),
                ],
            ]
        );

        $repeater->add_control(
            'column_desktop',
            [
                'label' => esc_html__( 'Columns for Desktop', 'bixola-core' ),
                'description' => esc_html__( 'Screen width equal to or greater than 1200px', 'bixola-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    12 => esc_html__( '1 Columns', 'bixola-core' ),
                    8 => esc_html__( '2 Large Columns', 'bixola-core' ),
                    6 => esc_html__( '2 Columns', 'bixola-core' ),
                    4 => esc_html__( '3 Columns', 'bixola-core' ),
                    3 => esc_html__( '4 Columns', 'bixola-core' ),
                    5 => esc_html__( '5 Columns (For Carousel Item)', 'bixola-core' ),
                    2 => esc_html__( '6 Columns', 'bixola-core' ),
                    1 => esc_html__( '12 Columns', 'bixola-core' ),
                ],
                'separator' => 'before',
                'default' => 4,
                'style_transfer' => true,
            ]
        );

        $repeater->add_control(
            'column_laptop',
            [
                'label' => esc_html__( 'Columns for Large', 'bixola-core' ),
                'description' => esc_html__( 'Screen width equal to or greater than 992px', 'bixola-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    12 => esc_html__( '1 Columns', 'bixola-core' ),
                    8 => esc_html__( '2 Large Columns', 'bixola-core' ),
                    6 => esc_html__( '2 Columns', 'bixola-core' ),
                    4 => esc_html__( '3 Columns', 'bixola-core' ),
                    3 => esc_html__( '4 Columns', 'bixola-core' ),
                    5 => esc_html__( '5 Columns (For Carousel Item)', 'bixola-core' ),
                    2 => esc_html__( '6 Columns', 'bixola-core' ),
                    1 => esc_html__( '12 Columns', 'bixola-core' ),
                ],
                'separator' => 'before',
                'default' => 4,
                'style_transfer' => true,
            ]
        );

        $repeater->add_control(
            'column_tablet',
            [
                'label' => esc_html__( 'Columns for Tablet', 'bixola-core' ),
                'description' => esc_html__( 'Screen width equal to or greater than 768px', 'bixola-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    12 => esc_html__( '1 Columns', 'bixola-core' ),
                    6 => esc_html__( '2 Columns', 'bixola-core' ),
                    4 => esc_html__( '3 Columns', 'bixola-core' ),
                    3 => esc_html__( '4 Columns', 'bixola-core' ),
                    5 => esc_html__( '5 Columns (For Carousel Item)', 'bixola-core' ),
                    2 => esc_html__( '6 Columns', 'bixola-core' ),
                    1 => esc_html__( '12 Columns', 'bixola-core' ),
                ],
                'separator' => 'before',
                'default' => 6,
                'style_transfer' => true,
            ]
        );

        $repeater->add_control(
            'column_mobile',
            [
                'label' => esc_html__( 'Columns for Mobile', 'bixola-core' ),
                'description' => esc_html__( 'Screen width less than 767px', 'bixola-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    12 => esc_html__( '1 Columns', 'bixola-core' ),
                    6 => esc_html__( '2 Columns', 'bixola-core' ),
                    4 => esc_html__( '3 Columns', 'bixola-core' ),
                    3 => esc_html__( '4 Columns', 'bixola-core' ),
                    5 => esc_html__( '5 Columns (For Carousel Item)', 'bixola-core' ),
                    2 => esc_html__( '6 Columns', 'bixola-core' ),
                    1 => esc_html__( '12 Columns', 'bixola-core' ),
                ],
                'separator' => 'before',
                'default' => 12,
                'style_transfer' => true,
            ]
        );

        $this->add_control(
            'case_study_list',
            [
                'label' => esc_html__( 'Case Study List', 'bixola-core'),
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'case_study_image' => Utils::get_placeholder_image_src(),
                        'case_study_subheading' => __( 'Business Audit', 'bixola-core' ),
                        'case_study_title' => __( 'Printing And Industry', 'bixola-core' ),
                        'animation_delay' => '100ms',
                    ],
                    [
                        'case_study_image' => Utils::get_placeholder_image_src(),
                        'case_study_subheading' => __( 'Business Audit', 'bixola-core' ),
                        'case_study_title' => __( 'Our Business Growth', 'bixola-core' ),
                        'animation_delay' => '200ms',
                    ],
                    [
                        'case_study_image' => Utils::get_placeholder_image_src(),
                        'case_study_subheading' => __( 'Business Audit', 'bixola-core' ),
                        'case_study_title' => __( 'Industrys Standard', 'bixola-core' ),
                        'animation_delay' => '300ms',
                    ],
                    [
                        'case_study_image' => Utils::get_placeholder_image_src(),
                        'case_study_subheading' => __( 'Business Audit', 'bixola-core' ),
                        'case_study_title' => __( 'Business Consulting', 'bixola-core' ),
                        'animation_delay' => '400ms',
                    ],
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_content_button',
            [
                'label' => esc_html__( 'Button', 'text-domain' ),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );
        
        $this->add_control(
            'section_button_text',
            [
                'label' => esc_html__( 'Button Text', 'text-domain' ),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__( 'Button Text', 'text-domain' ),
                'label_block' => true,
            ]
        );
        
        $this->add_control(
            'section_button_link_type',
            [
                'label' => esc_html__( 'Button Link Type', 'text-domain' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '1' => 'Custom Link',
                    '2' => 'Internal Page',
                ],
                'default' => '1',
                'label_block' => true,
            ]
        );
        
        $this->add_control(
            'section_button_link',
            [
                'label' => esc_html__( 'Button link', 'text-domain' ),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => esc_html__('https://your-link.com', 'text-domain'),
                'show_external' => false,
                'default' => [
                    'url' => '#',
                    'is_external' => true,
                    'nofollow' => true,
                    'custom_attributes' => '',
                ],
                'condition' => [
                    'section_button_link_type' => '1',
                ],
                'label_block' => true,
            ]
        );
        $this->add_control(
            'section_button_page_link',
            [
                'label' => esc_html__( 'Select Button Page', 'text-domain' ),
                'type' => Controls_Manager::SELECT2,
                'label_block' => true,
                'options' => bixola_get_all_pages(),
                'condition' => [
                    'section_button_link_type' => '2',
                ]
            ]
        );
        
        $this->add_responsive_control(
            'section_button_align',
            [
                'label' => esc_html__( 'Alignment', 'text-domain' ),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'left' => [
                        'title' => esc_html__( 'Left', 'text-domain' ),
                        'icon' => 'eicon-text-align-left',
                    ],
                    'center' => [
                        'title' => esc_html__( 'Center', 'text-domain' ),
                        'icon' => 'eicon-text-align-center',
                    ],
                    'right' => [
                        'title' => esc_html__( 'Right', 'text-domain' ),
                        'icon' => 'eicon-text-align-right',
                    ],
                ],
                'default' => 'left',
                'toggle' => false,
                'selectors' => [
                    '{{WRAPPER}} ' => 'text-align: {{VALUE}};'
                ]
            ]
        );
        
        $this->add_control(
            'section_button_icon',
            [
                'label' => esc_html__( 'Icon', 'text-domain' ),
                'type' => Controls_Manager::ICONS,
                'fa4compatibility' => 'icon',
                'label_block' => true,
                'default' => [
                    'value' => 'fas fa-long-arrow-alt-right',
                    'library' => 'solid',
                ],
            ]
        );
        
        $this->add_control(
            'section_button_icon_alignment',
            [
                'label'   => esc_html__( 'Icon Position', 'text-domain' ),
                'type'    => Controls_Manager::SELECT,
                'default' => 'before-icon',
                'options' => [
                    'before-icon' => esc_html__( 'Before', 'text-domain' ),
                    'after-icon'  => esc_html__( 'After', 'text-domain' ),
                ],
            ]
        );
        
        $this->add_control(
            'section_button_icon_spacing',
            [
                'label'     => esc_html__( 'Icon Spacing', 'text-domain' ),
                'type'      => Controls_Manager::SLIDER,
                'range'     => [
                    'px' => [
                        'max' => 100,
                    ],
                ],
                'default'   => [
                    'size' => 16,
                ],
                'selectors' => [
                    '{{WRAPPER}} .before-icon i' => 'padding-right: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .after-icon i'  => 'padding-left: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        
        $this->add_control(
            'section_button_icon_size',
            [
                'label'     => esc_html__( 'Icon Size', 'text-domain' ),
                'type'      => Controls_Manager::SLIDER,
                'range'     => [
                    'px' => [
                        'max' => 100,
                    ],
                ],
                'default'   => [
                    'size' => 14,
                ],
                'selectors' => [
                    '{{WRAPPER}} .before-icon i' => 'font-size: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .after-icon i'  => 'font-size: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
			'bixola_layout_style',
			[
				'label' => __( 'Design Layout', 'bixola-core' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

        $this->add_responsive_control(
            'content_padding',
            [
                'label' => __( 'Content Padding', 'bixola-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .project-one' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'content_background',
            [
                'label' => __( 'Content Background', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .project-one' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_section();

        // TAB_STYLE
        $this->start_controls_section(
            '_style_case_study_list',
            [
                'label' => __( 'Case Study List', 'bixola-core' ),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'project_image_background_hover',
            [
                'label' => __( 'Image Background (Hover)', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .project-one__img:before' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            '_icon_title',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Icon', 'bixola-core' ),
                'separator' => 'before'
            ]
        );

        $this->start_controls_tabs( 'tabs_icon_style' );

        $this->start_controls_tab(
            'icon_tab',
            [
                'label' => esc_html__( 'Normal', 'bixola-core' ),
            ]
        );

        $this->add_control(
            'icon_color',
            [
                'label' => __( 'Text', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .project-one__hover a' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'icon_background_color',
            [
                'label' => __( 'Background', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .project-one__hover a' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'icon_hover_tab',
            [
                'label' => esc_html__( 'Hover', 'bixola-core' ),
            ]
        );

        $this->add_control(
            'icon_color_hover',
            [
                'label' => __( 'Text', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .project-one__hover a:hover' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .project-one__hover a:before' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'icon_background_color_hover',
            [
                'label' => __( 'Background', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .project-one__hover a:hover' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .project-one__hover a:before' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->end_controls_section();
    }

    /**
     * Render the widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     *
     * @access protected
     */
    protected function render() {
        $settings = $this->get_settings_for_display();

        if ('2' == $settings['section_button_link_type']) {
            $this->add_render_attribute('section-button-arg', 'href', get_permalink($settings['section_button_page_link']));
            $this->add_render_attribute('section-button-arg', 'target', '_self');
            $this->add_render_attribute('section-button-arg', 'rel', 'nofollow');
            $this->add_render_attribute('section-button-arg', 'class', 'portfolio-one__btn thm-btn');
        } else {
            if ( ! empty( $settings['section_button_link']['url'] ) ) {
                $this->add_link_attributes( 'section-button-arg', $settings['section_button_link'] );
                $this->add_render_attribute('section-button-arg', 'class', 'portfolio-one__btn thm-btn');
            }
        }

        ?>

            <?php if ( $settings['design_style']  == 'layout-1' ): ?>
            
                <section class="portfolio-one">
                    <?php if ( !empty( $settings['shape_switch'] ) ) : ?>
                        <div class="portfolio-one__shape-1 float-bob-x">
                            <img src="<?php print get_template_directory_uri(); ?>/assets/images/shapes/portfolio-one-shape-1.png" alt="">
                        </div>
                        <div class="portfolio-one__shape-2 rotate-me">
                            <img src="<?php print get_template_directory_uri(); ?>/assets/images/shapes/portfolio-one-shape-2.png" alt="">
                        </div>
                    <?php endif; ?>
                    <div class="container">
                        <div class="section-title text-center">
                            <?php if ( !empty( $settings['section_subheading'] ) ) : ?>
                                <div class="section-title__tagline-box">
                                    <p class="section-title__tagline">
                                        <?php print twinkle_kses($settings['section_subheading']); ?>
                                    </p>
                                </div>
                            <?php endif; ?>
                            <?php if ( !empty( $settings['section_title'] ) ) : ?>
                                <h2 class="section-title__title">
                                    <?php print twinkle_kses($settings['section_title']); ?>
                                </h2>
                            <?php endif; ?>
                        </div>
                        <div class="row">
                            <?php foreach ($settings['case_study_list'] as $key => $item) :
                                // Link
                                if ('2' == $item['case_study_link_type']) {
                                    $link = get_permalink($item['case_study_page_link']);
                                    $target = '_self';
                                    $rel = 'nofollow';
                                } else {
                                    $link = !empty($item['case_study_link']['url']) ? $item['case_study_link']['url'] : '';
                                    $target = !empty($item['case_study_link']['is_external']) ? '_blank' : '';
                                    $rel = !empty($item['case_study_link']['nofollow']) ? 'nofollow' : '';
                                }
                            ?>
                            <div class="col-xl-<?php print esc_attr($item['column_desktop']); ?> col-lg-<?php print esc_attr($item['column_laptop']); ?> col-md-<?php print esc_attr($item['column_tablet']); ?> col-sm-<?php print esc_attr($item['column_mobile']); ?> wow fadeInUp" data-wow-delay="<?php print esc_attr($item['animation_delay']); ?>">
                                <div class="portfolio-one__single">
                                    <div class="portfolio-one__img-box">
                                        <div class="portfolio-one__img">
                                            <img src="<?php echo $item['case_study_image']['url']; ?>" alt="<?php echo get_post_meta(attachment_url_to_postid($item['case_study_image']['url']), '_wp_attachment_image_alt', true); ?>">
                                        </div>
                                        <div class="portfolio-one__content">
                                            <?php if ( !empty( $item['case_study_subheading'] ) ) : ?>
                                                <p class="portfolio-one__sub-title">
                                                    <?php echo twinkle_kses( $item['case_study_subheading'] ); ?>
                                                </p>
                                            <?php endif; ?>
                                            <h3 class="portfolio-one__title">
                                                <a target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>" href="<?php echo esc_url($link); ?>">
                                                    <?php echo twinkle_kses( $item['case_study_title'] ); ?>
                                                </a>
                                            </h3>
                                        </div>
                                        <div class="portfolio-one__arrow">
                                            <a href="<?php echo $item['case_study_image']['url']; ?>" class="img-popup">
                                                <span class="icon-top-right-1"></span>
                                            </a>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                        <?php if ( !empty( $settings['section_button_text'] ) ) : ?>
                            <div class="portfolio-one__btn-box">
                                <a <?php echo $this->get_render_attribute_string( 'section-button-arg' ); ?>>
                                    <?php if ( !empty($settings['section_button_icon']['value']) ) : ?>
                                        <span class="<?php echo esc_attr( $settings['section_button_icon_alignment'] ); ?>">
                                            <?php bixola_render_icon($settings, 'section_button_icon'); ?>
                                        </span>
                                    <?php endif; ?>
                                    <?php print twinkle_kses($settings['section_button_text']); ?>
                                </a>
                            </div>
                        <?php endif; ?>
                    </div>
                </section>
            
            <?php elseif ( $settings['design_style']  == 'layout-2' ): ?>
            
                <section class="portfolio-two">
                    <?php if ( !empty( $settings['shape_switch'] ) ) : ?>
                        <div class="portfolio-two__shape-1 float-bob-y">
                            <img src="<?php print get_template_directory_uri(); ?>/assets/images/shapes/portfolio-two-shape-1.png" alt="">
                        </div>
                        <div class="portfolio-two__shape-2 img-bounce">
                            <img src="<?php print get_template_directory_uri(); ?>/assets/images/shapes/portfolio-two-shape-2.png" alt="">
                        </div>
                    <?php endif; ?>
                    <div class="container">
                        <div class="section-title text-center">
                            <?php if ( !empty( $settings['section_subheading'] ) ) : ?>
                                <div class="section-title__tagline-box">
                                    <p class="section-title__tagline">
                                        <?php print twinkle_kses($settings['section_subheading']); ?>
                                    </p>
                                </div>
                            <?php endif; ?>
                            <?php if ( !empty( $settings['section_title'] ) ) : ?>
                                <h2 class="section-title__title">
                                    <?php print twinkle_kses($settings['section_title']); ?>
                                </h2>
                            <?php endif; ?>
                        </div>
                        <div class="row">
                        <?php foreach ($settings['case_study_list'] as $key => $item) :
                            // Link
                            if ('2' == $item['case_study_link_type']) {
                                $link = get_permalink($item['case_study_page_link']);
                                $target = '_self';
                                $rel = 'nofollow';
                            } else {
                                $link = !empty($item['case_study_link']['url']) ? $item['case_study_link']['url'] : '';
                                $target = !empty($item['case_study_link']['is_external']) ? '_blank' : '';
                                $rel = !empty($item['case_study_link']['nofollow']) ? 'nofollow' : '';
                            }
                        ?>
                        <div class="col-xl-<?php print esc_attr($item['column_desktop']); ?> col-lg-<?php print esc_attr($item['column_laptop']); ?> col-md-<?php print esc_attr($item['column_tablet']); ?> col-sm-<?php print esc_attr($item['column_mobile']); ?> wow fadeInUp" data-wow-delay="<?php print esc_attr($item['animation_delay']); ?>">
                                <div class="portfolio-two__single">
                                    <div class="portfolio-two__img-box">
                                    <div class="portfolio-two__img">
                                        <img src="<?php echo $item['case_study_image']['url']; ?>" alt="<?php echo get_post_meta(attachment_url_to_postid($item['case_study_image']['url']), '_wp_attachment_image_alt', true); ?>">
                                    </div>
                                    <div class="portfolio-two__content">
                                        <div class="portfolio-two__title-box">
                                            <h3 class="portfolio-two__title">
                                                <a target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>" href="<?php echo esc_url($link); ?>">
                                                    <?php echo twinkle_kses( $item['case_study_title'] ); ?>
                                                </a>
                                            </h3>
                                            <?php if ( !empty( $item['case_study_subheading'] ) ) : ?>
                                                <p class="portfolio-two__sub-title">
                                                    <?php echo twinkle_kses( $item['case_study_subheading'] ); ?>
                                                </p>
                                            <?php endif; ?>
                                        </div>
                                        <div class="portfolio-two__arrow">
                                            <a href="<?php echo $item['case_study_image']['url']; ?>" class="img-popup">
                                                <span class="icon-top-right-1"></span>
                                            </a>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <?php endforeach; ?>
                    </div>
                </section>
            
            <?php endif; ?>
        <?php 
    }
}

$widgets_manager->register( new Bixola_Case_Study() );