<?php
namespace BixolaCore\Widgets;

use \Elementor\Widget_Base;
use \Elementor\Control_Media;
use \Elementor\Group_Control_Background;
use \Elementor\Group_Control_Css_Filter;
use \Elementor\Repeater;
use \Elementor\Controls_Manager;
use \Elementor\Group_Control_Border;
use \Elementor\Group_Control_Typography;
use \Elementor\Utils;
use \Elementor\Group_Control_Box_Shadow;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Bixola Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class Bixola_Why_Choose_Us extends Widget_Base {

	/**
	 * Retrieve the widget name.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'bixola_why_choose_us';
	}

	/**
	 * Retrieve the widget title.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Why Choose Us', 'bixola-core' );
	}

	/**
	 * Retrieve the widget icon.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'bixola-icon';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * Note that currently Elementor supports only one category.
	 * When multiple categories passed, Elementor uses the first one.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'bixola_core' ];
	}

	/**
	 * Retrieve the list of scripts the widget depended on.
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'bixola-core' ];
	}

	/**
	 * Register the widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function register_controls() {
        
        $this->start_controls_section(
            '_content_design_layout',
            [
                'label' => esc_html__('Design Layout', 'bixola-core'),
            ]
        );

        $this->add_control(
            'design_style',
            [
                'label' => esc_html__('Select Layout', 'bixola-core'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'layout-1' => esc_html__('Layout 1', 'bixola-core'),
                    'layout-2' => esc_html__('Layout 2', 'bixola-core'),
                ],
                'default' => 'layout-1',
            ]
        );

        $this->add_control(
            'shape_switch',
            [
                'label' => esc_html__( 'Shape ON/OFF', 'text-domain' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Show', 'text-domain' ),
                'label_off' => esc_html__( 'Hide', 'text-domain' ),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_content_title',
            [
                'label'       => esc_html__( 'Title & Content', 'bixola-core' ),
                'tab'         => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'section_subheading',
            [
                'label' => esc_html__('Subheading', 'bixola-core'),
                'description' => bixola_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'default' => __('Why Choose Us', 'bixola-core'),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'section_title',
            [
                'label' => esc_html__( 'Title', 'bixola-core' ),
                'description' => bixola_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'We are awards winning <br> business consulting firms', 'bixola-core' ),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'section_description',
            [
                'label' => esc_html__('Description', 'bixola-core'),
                'description' => bixola_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXTAREA,
                'default' => __('Lorem Ipsum is simply dummy text of the printing and typesetting industry. Lorem Ipsum has been the industry standard dummy', 'bixola-core'),
                'label_block' => true,
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_content_image',
            [
                'label' => esc_html__('Image', 'bixola-core' ),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'bixola_image',
            [
                'label' => esc_html__( 'Image', 'bixola-core' ),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_content_info_list',
            [
                'label' => esc_html__( 'Info List', 'bixola-core' ),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $repeater = new Repeater();

        $repeater->add_control(
            'icon_type',
            [
                'label' => esc_html__('Icon Type', 'bixola-core'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'icon' => esc_html__('Icon', 'bixola-core'),
                    'image' => esc_html__('Image', 'bixola-core'),
                ],
                'default' => 'icon',
            ]
        );

        $repeater->add_control(
            'why_choose_us_icon',
            [
                'label' => esc_html__('Icon', 'bixola-core'),
                'type' => Controls_Manager::ICONS,
                'fa4compatibility' => 'icon',
                'label_block' => true,
                'condition'   => [
                    'icon_type' => 'icon',
                ],
            ]
        );

        $repeater->add_control(
            'why_choose_us_image_icon',
            [
                'label'     => esc_html__( 'Choose Image', 'bixola-core' ),
                'type'      => Controls_Manager::MEDIA,
                'default'   => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
                'condition' => [
                    'icon_type'   => ['image'],
                ],
            ]
        );

        $repeater->add_control(
            'info_title',
            [
                'label' => esc_html__( 'Title', 'bixola-core' ),
                'description' => bixola_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'info_description',
            [
                'label' => esc_html__( 'Description', 'bixola-core' ),
                'description' => bixola_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXTAREA,
                'label_block' => true,
            ]
        );

        $this->add_control(
            'bixola_info_list',
            [
                'label' => esc_html__( 'Info - List', 'bixola-core'),
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'info_title' => __( 'Best Business consulting', 'bixola-core' ),
                        'info_description' => __( 'Lorem Ipsum is simply dummy text of the printing and typesetting industry. Lorem Ipsum has been the industry standard dummy', 'bixola-core' ),
                    ],
                    [
                        'info_title' => __( '24/7 customer support', 'bixola-core' ),
                        'info_description' => __( 'Lorem Ipsum is simply dummy text of the printing and typesetting industry. Lorem Ipsum has been the industry standard dummy', 'bixola-core' ),
                    ],
                ],
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
            '_content_button',
            [
                'label' => esc_html__( 'Button', 'bixola-core' ),
                'tab' => Controls_Manager::TAB_CONTENT,
                'condition' => [
                    'design_style' => 'layout-1',
                ],
            ]
        );
        
        $this->add_control(
            'section_button_text',
            [
                'label' => esc_html__( 'Button Text', 'bixola-core' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'Contact Us', 'bixola-core' ),
                'label_block' => true,
            ]
        );
        
        $this->add_control(
            'section_button_link_type',
            [
                'label' => esc_html__( 'Button Link Type', 'bixola-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '1' => 'Custom Link',
                    '2' => 'Internal Page',
                ],
                'default' => '1',
                'label_block' => true,
            ]
        );
        
        $this->add_control(
            'section_button_link',
            [
                'label' => esc_html__( 'Button link', 'bixola-core' ),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => esc_html__('https://your-link.com', 'bixola-core'),
                'show_external' => false,
                'default' => [
                    'url' => '#',
                    'is_external' => true,
                    'nofollow' => true,
                    'custom_attributes' => '',
                ],
                'condition' => [
                    'section_button_link_type' => '1',
                ],
                'label_block' => true,
            ]
        );

        $this->add_control(
            'section_button_page_link',
            [
                'label' => esc_html__( 'Select Button Page', 'bixola-core' ),
                'type' => Controls_Manager::SELECT2,
                'label_block' => true,
                'options' => bixola_get_all_pages(),
                'condition' => [
                    'section_button_link_type' => '2',
                ]
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
            '_style_design_layout',
            [
                'label' => __( 'Design Layout', 'bixola-core' ),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'design_layout_padding',
            [
                'label' => __( 'Padding', 'bixola-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-section' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'design_layout_margin',
            [
                'label' => __( 'Margin', 'bixola-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-section' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'design_layout_background',
            [
                'label' => esc_html__( 'Background', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-section' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_style_title',
            [
                'label' => esc_html__( 'Title & Content', 'bixola-core' ),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            '_heading_subheading',
            [
                'type' => Controls_Manager::HEADING,
                'label' => esc_html__( 'Subheading', 'bixola-core' ),
            ]
        );

        $this->add_responsive_control(
            'subheading_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'bixola-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-section-subheading' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'subheading_color',
            [
                'label' => esc_html__( 'Color', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-section-subheading' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name' => 'subheading_background',
                'types' => [ 'classic', 'gradient' ],
                'selector' => '{{WRAPPER}} .bixola-el-section-subheading:before',
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'subheading_typography',
                'selector' => '{{WRAPPER}} .bixola-el-section-subheading',
            ]
        );

        $this->add_control(
            '_section_heading_title',
            [
                'type' => Controls_Manager::HEADING,
                'label' => esc_html__( 'Title', 'bixola-core' ),
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'section_title_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'bixola-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-section-title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'section_title_color',
            [
                'label' => esc_html__( 'Color', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-section-title' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'section_title_typography',
                'selector' => '{{WRAPPER}} .bixola-el-section-title',
            ]
        );

        $this->add_control(
            '_section_heading_description',
            [
                'type' => Controls_Manager::HEADING,
                'label' => esc_html__( 'Description', 'bixola-core' ),
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'section_description_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'bixola-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-section-description' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'section_description_color',
            [
                'label' => esc_html__( 'Color', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-section-description' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'section_description_typography',
                'selector' => '{{WRAPPER}} .bixola-el-section-description',
            ]
        );

        $this->end_controls_section();
    
        // TAB_STYLE
        $this->start_controls_section(
            '_style_info',
            [
                'label' => __( 'Info List', 'bixola-core' ),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            '_heading_style_info_icon',
            [
                'label' => esc_html__( 'Icon', 'bixola-core' ),
                'type' => Controls_Manager::HEADING,
            ]
        );

        $this->add_control(
            'info_list_icon_size',
            [
                'label'     => esc_html__( 'Icon Size', 'bixola-core' ),
                'type'      => Controls_Manager::SLIDER,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .why-choose-one__points li .icon span' => 'font-size: {{SIZE}}{{UNIT}};',
                ],
            ] 
        );

        $this->add_control(
            'info_list_color',
            [
                'label' => esc_html__( 'Color', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .why-choose-one__points li .icon span' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            '_heading_style_info_title',
            [
                'label' => esc_html__( 'Title', 'bixola-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'info_title_bottom_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'bixola-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .why-choose-one__points li .content h3' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .why-choose-two__points li .content h3' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'info_title_color',
            [
                'label' => esc_html__( 'Color', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .why-choose-one__points li .content h3' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .why-choose-two__points li .content h3' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'info_title_typography',
                'selector' => '{{WRAPPER}} .why-choose-one__points li .content h3, .why-choose-two__points li .content h3',
            ]
        );

        $this->add_control(
            '_heading_style_info_description',
            [
                'label' => esc_html__( 'Description', 'bixola-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'info_description_bottom_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'bixola-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .why-choose-one__points li .content p' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .why-choose-two__points li .content p' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'info_description_color',
            [
                'label' => esc_html__( 'Color', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .why-choose-one__points li .content p' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .why-choose-two__points li .content p' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'info_description_typography',
                'selector' => '{{WRAPPER}} .why-choose-one__points li .content p, .why-choose-two__points li .content p',
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_style_section_button',
            [
                'label' => esc_html__( 'Button', 'bixola-core' ),
                'tab' => Controls_Manager::TAB_STYLE,
                'condition' => [
                    'design_style' => 'layout-1',
                ],
            ]
        );
        
        $this->start_controls_tabs( 'tabs_style_section_button' );
        
        $this->start_controls_tab(
            'section_button_tab',
            [
                'label' => esc_html__( 'Normal', 'bixola-core' ),
            ]
        );
        
        $this->add_control(
            'section_button_color',
            [
                'label'     => esc_html__( 'Color', 'bixola-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-btn'    => 'color: {{VALUE}}',
                ],
            ]
        );
        
        $this->add_control(
            'section_button_background',
            [
                'label'     => esc_html__( 'Background', 'bixola-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-btn' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        
        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'label'    => esc_html__( 'Border', 'bixola-core' ),
                'name'     => 'section_button_border',
                'selector' => '{{WRAPPER}} .bixola-el-btn',
            ]
        );
        
        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name'     => 'section_button_box_shadow',
                'selector' => '{{WRAPPER}} .bixola-el-btn',
            ]
        );
        
        $this->end_controls_tab();
        
        $this->start_controls_tab(
            'section_button_hover_tab',
            [
                'label' => esc_html__( 'Hover', 'bixola-core' ),
            ]
        );
        
        $this->add_control(
            'section_button_color_hover',
            [
                'label'     => esc_html__( 'Color', 'bixola-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-btn:hover' => 'color: {{VALUE}};',
                ],
            ]
        );
        
        $this->add_control(
            'section_button_background_hover',
            [
                'label'     => esc_html__( 'Background', 'bixola-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-btn:after, .bixola-el-btn:before' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        
        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'label'    => esc_html__( 'Border', 'bixola-core' ),
                'name'     => 'section_button_border_hover',
                'selector' => '{{WRAPPER}} .bixola-el-btn:hover',
            ]
        );
        
        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name'     => 'section_button_box_shadow_hover',
                'selector' => '{{WRAPPER}} .bixola-el-btn:hover',
            ]
        );
        
        $this->end_controls_tab();
        
        $this->end_controls_tabs();
        
        $this->add_control(
            'section_button_border_radius',
            [
                'label'     => esc_html__( 'Border Radius', 'bixola-core' ),
                'type'      => Controls_Manager::SLIDER,
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-btn' => 'border-radius: {{SIZE}}px;',
                    '{{WRAPPER}} .bixola-el-btn:before' => 'border-radius: {{SIZE}}px;',
                ],
            ]
        );
        
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'label'    => esc_html__( 'Typography', 'bixola-core' ),
                'name'     => 'section_button_typography',
                'selector' => '{{WRAPPER}} .bixola-el-btn',
            ]
        );
        
        $this->add_control(
            'section_button_padding',
            [
                'label'      => esc_html__( 'Padding', 'bixola-core' ),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .bixola-el-btn' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        
        $this->add_control(
            'section_button_margin',
            [
                'label'      => esc_html__( 'Margin', 'bixola-core' ),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .bixola-el-btn' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        
        $this->end_controls_section();
	}

	/**
	 * Render the widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

        if ( !empty($settings['bixola_image']['url']) ) {
            $bixola_image = !empty($settings['bixola_image']['id']) ? wp_get_attachment_image_url( $settings['bixola_image']['id'], 'full' ) : $settings['bixola_image']['url'];
            $bixola_image_alt = get_post_meta($settings["bixola_image"]["id"], "_wp_attachment_image_alt", true);
        }

		?>

        <?php if ( $settings['design_style']  == 'layout-1' ) : 

                if ('2' == $settings['section_button_link_type']) {
                    $this->add_render_attribute('section-button-arg', 'href', get_permalink($settings['section_button_page_link']));
                    $this->add_render_attribute('section-button-arg', 'target', '_self');
                    $this->add_render_attribute('section-button-arg', 'rel', 'nofollow');
                    $this->add_render_attribute('section-button-arg', 'class', 'bixola-el-btn why-choose-one__btn thm-btn');
                } else {
                    if ( ! empty( $settings['section_button_link']['url'] ) ) {
                        $this->add_link_attributes( 'section-button-arg', $settings['section_button_link'] );
                        $this->add_render_attribute('section-button-arg', 'class', 'bixola-el-btn why-choose-one__btn thm-btn');
                    }
                }
            
            ?>

            <section class="bixola-el-section why-choose-one">
                <?php if ( !empty( $settings['shape_switch'] ) ) : ?>
                    <div class="why-choose-one__shape-3 float-bob-y-2">
                        <img src="<?php print get_template_directory_uri(); ?>/assets/images/shapes/why-choose-one-shape-3.png" alt="">
                    </div>
                <?php endif; ?>
                <div class="container">
                    <div class="row">
                        <div class="col-xl-6">
                            <div class="why-choose-one__left">
                                <div class="section-title text-left">
                                    <?php if(!empty($settings['section_subheading'])): ?>
                                        <div class="section-title__tagline-box">
                                            <p class="bixola-el-section-subheading section-title__tagline">
                                                <?php echo twinkle_kses($settings['section_subheading']); ?>
                                            </p>
                                        </div>
                                    <?php endif; ?>
                                    <?php if(!empty($settings['section_title'])): ?>
                                        <h2 class="bixola-el-section-title section-title__title">
                                            <?php echo twinkle_kses($settings['section_title']); ?>
                                        </h2>
                                    <?php endif; ?>
                                </div>
                                <?php if(!empty($settings['section_description'])): ?>
                                    <p class="bixola-el-section-description why-choose-one__text">
                                        <?php echo twinkle_kses($settings['section_description']); ?>
                                    </p>
                                <?php endif; ?>
                                <ul class="why-choose-one__points list-unstyled">
                                    <?php foreach ($settings['bixola_info_list'] as $key => $item) : 
                                        if ( !empty($item['why_choose_us_image_icon']['url']) ) {
                                            $why_choose_us_image_icon = !empty($item['why_choose_us_image_icon']['id']) ? wp_get_attachment_image_url( $item['why_choose_us_image_icon']['id'], 'full') : $item['why_choose_us_image_icon']['url'];
                                            $why_choose_us_image_icon_alt = get_post_meta($item["why_choose_us_image_icon"]["id"], "_wp_attachment_image_alt", true);
                                        }
                                    ?>
                                    <li>
                                        <?php if ( 'icon' === $item['icon_type'] ) : ?>
                                            <div class="icon">
                                                <?php bixola_render_icon($item, 'why_choose_us_icon'); ?>
                                            </div>
                                        <?php elseif ( 'image' === $item['icon_type'] ) : ?>
                                            <div class="icon">
                                                <img src="<?php echo esc_url( $why_choose_us_image_icon ); ?>" alt="<?php echo esc_attr( $why_choose_us_image_icon_alt ); ?>">
                                            </div>
                                        <?php endif;?>
                                        <div class="content">
                                            <?php if (!empty($item['info_title'])): ?>
                                                <h3><?php print twinkle_kses($item['info_title' ]); ?></h3>
                                            <?php endif; ?>
                                            <?php if (!empty($item['info_description'])): ?>
                                                <p><?php print twinkle_kses($item['info_description' ]); ?></p>
                                            <?php endif; ?>
                                        </div>
                                    </li>
                                    <?php endforeach; ?>
                                </ul>
                                <div class="why-choose-one__btn-box">
                                    <a <?php echo $this->get_render_attribute_string( 'section-button-arg' ); ?>>
                                        <?php print bixola_kses($settings['section_button_text' ]); ?>
                                    </a>
                                </div>
                            </div>
                        </div>
                        <div class="col-xl-6">
                            <div class="why-choose-one__right">
                                <div class="why-choose-one__img wow slideInRight" data-wow-delay="100ms" data-wow-duration="2500ms">
                                    <?php if ( !empty( $settings['shape_switch'] ) ) : ?>
                                        <div class="why-choose-one__shape-1 float-bob-x">
                                            <img src="<?php print get_template_directory_uri(); ?>/assets/images/shapes/why-choose-one-shape-1.png" alt="">
                                        </div>
                                        <div class="why-choose-one__shape-2 float-bob-y">
                                            <img src="<?php print get_template_directory_uri(); ?>/assets/images/shapes/why-choose-one-shape-2.png" alt="">
                                        </div>
                                    <?php endif; ?>
                                    <img src="<?php print esc_url($bixola_image); ?>" alt="<?php print esc_attr($bixola_image_alt); ?>">
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </section>

        <?php elseif ( $settings['design_style']  == 'layout-2' ): ?>

            <section class="bixola-el-section why-choose-two">
                <?php if ( !empty( $settings['shape_switch'] ) ) : ?>
                    <div class="why-choose-two__shape-3 img-bounce">
                        <img src="<?php print get_template_directory_uri(); ?>/assets/images/shapes/why-choose-two-shape-3.png" alt="">
                    </div>
                <?php endif; ?>
                <div class="container">
                    <div class="row">
                        <div class="col-xl-6">
                            <div class="why-choose-two__left">
                                <div class="section-title-three text-left">
                                    <?php if(!empty($settings['section_subheading'])): ?>
                                        <div class="section-title-three__tagline-box">
                                            <p class="bixola-el-section-subheading section-title-three__tagline">
                                                <?php echo twinkle_kses($settings['section_subheading']); ?>
                                            </p>
                                        </div>
                                    <?php endif; ?>
                                    <?php if(!empty($settings['section_title'])): ?>
                                        <h2 class="bixola-el-section-title section-title-three__title">
                                            <?php echo twinkle_kses($settings['section_title']); ?>
                                        </h2>
                                    <?php endif; ?>
                                </div>
                                <?php if(!empty($settings['section_description'])): ?>
                                    <p class="bixola-el-section-description why-choose-two__text">
                                        <?php echo twinkle_kses($settings['section_description']); ?>
                                    </p>
                                <?php endif; ?>
                                <ul class="why-choose-two__points list-unstyled">
                                    <?php foreach ($settings['bixola_info_list'] as $key => $item) : 
                                        if ( !empty($item['why_choose_us_image_icon']['url']) ) {
                                            $why_choose_us_image_icon = !empty($item['why_choose_us_image_icon']['id']) ? wp_get_attachment_image_url( $item['why_choose_us_image_icon']['id'], 'full') : $item['why_choose_us_image_icon']['url'];
                                            $why_choose_us_image_icon_alt = get_post_meta($item["why_choose_us_image_icon"]["id"], "_wp_attachment_image_alt", true);
                                        }
                                    ?>
                                    <li>
                                        <?php if ( 'icon' === $item['icon_type'] ) : ?>
                                            <div class="icon">
                                                <?php bixola_render_icon($item, 'why_choose_us_icon'); ?>
                                            </div>
                                        <?php elseif ( 'image' === $item['icon_type'] ) : ?>
                                            <div class="icon">
                                                <img src="<?php echo esc_url( $why_choose_us_image_icon ); ?>" alt="<?php echo esc_attr( $why_choose_us_image_icon_alt ); ?>">
                                            </div>
                                        <?php endif;?>
                                        <div class="content">
                                            <?php if (!empty($item['info_title'])): ?>
                                                <h3><?php print twinkle_kses($item['info_title' ]); ?></h3>
                                            <?php endif; ?>
                                            <?php if (!empty($item['info_description'])): ?>
                                                <p><?php print twinkle_kses($item['info_description' ]); ?></p>
                                            <?php endif; ?>
                                        </div>
                                    </li>
                                    <?php endforeach; ?>
                                </ul>
                            </div>
                        </div>
                        <div class="col-xl-6">
                            <div class="why-choose-two__right">
                                <div class="why-choose-two__img wow slideInRight" data-wow-delay="100ms" data-wow-duration="2500ms">
                                    <img src="<?php print esc_url($bixola_image); ?>" alt="<?php print esc_attr($bixola_image_alt); ?>">
                                    <?php if ( !empty( $settings['shape_switch'] ) ) : ?>
                                        <div class="why-choose-two__shape-1 float-bob-y">
                                            <img src="<?php print get_template_directory_uri(); ?>/assets/images/shapes/why-choose-two-shape-1.png" alt="">
                                        </div>
                                        <div class="why-choose-two__shape-2 zoominout">
                                            <img src="<?php print get_template_directory_uri(); ?>/assets/images/shapes/why-choose-two-shape-2.png" alt="">
                                        </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </section>

        <?php endif;
	}

}

$widgets_manager->register( new Bixola_Why_Choose_Us() );