<?php
namespace BixolaCore\Widgets;

use \Elementor\Widget_Base;
use \Elementor\Controls_Manager;
use \Elementor\Group_Control_Background;
use \Elementor\Group_Control_Image_Size;
use \Elementor\Group_Control_Typography;
use \Elementor\Group_Control_Border;
use \Elementor\Repeater;
use \Elementor\Utils;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Bixola Core
 * @since 1.0.0
 */
class Bixola_Testimonial_Slider extends Widget_Base {

	/**
	 * Retrieve the widget name.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'bixola_testimonial_slider';
	}

	/**
	 * Retrieve the widget title.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Testimonial Slider', 'bixola-core' );
	}

	/**
	 * Retrieve the widget icon.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'bixola-icon';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * Note that currently Elementor supports only one category.
	 * When multiple categories passed, Elementor uses the first one.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'bixola_core' ];
	}

	/**
	 * Retrieve the list of scripts the widget depended on.
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'bixola-core' ];
	}

	/**
	 * Register the widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function register_controls() {

        $this->start_controls_section(
            '_content_design_layout',
            [
                'label' => esc_html__( 'Design Layout', 'bixola-core' ),
            ]
        );

        $this->add_control(
            'design_style',
            [
                'label' => esc_html__( 'Style', 'bixola-core' ),
                'type' => Controls_Manager::SELECT,
                'default' => 'layout-1',
                'options' => [
                    'layout-1' => esc_html__( 'Layout 1', 'bixola-core' ),
                    'layout-2'  => esc_html__( 'Layout 2', 'bixola-core' ),
                    'layout-3'  => esc_html__( 'Layout 3', 'bixola-core' ),
                    'layout-4'  => esc_html__( 'Layout 4', 'bixola-core' ),
                ],
            ]
        );

        $this->add_control(
            'shape_switch',
            [
                'label' => esc_html__( 'Shape ON/OFF', 'text-domain' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Show', 'text-domain' ),
                'label_off' => esc_html__( 'Hide', 'text-domain' ),
                'return_value' => 'yes',
                'default' => 'yes',
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3' ],
                ],
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
            '_content_title',
            [
                'label' => esc_html__( 'Title & Content',  'text-domain'  ),
                'tab' => Controls_Manager::TAB_CONTENT,
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-2', 'layout-3' ],
                ],
            ]
        );
        
        $this->add_control(
            'section_subheading',
            [
                'label' => esc_html__( 'Subheading', 'text-domain' ),
                'description' => bixola_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'Testimonials', 'text-domain' ),
                'label_block' => true,
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3' ],
                ],
            ]
        );

        $this->add_control(
            'section_title',
            [
                'label' => esc_html__( 'Title', 'text-domain' ),
                'description' => bixola_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'What our clients <br> says about our <br> best work.', 'text-domain' ),
                'label_block' => true,
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-2', 'layout-3' ],
                ],
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
            'testimonial_list_content',
            [
                'label' => esc_html__( 'Testimonial List', 'bixola-core' ),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $repeater = new Repeater();

        $repeater->add_control(
            'testimonial_image',
            [
                'label' => esc_html__( 'Testimonial Image', 'bixola-core' ),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );

        $repeater->add_control(
            'testimonial_name', [
                'label' => esc_html__( 'Name', 'bixola-core' ),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__( 'Rashal Khan' , 'bixola-core' ),
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'testimonial_designation', [
                'label' => esc_html__( 'Position', 'bixola-core' ),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__( 'Founder' , 'bixola-core' ),
                'label_block' => true,
            ]
        );
        $repeater->add_control(
            'testimonial_description',
            [
                'label' => esc_html__( 'Content', 'bixola-core' ),
                'type' => Controls_Manager::TEXTAREA,
                'rows' => 10,
                'default' => esc_html__( 'Hiring managers are busy people, so you need to make yourout the crowd as quickly as possible. In the first section. This should headline achievements', 'bixola-core' ),
            ]
        );

        $repeater->add_control(
            'testimonial_rating',
            [
                'label' => esc_html__( 'Rating', 'bixola-core' ),
                'type' => Controls_Manager::NUMBER,
                'min' => 1,
                'max' => 5,
                'step' => .5,
                'default' => 5,
            ]
        );

        $this->add_control(
            'testimonial_list',
            [
                'label' => esc_html__( 'Testimonial List', 'bixola-core' ),
                'type' => Controls_Manager::REPEATER,
                'fields' =>  $repeater->get_controls(),
                'default' => [
                    [
                        'testimonial_name' => esc_html__( 'Indigo Violet', 'bixola-core' ),
                        'testimonial_designation' => esc_html__( 'Director, Thump Coffee', 'bixola-core' ),
                        'testimonial_description' => esc_html__( '“Morbi consectetur elementum purus mattis cursus purus metus iaculis sagittis. Vestibulum molestie bibendum turpis luctus sem lacinia quis. Quisque amet velit sit amet dui hendrerit ultricies a id ipsum Mauris sit amet lacinia
                        est”', 'bixola-core' ),
                    ],
                    [
                        'testimonial_name' => esc_html__( 'Michael Thomas', 'bixola-core' ),
                        'testimonial_designation' => esc_html__( 'Director, Plan4Demand', 'bixola-core' ),
                        'testimonial_description' => esc_html__( '“Pellentesque habitant morbi tristique senectus et netus et malesuada fames ac turpis egestas. Donec euismod, sapien ac fringilla tincidunt, eros nisl ultricies justo, a tincidunt eros mi ut velit. Mauris semper, massa eu semper
                        mollis, tortor eros tristique erat, id lacinia lectus quam eu arcu.”', 'bixola-core' ),
                    ],
                    [
                        'testimonial_name' => esc_html__( 'Matthew Martin', 'bixola-core' ),
                        'testimonial_designation' => esc_html__( 'Director, Hobby Lobby', 'bixola-core' ),
                        'testimonial_description' => esc_html__( '“Vivamus sit amet risus vitae leo semper semper. Nullam vel ligula et purus egestas semper. Phasellus ac elit eget quam pulvinar gravida. Sed mattis, nisi vel ullamcorper semper, tortor mauris fringilla sem, a gravida eros nulla
                        sed augue. Donec elementum.”', 'bixola-core' ),
                    ],
                    [
                        'testimonial_name' => esc_html__( 'Julian Walker', 'bixola-core' ),
                        'testimonial_designation' => esc_html__( 'Director, 7Eleven', 'bixola-core' ),
                        'testimonial_description' => esc_html__( '“Vestibulum quis magna sed ligula lacinia vehicula. Nunc ac semper dolor. Donec ut quam eget augue semper iaculis. Vivamus egestas quam erat, eu tincidunt eros ultrices et. Donec iaculis, tellus a semper ultricies, enim tortor
                        luctus nunc, et aliquam quam urna eu quam.”', 'bixola-core' ),
                    ],
                    [
                        'testimonial_name' => esc_html__( 'Henry Baker', 'bixola-core' ),
                        'testimonial_designation' => esc_html__( 'Director, Puzzle Huddle', 'bixola-core' ),
                        'testimonial_description' => esc_html__( '“Sed ac sapien eu enim ultricies faucibus. Nulla facilisi. Nunc et orci id sem interdum congue. Sed ac felis sit amet nisi faucibus bibendum. In hac habitasse platea dictumst.”', 'bixola-core' ),
                    ],

                ],
                'title_field' => '{{{ testimonial_name }}}',
            ]
        );

        $this->end_controls_section();

		$this->start_controls_section(
			'_style_design_layout',
			[
				'label' => __( 'Design Layout', 'bixola-core' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_responsive_control(
            'design_layout_padding',
            [
                'label' => __( 'Padding', 'bixola-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-section' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

		$this->add_responsive_control(
            'design_layout_margin',
            [
                'label' => __( 'Margin', 'bixola-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-section' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'design_layout_background',
            [
                'label' => esc_html__( 'Background', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-section' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            '_heading_style_slider_navigation',
            [
                'label' => esc_html__( 'Arrow / Dots', 'text-domain' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->start_controls_tabs( '_tabs_slider_navigation' );
        
        $this->start_controls_tab(
            'slider_navigation_normal_tab',
            [
                'label' => esc_html__( 'Normal', 'text-domain' ),
            ]
        );

        $this->add_control(
            'slider_navigation_color',
            [
                'label' => esc_html__( 'Color', 'text-domain' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .testimonial-two__carousel.owl-theme .owl-nav .owl-next' => 'color: {{VALUE}}!important',
                    '{{WRAPPER}} .testimonial-two__carousel.owl-theme .owl-nav .owl-prev' => 'color: {{VALUE}}!important',
                    '{{WRAPPER}} .testimonial-one__carousel.owl-theme .owl-nav .owl-next' => 'color: {{VALUE}}!important',
                    '{{WRAPPER}} .testimonial-one__carousel.owl-theme .owl-nav .owl-prev' => 'color: {{VALUE}}!important',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-2' ],
                ],
            ]
        );
        
        $this->add_control(
            'slider_navigation_background',
            [
                'label' => esc_html__( 'Background', 'text-domain' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .carousel-dot-style.owl-carousel .owl-dots .owl-dot' => 'background-color: {{VALUE}}!important',
                    '{{WRAPPER}} .testimonial-three__carousel.owl-carousel .owl-dot' => 'background-color: {{VALUE}}!important',
                    '{{WRAPPER}} .testimonial-two__carousel.owl-theme .owl-nav .owl-next' => 'background-color: {{VALUE}}!important',
                    '{{WRAPPER}} .testimonial-two__carousel.owl-theme .owl-nav .owl-prev' => 'background-color: {{VALUE}}!important',
                    '{{WRAPPER}} .testimonial-one__carousel.owl-theme .owl-nav .owl-next' => 'background-color: {{VALUE}}!important',
                    '{{WRAPPER}} .testimonial-one__carousel.owl-theme .owl-nav .owl-prev' => 'background-color: {{VALUE}}!important',
                ],
            ]
        );
        
        $this->end_controls_tab();
        
        $this->start_controls_tab(
            'slider_navigation_hover_tab',
            [
                'label' => esc_html__( 'Hover', 'text-domain' ),
            ]
        );

        $this->add_control(
            'slider_navigation_color_hover',
            [
                'label' => esc_html__( 'Color', 'text-domain' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .testimonial-two__carousel.owl-theme .owl-nav .owl-next:hover' => 'color: {{VALUE}}!important',
                    '{{WRAPPER}} .testimonial-two__carousel.owl-theme .owl-nav .owl-prev:hover' => 'color: {{VALUE}}!important',
                    '{{WRAPPER}} .testimonial-one__carousel.owl-theme .owl-nav .owl-next:hover' => 'color: {{VALUE}}!important',
                    '{{WRAPPER}} .testimonial-one__carousel.owl-theme .owl-nav .owl-prev:hover' => 'color: {{VALUE}}!important',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-2' ],
                ],
            ]
        );
        
        $this->add_control(
            'slider_navigation_background_hover',
            [
                'label' => esc_html__( 'Background', 'text-domain' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .carousel-dot-style.owl-carousel .owl-dot.active' => 'background-color: {{VALUE}}!important',
                    '{{WRAPPER}} .testimonial-three__carousel.owl-carousel .owl-dot.active' => 'background-color: {{VALUE}}!important',
                    '{{WRAPPER}} .testimonial-two__carousel.owl-theme .owl-nav .owl-next:hover' => 'background-color: {{VALUE}}!important',
                    '{{WRAPPER}} .testimonial-two__carousel.owl-theme .owl-nav .owl-prev:hover' => 'background-color: {{VALUE}}!important',
                    '{{WRAPPER}} .testimonial-one__carousel.owl-theme .owl-nav .owl-next:hover' => 'background-color: {{VALUE}}!important',
                    '{{WRAPPER}} .testimonial-one__carousel.owl-theme .owl-nav .owl-prev:hover' => 'background-color: {{VALUE}}!important',
                ],
            ]
        );
        
        $this->end_controls_tab();
        
        $this->end_controls_tabs();

		$this->end_controls_section();

        $this->start_controls_section(
            '_style_title',
            [
                'label' => esc_html__( 'Title',  'text-domain'  ),
                'tab' => Controls_Manager::TAB_STYLE,
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-2', 'layout-3' ],
                ],
            ]
        );
        
        $this->add_control(
            '_heading_style_section_title',
            [
                'label' => esc_html__( 'Title & Content', 'text-domain' ),
                'type' => Controls_Manager::HEADING,
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-2', 'layout-3' ],
                ],
            ]
        );
        
        $this->add_responsive_control(
            'section_title_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'text-domain' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .testimonial-one__left .section-title__title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .testimonial-two .section__title-two' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .section-title-three__title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-2', 'layout-3' ],
                ],
            ]
        );
        
        $this->add_control(
            'section_title_color',
            [
                'label' => __( 'Color', 'text-domain' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .testimonial-one__left .section-title__title' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .testimonial-two .section__title-two' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .section-title-three__title' => 'color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-2', 'layout-3' ],
                ],
            ]
        );
        
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'section_title_typography',
                'selector' => '{{WRAPPER}} .testimonial-one__left .section-title__title, .testimonial-two .section__title-two, .section-title-three__title',
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-2', 'layout-3' ],
                ],
            ]
        );

        $this->add_control(
            '_heading_style_section_subheading',
            [
                'label' => esc_html__( 'Subheading', 'text-domain' ),
                'type' => Controls_Manager::HEADING,
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3' ],
                ],
            ]
        );
        
        $this->add_responsive_control(
            'section_subheading_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'text-domain' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .testimonial-one__left .section-title__tagline' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3' ],
                ],
            ]
        );

        $this->add_control(
            'section_subheading_border_color',
            [
                'label' => esc_html__( 'Border Color', 'text-domain' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .testimonial-one__left .section-title__tagline:before' => 'background-color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1' ],
                ],
            ]
        );
        
        $this->add_control(
            'section_subheading_color',
            [
                'label' => __( 'Color', 'text-domain' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .testimonial-one__left .section-title__tagline' => 'color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3' ],
                ],
            ]
        );
        
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'section_subheading_typography',
                'selector' => '{{WRAPPER}} .testimonial-one__left .section-title__tagline',
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-3' ],
                ],
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
			'_style_testimonial_list',
			[
				'label' => __( 'Testimonial List', 'bixola-core' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

        $this->add_control(
            '_heading_style_testimonial_name',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Name', 'bixola-core' ),
            ]
        );

        $this->add_responsive_control(
            'testimonial_name_spacing',
            [
                'label' => __( 'Bottom Spacing', 'bixola-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .testimonial-two__client-info h3' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .testimonial-three__client-info h3' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'testimonial_name_border_color',
            [
                'label' => esc_html__( 'Border', 'text-domain' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .testimonial-three__client-info h3::before' => 'background-color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => 'layout-3',
                ],
            ]
        );

        $this->add_control(
            'testimonial_name_color',
            [
                'label' => __( 'Color', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .testimonial-one__client-info h3' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .testimonial-two__client-info h3' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .testimonial-three__client-info h3' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'testimonial_name_typography',
                'selector' => '{{WRAPPER}} .testimonial-one__client-info h3, .testimonial-two__client-info h3, .testimonial-three__client-info h3',
            ]
        );

        $this->add_control(
            '_heading_testimonial_designation',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Designation', 'bixola-core' ),
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'testimonial_designation_color',
            [
                'label' => __( 'Color', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .testimonial-one__client-info p' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .testimonial-two__client-info p' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .testimonial-three__client-info p' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'testimonial_designation_typography',
                'selector' => '.testimonial-one__client-info p, .testimonial-two__client-info p, .testimonial-three__client-info p',
            ]
        );

        $this->add_control(
            '_content_testimonial_description',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Description', 'bixola-core' ),
                'separator' => 'before'
            ]
        );

        $this->add_responsive_control(
            'testimonial_description_spacing',
            [
                'label' => __( 'Bottom Spacing', 'bixola-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .testimonial-one__text' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .testimonial-two__text' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .testimonial-three__text' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'testimonial_description_color',
            [
                'label' => __( 'Color', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .testimonial-one__text' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .testimonial-two__text' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .testimonial-three__text' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'testimonial_description_typography',
                'selector' => '{{WRAPPER}} .testimonial-one__text, .testimonial-two__text, .testimonial-three__text',
            ]
        );

        $this->add_control(
            '_heading_style_testimonial_rating',
            [
                'label' => esc_html__( 'Rating', 'text-domain' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
                'condition' => [
                    'design_style' => [ 'layout-2', 'layout-4' ],
                ],
            ]
        );

        $this->add_control(
            'testimonial_rating_color',
            [
                'label' => esc_html__( 'Color', 'text-domain' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .testimonial-two__ratting span' => 'color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => [ 'layout-2', 'layout-4' ],
                ],
            ]
        );

        $this->add_responsive_control(
            'testimonial_rating_icon_size',
            [
                'label' => esc_html__( 'Icon Size', 'text-domain' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .testimonial-two__ratting span' => 'font-size: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'design_style' => [ 'layout-2', 'layout-4' ],
                ],
            ]
        );

        $this->add_control(
            '_heading_style_testimonial_quote_icon',
            [
                'label' => esc_html__( 'Quote Icon', 'text-domain' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
                'condition' => [
                    'design_style' => [ 'layout-3' ],
                ],
            ]
        );

        $this->add_control(
            'testimonial_quote_icon_color',
            [
                'label' => esc_html__( 'Color', 'text-domain' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .testimonial-three__quote span' => 'color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => [ 'layout-3' ],
                ],
            ]
        );

        $this->add_responsive_control(
            'testimonial_quote_icon_size',
            [
                'label' => esc_html__( 'Icon Size', 'text-domain' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .testimonial-three__quote span' => 'font-size: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'design_style' => [ 'layout-3' ],
                ],
            ]
        );

        $this->add_control(
            '_heading_style_testimonial_list',
            [
                'label' => esc_html__( 'Layout', 'text-domain' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-2', 'layout-4' ],
                ],
            ]
        );

        $this->add_control(
            'testimonial_list_background',
            [
                'label' => esc_html__( 'Background', 'text-domain' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .testimonial-one__single' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .testimonial-two__single' => 'background-color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-2', 'layout-4' ],
                ],
            ]
        );

        $this->add_responsive_control(
            'testimonial_list_padding',
            [
                'label' => esc_html__( 'Padding', 'text-domain' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .testimonial-one__single' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .testimonial-two__single' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-2', 'layout-4' ],
                ],
            ]
        );

        $this->add_responsive_control(
            'testimonial_list_border_radius',
            [
                'label' => esc_html__( 'Border Radius', 'text-domain' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'selectors' => [
                    '{{WRAPPER}} .testimonial-one__single' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .testimonial-two__single' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'condition' => [
                    'design_style' => [ 'layout-1', 'layout-2', 'layout-4' ],
                ],
            ]
        );

		$this->end_controls_section();

	}

	/**
	 * Render the widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		?>

        <?php if ( $settings['design_style']  == 'layout-1' ): ?>

            <section class="bixola-el-section testimonial-one">
                <?php if ( !empty( $settings['shape_switch'] ) ) : ?>
                    <div class="testimonial-one__bg float-bob-x" style="background-image: url(<?php print get_template_directory_uri(); ?>/assets/images/backgrounds/testimonial-one-bg.png);"></div>
                <?php endif; ?>
                <div class="container">
                    <div class="row">
                        <div class="col-xl-5">
                            <div class="testimonial-one__left">
                                <div class="section-title text-left">
                                    <?php if ( !empty( $settings['section_subheading'] ) ) : ?>
                                        <div class="section-title__tagline-box">
                                            <p class="section-title__tagline">
                                                <?php print twinkle_kses($settings['section_subheading']); ?>
                                            </p>
                                        </div>
                                    <?php endif; ?>
                                    <?php if ( !empty( $settings['section_title'] ) ) : ?>
                                        <h2 class="section-title__title">
                                            <?php print twinkle_kses($settings['section_title']); ?>
                                        </h2>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>
                        <div class="col-xl-7">
                            <div class="testimonial-one__right">
                                <div class="testimonial-one__carousel owl-carousel owl-theme thm-owl__carousel">
                                    <?php foreach ($settings['testimonial_list'] as $index => $item) : 
                                        if ( !empty($item['testimonial_image']['url']) ) {
                                            $testimonial_image = !empty($item['testimonial_image']['id']) ? wp_get_attachment_image_url( $item['testimonial_image']['id']) : $item['testimonial_image']['url'];
                                            $testimonial_image_alt = get_post_meta($item["testimonial_image"]["id"], "_wp_attachment_image_alt", true);
                                        }
                                    ?>
                                    <div class="testimonial-one__single">
                                        <?php if ( !empty($item['testimonial_description']) ) : ?>
                                            <p class="testimonial-one__text">
                                                <?php echo twinkle_kses($item['testimonial_description']); ?>
                                            </p>
                                        <?php endif; ?>
                                        <div class="testimonial-one__client-info-and-img">
                                            <?php if ( !empty($testimonial_image) ) : ?>
                                                <div class="testimonial-one__client-img">
                                                    <img src="<?php echo esc_url($testimonial_image); ?>" alt="<?php echo esc_url($testimonial_image_alt); ?>">
                                                </div>
                                            <?php endif; ?>
                                            <div class="testimonial-one__client-info">
                                                <?php if ( !empty($item['testimonial_name']) ) : ?>
                                                    <h3><?php echo twinkle_kses($item['testimonial_name']); ?></h3>
                                                <?php endif; ?>
                                                <?php if ( !empty($item['testimonial_designation']) ) : ?>
                                                    <p><?php echo twinkle_kses($item['testimonial_designation']); ?></p>
                                                <?php endif; ?>
                                            </div>
                                        </div>
                                    </div>
                                    <?php endforeach; ?>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </section>

        <?php elseif ( $settings['design_style']  == 'layout-2' ): 
                if ( !empty($settings['testimonial_image']['url']) ) {
                    $bixola_testimonial_image = !empty($settings['testimonial_image']['id']) ? wp_get_attachment_image_url( $settings['testimonial_image']['id']) : $settings['testimonial_image']['url'];
                    $bixola_testimonial_image_alt = get_post_meta($settings["testimonial_image"]["id"], "_wp_attachment_image_alt", true);
                }
            ?>

                <section class="bixola-el-section testimonial-two">
                    <div class="container">
                        <?php if ( !empty( $settings['section_title'] ) ) : ?>
                            <h3 class="section__title-two">
                                <?php print twinkle_kses($settings['section_title']); ?>
                            </h3>
                        <?php endif; ?>
                        <div class="testimonial-two__bottom">
                            <div class="testimonial-two__carousel owl-carousel owl-theme thm-owl__carousel">
                                <?php foreach ($settings['testimonial_list'] as $index => $item) : 
                                    if ( !empty($item['testimonial_image']['url']) ) {
                                        $testimonial_image = !empty($item['testimonial_image']['id']) ? wp_get_attachment_image_url( $item['testimonial_image']['id']) : $item['testimonial_image']['url'];
                                        $testimonial_image_alt = get_post_meta($item["testimonial_image"]["id"], "_wp_attachment_image_alt", true);
                                    }
                                ?>
                                <div class="testimonial-two__single">
                                    <?php if ( !empty($testimonial_image) ) : ?>
                                        <div class="testimonial-two__client-img">
                                            <img src="<?php echo esc_url($testimonial_image); ?>" alt="<?php echo esc_url($testimonial_image_alt); ?>">
                                        </div>
                                    <?php endif; ?>
                                    <div class="testimonial-two__client-info">
                                        <?php if ( !empty($item['testimonial_name']) ) : ?>
                                            <h3><?php echo twinkle_kses($item['testimonial_name']); ?></h3>
                                        <?php endif; ?>
                                        <?php if ( !empty($item['testimonial_designation']) ) : ?>
                                            <p><?php echo twinkle_kses($item['testimonial_designation']); ?></p>
                                        <?php endif; ?>
                                    </div>
                                    <?php if ( !empty($item['testimonial_description']) ) : ?>
                                        <p class="testimonial-two__text">
                                            <?php echo twinkle_kses($item['testimonial_description']); ?>
                                        </p>
                                    <?php endif; ?>
                                    <div class="testimonial-two__ratting">
                                        <?php 
                                        $testimonial_rating = $item['testimonial_rating'];

                                        // Calculate full stars, half stars, and empty stars
                                        $full_stars = floor($testimonial_rating);
                                        $half_star = ($testimonial_rating - $full_stars >= 0.5) ? 1 : 0;
                                        $empty_stars = 5 - ($full_stars + $half_star);
                                        for ($i = 1; $i <= $full_stars; $i++): ?>
                                            <li><span class="fas fa-star"></span></li>
                                        <?php endfor; ?>

                                        <?php if ($half_star): ?>
                                            <li><span class="fas fa-star-half-alt"></span></li>
                                        <?php endif; ?>

                                        <?php for ($i = 1; $i <= $empty_stars; $i++): ?>
                                            <li><span class="far fa-star"></span></li>
                                        <?php endfor; ?>
                                    </div>
                                </div>
                                <?php endforeach; ?>
                            </div>
                        </div>
                    </div>
                </section>

            <?php elseif ( $settings['design_style']  == 'layout-3' ) : ?>

                <section class="bixola-el-section testimonial-three">
                    <?php if ( !empty( $settings['shape_switch'] ) ) : ?>
                        <div class="testimonial-three__shape-1 float-bob-x">
                            <img src="<?php print get_template_directory_uri(); ?>/assets/images/shapes/testimonial-three-shape-1.png" alt="<?php print esc_attr( 'shape', 'bixola-core' ); ?>">
                        </div>
                        <div class="testimonial-three__shape-2 float-bob-x">
                            <img src="<?php print get_template_directory_uri(); ?>/assets/images/shapes/testimonial-three-shape-2.png" alt="<?php print esc_attr( 'shape', 'bixola-core' ); ?>">
                        </div>
                    <?php endif; ?>
                    
                    <div class="container">
                        <div class="section-title-three text-center">
                            <?php if ( !empty( $settings['section_subheading'] ) ) : ?>
                                <div class="section-title-three__tagline-box">
                                    <p class="section-title-three__tagline">
                                        <?php print twinkle_kses($settings['section_subheading']); ?>
                                    </p>
                                </div>
                            <?php endif; ?>
                            <?php if ( !empty( $settings['section_title'] ) ) : ?>
                                <h2 class="section-title-three__title">
                                    <?php print twinkle_kses($settings['section_title']); ?>
                                </h2>
                            <?php endif; ?>
                        </div>
                        <div class="testimonial-three__bottom">
                            <div class="testimonial-three__carousel owl-carousel owl-theme thm-owl__carousel">
                                <?php foreach ($settings['testimonial_list'] as $index => $item) : 
                                    if ( !empty($item['testimonial_image']['url']) ) {
                                        $bixola_testimonial_image = !empty($item['testimonial_image']['id']) ? wp_get_attachment_image_url( $item['testimonial_image']['id']) : $item['testimonial_image']['url'];
                                        $bixola_testimonial_image_alt = get_post_meta($item["testimonial_image"]["id"], "_wp_attachment_image_alt", true);
                                    }
                                ?>
                                <div class="testimonial-three__single">
                                    <div class="testimonial-three__quote">
                                        <span class="icon-quote11"></span>
                                    </div>
                                    <?php if ( !empty($item['testimonial_description']) ) : ?>
                                        <p class="testimonial-three__text">
                                            <?php echo twinkle_kses($item['testimonial_description']); ?>
                                        </p>
                                    <?php endif; ?>
                                    <div class="testimonial-three__client-info">
                                        <?php if ( !empty($item['testimonial_name']) ) : ?>
                                            <h3><?php echo twinkle_kses($item['testimonial_name']); ?></h3>
                                        <?php endif; ?>
                                        <?php if ( !empty($item['testimonial_designation']) ) : ?>
                                            <p><?php echo twinkle_kses($item['testimonial_designation']); ?></p>
                                        <?php endif; ?>
                                    </div>
                                </div>
                                <?php endforeach; ?>
                            </div>
                        </div>
                    </div>
                </section>

            <?php elseif ( $settings['design_style']  == 'layout-4' ) : ?>

                <section class="bixola-el-section testimonials-carousel-page">
                    <div class="container">
                        <div class="testimonials-carousel-style thm-owl__carousel owl-theme owl-carousel carousel-dot-style">
                            <?php foreach ($settings['testimonial_list'] as $index => $item) : 
                                if ( !empty($item['testimonial_image']['url']) ) {
                                    $bixola_testimonial_image = !empty($item['testimonial_image']['id']) ? wp_get_attachment_image_url( $item['testimonial_image']['id']) : $item['testimonial_image']['url'];
                                    $bixola_testimonial_image_alt = get_post_meta($item["testimonial_image"]["id"], "_wp_attachment_image_alt", true);
                                }
                            ?>
                            <div class="item">
                                <div class="testimonial-two__single">
                                    <div class="testimonial-two__client-img">
                                        <img src="<?php echo esc_url($bixola_testimonial_image); ?>" alt="<?php echo esc_url($bixola_testimonial_image_alt); ?>">
                                    </div>
                                    <div class="testimonial-two__client-info">
                                        <?php if ( !empty($item['testimonial_name']) ) : ?>
                                            <h3><?php print twinkle_kses($item['testimonial_name']); ?></h3>
                                        <?php endif; ?>
                                        <?php if ( !empty($item['testimonial_designation']) ) : ?>
                                            <p><?php echo twinkle_kses($item['testimonial_designation']); ?></p>
                                        <?php endif; ?>
                                    </div>
                                    <?php if ( !empty($item['testimonial_description']) ) : ?>
                                        <p class="testimonial-two__text">
                                            <?php echo twinkle_kses($item['testimonial_description']); ?>
                                        </p>
                                    <?php endif; ?>
                                    <div class="testimonial-two__ratting">
                                        <?php 
                                        $testimonial_rating = $item['testimonial_rating'];

                                        // Calculate full stars, half stars, and empty stars
                                        $full_stars = floor($testimonial_rating);
                                        $half_star = ($testimonial_rating - $full_stars >= 0.5) ? 1 : 0;
                                        $empty_stars = 5 - ($full_stars + $half_star);
                                        for ($i = 1; $i <= $full_stars; $i++): ?>
                                            <li><span class="fas fa-star"></span></li>
                                        <?php endfor; ?>

                                        <?php if ($half_star): ?>
                                            <li><span class="fas fa-star-half-alt"></span></li>
                                        <?php endif; ?>

                                        <?php for ($i = 1; $i <= $empty_stars; $i++): ?>
                                            <li><span class="far fa-star"></span></li>
                                        <?php endfor; ?>
                                    </div>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </section>
        
            <?php endif; ?>

        <?php 
	}
}

$widgets_manager->register( new Bixola_Testimonial_Slider() );