<?php
namespace BixolaCore\Widgets;

use \Elementor\Widget_Base;
use \Elementor\Control_Media;
use \Elementor\Controls_Manager;
use \Elementor\Group_Control_Border;
use \Elementor\Group_Control_Typography;
use \Elementor\Repeater;
use \Elementor\Utils;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Bixola Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class Bixola_Portfolio_Info extends Widget_Base {

	/**
	 * Retrieve the widget name.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'bixola_portfolio_info';
	}

	/**
	 * Retrieve the widget title.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Portfolio Info', 'bixola-core' );
	}

	/**
	 * Retrieve the widget icon.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'bixola-icon';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * Note that currently Elementor supports only one category.
	 * When multiple categories passed, Elementor uses the first one.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'bixola_core' ];
	}

	/**
	 * Retrieve the list of scripts the widget depended on.
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'bixola-core' ];
	}

	/**
	 * Register the widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function register_controls() {

        $this->start_controls_section(
            '_content_portfolio_title',
            [
                'label' => esc_html__( 'Title & Content', 'bixola-core' ),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'bixola_image',
            [
                'label' => esc_html__( 'Testimonial Image', 'bixola-core' ),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );

        $this->add_control(
            'section_subheading',
            [
                'label' => esc_html__( 'Subheading', 'text-domain' ),
                'description' => bixola_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'Business Audit', 'text-domain' ),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'section_title',
            [
                'label' => esc_html__( 'Title', 'text-domain' ),
                'description' => bixola_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'Our Business Growth', 'text-domain' ),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'section_description',
            [
                'label' => esc_html__( 'Description', 'text-domain' ),
                'description' => bixola_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua. Ut enim ad minim veniam, quis nostrud exercitation ullamco laboris nisi ut aliquip ex ea commodo consequat. Duis aute irure dolor in reprehenderit in voluptate velit esse cillum dolore eu fugiat nulla pariatur. Excepteur sint occaecat cupidatat non proident, sunt in culpa qui officia deserunt mollit anim id est laborum.', 'text-domain' ),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'section_description_two',
            [
                'label' => esc_html__( 'Description', 'text-domain' ),
                'description' => bixola_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua. Ut enim ad minim veniam, quis nostrud exercitation ullamco laboris.', 'text-domain' ),
                'label_block' => true,
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_content_portfolio_info_list',
            [
                'label' => esc_html__( 'Portfolio Info List',  'text-domain'  ),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'portfolio_info_list_title',
            [
                'label' => esc_html__( 'Title', 'text-domain' ),
                'description' => bixola_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'Project Information', 'text-domain' ),
                'label_block' => true,
            ]
        );
        
        $repeater = new Repeater();

        $repeater->add_control(
            'portfolio_info_icon_type',
            [
                'label' => esc_html__( 'Image Type', 'text-domain' ),
                'type' => Controls_Manager::SELECT,
                'default' => 'icon',
                'options' => [
                    'icon' => esc_html__( 'Icon', 'text-domain' ),
                    'image' => esc_html__( 'Image', 'text-domain' ),
                    'svg' => esc_html__( 'SVG', 'text-domain' ),
                ],
            ]
        );
        
        $repeater->add_control(
            'portfolio_info_image_icon',
            [
                'label' => esc_html__( 'Upload Image', 'text-domain' ),
                'type' => Controls_Manager::MEDIA,
                'condition' => [
                    'portfolio_info_icon_type' => 'image',
                ],
            ]
        );
        
        $repeater->add_control(
            'portfolio_info_icon',
            [
                'label' => esc_html__( 'Icon', 'text-domain' ),
                'type' => Controls_Manager::ICONS,
                'fa4compatibility' => 'icon',
                'label_block' => true,
                'condition' => [
                    'portfolio_info_icon_type' => 'icon',
                ],
            ]
        );
        
        $repeater->add_control(
            'portfolio_info_svg_icon',
            [
                'label' => esc_html__( 'SVG Icon', 'text-domain' ),
                'type' => Controls_Manager::TEXTAREA,
                'rows' => 10,
                'condition' => [
                    'portfolio_info_icon_type' => 'svg',
                ],
            ]
        );

        $repeater->add_control(
            'portfolio_info_label', [
                'label' => esc_html__('Label', 'bixola-core'),
                'description' => bixola_get_allowed_html_desc( 'basic' ),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'portfolio_info_title', [
                'label' => esc_html__('Text', 'bixola-core'),
                'description' => bixola_get_allowed_html_desc( 'basic' ),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
            ]
        );

        $this->add_control(
            'bixola_portfolio_info_list',
            [
                'label' => esc_html__('Info List', 'bixola-core'),
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'portfolio_info_label' => __( 'Clients:', 'bixola-core' ),
                        'portfolio_info_title' => __( 'Damien Wilson', 'bixola-core' ),
                        'portfolio_info_icon' => 'icon-user-2',
                        'portfolio_info_icon_type' => 'icon',
                    ],
                    [
                        'portfolio_info_label' => __( 'Category:', 'bixola-core' ),
                        'portfolio_info_title' => __( 'Business Audit', 'bixola-core' ),
                        'portfolio_info_icon' => 'icon-category',
                        'portfolio_info_icon_type' => 'icon',
                    ],
                    [
                        'portfolio_info_label' => __( 'Date:', 'bixola-core' ),
                        'portfolio_info_title' => __( '25 March, 2023', 'bixola-core' ),
                        'portfolio_info_icon' => 'icon-date11',
                        'portfolio_info_icon_type' => 'icon',
                    ],
                    [
                        'portfolio_info_label' => __( 'Location:', 'bixola-core' ),
                        'portfolio_info_title' => __( 'New York, NY, USA', 'bixola-core' ),
                        'portfolio_info_icon' => 'icon-location11',
                        'portfolio_info_icon_type' => 'icon',
                    ],
                    [
                        'portfolio_info_label' => __( 'Website:', 'bixola-core' ),
                        'portfolio_info_title' => __( 'https://busicon.com', 'bixola-core' ),
                        'portfolio_info_icon' => 'icon-website1',
                        'portfolio_info_icon_type' => 'icon',
                    ],
                ],
                'title_field' => '{{{ portfolio_info_label }}}',
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
            '_content_navigation',
            [
                'label' => esc_html__( 'Navigation',  'text-domain'  ),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );
        
        $this->add_control(
            '_heading_content_navigation_previous',
            [
                'label' => esc_html__( 'Previous', 'text-domain' ),
                'type' => Controls_Manager::HEADING,
            ]
        );

        $this->add_control(
            'navigation_previous',
            [
                'label' => esc_html__( 'Previous', 'text-domain' ),
                'description' => bixola_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'Prev', 'text-domain' ),
                'label_block' => true,
            ]
        );
        
        $this->add_control(
            'navigation_previous_button_link_type',
            [
                'label' => esc_html__( 'Button Link Type', 'text-domain' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '1' => 'Custom Link',
                    '2' => 'Internal Page',
                ],
                'default' => '1',
                'label_block' => true,
            ]
        );
        
        $this->add_control(
            'navigation_previous_button_link',
            [
                'label' => esc_html__( 'Button link', 'text-domain' ),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => esc_html__('https://your-link.com', 'text-domain'),
                'show_external' => false,
                'default' => [
                    'url' => '#',
                    'is_external' => true,
                    'nofollow' => true,
                    'custom_attributes' => '',
                ],
                'condition' => [
                    'navigation_previous_button_link_type' => '1',
                ],
                'label_block' => true,
            ]
        );

        $this->add_control(
            'navigation_previous_button_page_link',
            [
                'label' => esc_html__( 'Select Button Page', 'text-domain' ),
                'type' => Controls_Manager::SELECT2,
                'label_block' => true,
                'options' => bixola_get_all_pages(),
                'condition' => [
                    'navigation_previous_button_link_type' => '2',
                ]
            ]
        );

        $this->add_control(
            '_heading_content_navigation_next',
            [
                'label' => esc_html__( 'Next', 'text-domain' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'navigation_next',
            [
                'label' => esc_html__( 'Next', 'text-domain' ),
                'description' => bixola_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'Next', 'text-domain' ),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'navigation_next_button_link_type',
            [
                'label' => esc_html__( 'Button Link Type', 'text-domain' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '1' => 'Custom Link',
                    '2' => 'Internal Page',
                ],
                'default' => '1',
                'label_block' => true,
            ]
        );
        
        $this->add_control(
            'navigation_next_button_link',
            [
                'label' => esc_html__( 'Button link', 'text-domain' ),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => esc_html__('https://your-link.com', 'text-domain'),
                'show_external' => false,
                'default' => [
                    'url' => '#',
                    'is_external' => true,
                    'nofollow' => true,
                    'custom_attributes' => '',
                ],
                'condition' => [
                    'navigation_next_button_link_type' => '1',
                ],
                'label_block' => true,
            ]
        );
        
        $this->add_control(
            'navigation_next_button_page_link',
            [
                'label' => esc_html__( 'Select Button Page', 'text-domain' ),
                'type' => Controls_Manager::SELECT2,
                'label_block' => true,
                'options' => bixola_get_all_pages(),
                'condition' => [
                    'navigation_next_button_link_type' => '2',
                ]
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
            '_style_design_layout',
            [
                'label' => __( 'Design Layout', 'bixola-core' ),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'design_style_margin',
            [
                'label' => esc_html__( 'Margin', 'text-domain' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em' ],
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-section' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'design_layout_padding',
            [
                'label' => __( 'Padding', 'bixola-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-section' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'design_layout_background',
            [
                'label' => esc_html__( 'Background', 'text-domain' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-section' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_style_title',
            [
                'label' => esc_html__( 'Title & Content',  'text-domain'  ),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );
        
        $this->add_control(
            '_heading_style_section_subheading',
            [
                'label' => esc_html__( 'Subheading', 'text-domain' ),
                'type' => Controls_Manager::HEADING,
            ]
        );
        
        $this->add_responsive_control(
            'section_subheading_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'text-domain' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .portfolio-details__sub-title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        
        $this->add_control(
            'section_subheading_color',
            [
                'label' => __( 'Color', 'text-domain' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio-details__sub-title' => 'color: {{VALUE}}',
                ],
            ]
        );
        
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'section_subheading_typography',
                'selector' => '{{WRAPPER}} .portfolio-details__sub-title',
            ]
        );

        $this->add_control(
            '_heading_style_section_title',
            [
                'label' => esc_html__( 'Title', 'text-domain' ),
                'type' => Controls_Manager::HEADING,
            ]
        );
        
        $this->add_responsive_control(
            'section_title_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'text-domain' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .portfolio-details__title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        
        $this->add_control(
            'section_title_color',
            [
                'label' => __( 'Color', 'text-domain' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio-details__title' => 'color: {{VALUE}}',
                ],
            ]
        );
        
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'section_title_typography',
                'selector' => '{{WRAPPER}} .portfolio-details__title',
            ]
        );

        $this->add_control(
            '_heading_style_section_description',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Description', 'text-domain' ),
                'separator' => 'before',
            ]
        );
        
        $this->add_responsive_control(
            'section_description_spacing',
            [
                'label' => __( 'Bottom Spacing', 'text-domain' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .portfolio-details__text-1' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        
        $this->add_control(
            'section_description_color',
            [
                'label' => __( 'Color', 'text-domain' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio-details__text-1' => 'color: {{VALUE}}',
                ],
            ]
        );
        
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'section_description_typography',
                'selector' => '{{WRAPPER}} .portfolio-details__text-1',
            ]
        );

        $this->add_control(
            '_heading_style_section_description_two',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Description Two', 'text-domain' ),
                'separator' => 'before',
            ]
        );
        
        $this->add_responsive_control(
            'section_description_two_spacing',
            [
                'label' => __( 'Bottom Spacing', 'text-domain' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .portfolio-details__text-2' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        
        $this->add_control(
            'section_description_two_color',
            [
                'label' => __( 'Color', 'text-domain' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio-details__text-2' => 'color: {{VALUE}}',
                ],
            ]
        );
        
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'section_description_two_typography',
                'selector' => '{{WRAPPER}} .portfolio-details__text-2',
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
            '_style_portfolio_info_list',
            [
                'label' => __( 'Portfolio Info List', 'bixola-core' ),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            '_heading_style_portfolio_info_title',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Title', 'bixola-core' ),
            ]
        );

        $this->add_control(
            'portfolio_info_list_color',
            [
                'label' => esc_html__( 'Color', 'text-domain' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio-details__information-title' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'portfolio_info_list_typography',
                'selector' => '{{WRAPPER}} .portfolio-details__information-title',
            ]
        );

        $this->add_control(
            '_heading_style_portfolio_info_list_icon',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Icon', 'bixola-core' ),
                'separator' => 'before',
            ]
        );

        $this->start_controls_tabs( '_tabs_portfolio_info_list_icon' );
        
        $this->start_controls_tab(
            'portfolio_info_list_icon_normal_tab',
            [
                'label' => esc_html__( 'Normal', 'text-domain' ),
            ]
        );
        
        $this->add_control(
            'portfolio_info_list_icon_color',
            [
                'label' => esc_html__( 'Color', 'text-domain' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio-details__information li .icon span' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'portfolio_info_list_icon_background',
            [
                'label' => esc_html__( 'Background', 'text-domain' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio-details__information li .icon' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        
        $this->end_controls_tab();
        
        $this->start_controls_tab(
            'portfolio_info_list_icon_hover_tab',
            [
                'label' => esc_html__( 'Hover', 'text-domain' ),
            ]
        );
        
        $this->add_control(
            'portfolio_info_list_icon_color_hover',
            [
                'label' => esc_html__( 'Color', 'text-domain' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio-details__information li .icon:hover span' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'portfolio_info_list_icon_background_hover',
            [
                'label' => esc_html__( 'Background', 'text-domain' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio-details__information li .icon:hover' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        
        $this->end_controls_tab();
        
        $this->end_controls_tabs();

        $this->add_control(
            '_heading_style_portfolio_info_list_label',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Label', 'bixola-core' ),
                'separator' => 'before'
            ]
        );

        $this->add_responsive_control(
            'portfolio_info_list_bottom_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'text-domain' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .portfolio-details__information li .content p' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'portfolio_info_list_label_color',
            [
                'label' => __( 'Text Color', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio-details__information li .content p' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'portfolio_info_list_label_typography',
                'selector' => '{{WRAPPER}} .portfolio-details__information li .content p',
            ]
        );

        $this->add_control(
            '_heading_style_portfolio_info_list_title',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Title', 'bixola-core' ),
                'separator' => 'before'
            ]
        );

        $this->add_control(
            'portfolio_info_list_title_color',
            [
                'label' => __( 'Text Color', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio-details__information li .content h4' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'portfolio_info_list_title_typography',
                'selector' => '{{WRAPPER}} .portfolio-details__information li .content h4',
            ]
        );

        $this->add_control(
            '_heading_style_portfolio_info_list_layout',
            [
                'label' => esc_html__( 'Layout', 'text-domain' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'portfolio_info_list_layout_border_color',
            [
                'label' => esc_html__( 'Border Color', 'text-domain' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio-details__information-title' => 'border-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'portfolio_info_list_layout_background',
            [
                'label' => esc_html__( 'Background', 'text-domain' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio-details__information-box' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_responsive_control(
            'portfolio_info_list_layout_padding',
            [
                'label' => esc_html__( 'Padding', 'text-domain' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .portfolio-details__information-title' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .portfolio-details__information' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'portfolio_info_list_layout_border_radius',
            [
                'label' => esc_html__( 'Border Radius', 'text-domain' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .portfolio-details__information-box' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_style_navigation',
            [
                'label' => esc_html__( 'Navigation',  'text-domain'  ),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'navigation_separator_color',
            [
                'label' => esc_html__( 'Separator Color', 'text-domain' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio-details__pagination-box' => 'border-color: {{VALUE}}',
                ],
            ]
        );
        
        $this->start_controls_tabs( '_tabs_navigation' );
        
        $this->start_controls_tab(
            'navigation_normal_tab',
            [
                'label' => esc_html__( 'Normal', 'text-domain' ),
            ]
        );
        
        $this->add_control(
            'navigation_color',
            [
                'label' => esc_html__( 'Color', 'text-domain' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio-details__pagination li a' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'navigation_background',
            [
                'label' => esc_html__( 'Background', 'text-domain' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio-details__pagination li a' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name' => 'navigation_border',
                'selector' => '{{WRAPPER}} .portfolio-details__pagination li a',
            ]
        );
        
        $this->end_controls_tab();
        
        $this->start_controls_tab(
            'navigation_hover_tab',
            [
                'label' => esc_html__( 'Hover', 'text-domain' ),
            ]
        );
        
        $this->add_control(
            'navigation_color_hover',
            [
                'label' => esc_html__( 'Color', 'text-domain' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio-details__pagination li a:hover' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'navigation_background_hover',
            [
                'label' => esc_html__( 'Background', 'text-domain' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio-details__pagination li a:hover' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name' => 'navigation_border_hover',
                'selector' => '{{WRAPPER}} .portfolio-details__pagination li a:hover',
            ]
        );
        
        $this->end_controls_tab();
        
        $this->end_controls_tabs();

        $this->add_responsive_control(
            'navigation_border_radius',
            [
                'label' => esc_html__( 'Border Radius', 'text-domain' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .portfolio-details__pagination li a' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'navigation_margin',
            [
                'label' => esc_html__( 'Margin', 'text-domain' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .portfolio-details__pagination li a' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'navigation_typography',
                'selector' => '{{WRAPPER}} .portfolio-details__pagination li a',
            ]
        );
        
        $this->end_controls_section();
	}

	/**
	 * Render the widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

        if ( !empty($settings['bixola_image']['url']) ) {
            $bixola_image = !empty($settings['bixola_image']['id']) ? wp_get_attachment_image_url( $settings['bixola_image']['id'], 'full') : $settings['bixola_image']['url'];
            $bixola_image_alt = get_post_meta($settings["bixola_image"]["id"], "_wp_attachment_image_alt", true);
        }

        if ('2' == $settings['navigation_previous_button_link_type']) {
            $this->add_render_attribute('navigation-previous-button-arg', 'href', get_permalink($settings['navigation_previous_button_page_link']));
            $this->add_render_attribute('navigation-previous-button-arg', 'target', '_self');
            $this->add_render_attribute('navigation-previous-button-arg', 'rel', 'nofollow');
        } else {
            if ( ! empty( $settings['_button_link']['url'] ) ) {
                $this->add_link_attributes( 'navigation-previous-button-arg', $settings['navigation_previous_button_link'] );
            }
        }

        if ('2' == $settings['navigation_next_button_link_type']) {
            $this->add_render_attribute('navigation-next-button-arg', 'href', get_permalink($settings['navigation_next_button_page_link']));
            $this->add_render_attribute('navigation-next-button-arg', 'target', '_self');
            $this->add_render_attribute('navigation-next-button-arg', 'rel', 'nofollow');
        } else {
            if ( ! empty( $settings['_button_link']['url'] ) ) {
                $this->add_link_attributes( 'navigation-next-button-arg', $settings['navigation_next_button_link'] );
            }
        }

		?>

            <section class="bixola-el-section portfolio-details">
                <div class="container">
                    <div class="row">
                        <div class="col-xl-12">
                            <div class="portfolio-details__img">
                                <img src="<?php echo esc_url($bixola_image); ?>" alt="<?php echo esc_url($bixola_image_alt); ?>">
                            </div>
                        </div>
                        <div class="portfolio-details__bottom">
                            <div class="row">
                                <div class="col-xl-8 col-lg-7">
                                    <div class="portfolio-details__left">
                                        <?php if ( !empty( $settings['section_subheading'] ) ) : ?>
                                            <p class="portfolio-details__sub-title">
                                                <?php print twinkle_kses($settings['section_subheading']); ?>
                                            </p>
                                        <?php endif; ?>
                                        <?php if ( !empty( $settings['section_title'] ) ) : ?>
                                            <h3 class="portfolio-details__title">
                                                <?php print twinkle_kses($settings['section_title']); ?>
                                            </h3>
                                        <?php endif; ?>
                                        <?php if ( !empty( $settings['section_description'] ) ) : ?>
                                            <p class="portfolio-details__text-1">
                                                <?php print twinkle_kses($settings['section_description']); ?>
                                            </p>
                                        <?php endif; ?>
                                        <?php if ( !empty( $settings['section_description_two'] ) ) : ?>
                                            <p class="portfolio-details__text-2">
                                                <?php print twinkle_kses($settings['section_description_two']); ?>
                                            </p>
                                        <?php endif; ?>
                                        <div class="portfolio-details__pagination-box">
                                            <ul class="portfolio-details__pagination list-unstyled clearfix">
                                                <?php if ( !empty( $settings['navigation_previous'] ) ) : ?>
                                                    <li class="next">
                                                        <a <?php echo $this->get_render_attribute_string( 'navigation-previous-button-arg' ); ?> aria-label="Previous">
                                                            <i class="icon-right-arrow"></i><?php print twinkle_kses($settings['navigation_previous']); ?>
                                                        </a>
                                                    </li>
                                                <?php endif; ?>
                                                <?php if ( !empty( $settings['navigation_next'] ) ) : ?>
                                                    <li class="previous">
                                                        <a <?php echo $this->get_render_attribute_string( 'navigation-next-button-arg' ); ?> aria-label="Next">
                                                            <?php print twinkle_kses($settings['navigation_next']); ?><i class="icon-right-arrow1"></i>
                                                        </a>
                                                    </li>
                                                <?php endif; ?>
                                            </ul>
                                        </div>
                                    </div>
                                </div>
                                <div class="col-xl-4 col-lg-5">
                                    <div class="portfolio-details__right">
                                        <div class="portfolio-details__information-box">
                                            <?php if ( !empty( $item['portfolio_info_list_title' ] ) ) : ?>
                                                <h3 class="portfolio-details__information-title">
                                                    <?php echo twinkle_kses($item['portfolio_info_list_title' ]); ?>
                                                </h3>
                                            <?php endif; ?>
                                            <ul class="portfolio-details__information list-unstyled">
                                                <?php foreach ($settings['bixola_portfolio_info_list'] as $item) : ?>
                                                    <?php if ( !empty($item['portfolio_info_label']) ) : ?>
                                                        <li>
                                                            <?php if ( $item['portfolio_info_icon_type']  == 'image' ): ?>
                                                                <div class="icon">
                                                                    <img class="img-fluid" src="<?php echo $item['portfolio_info_image_icon']['url']; ?>" alt="<?php echo get_post_meta(attachment_url_to_postid($item['portfolio_info_image_icon']['url']), '_wp_attachment_image_alt', true); ?>">
                                                                </div>
                                                            <?php elseif ( $item['portfolio_info_icon_type']  == 'icon' ): ?>
                                                                <div class="icon">
                                                                    <?php bixola_render_icon($item, 'portfolio_info_icon' ); ?>
                                                                </div>
                                                            <?php elseif ( $item['portfolio_info_icon_type']  == 'svg' ): ?>
                                                                <div class="icon">
                                                                    <?php print $item['project_info_svg_icon']; ?>
                                                                </div>
                                                            <?php endif; ?>
                                                            <div class="content">
                                                                <p><?php echo twinkle_kses($item['portfolio_info_label']);?></p>
                                                                <h4><?php echo twinkle_kses($item['portfolio_info_title' ]); ?></h4>
                                                            </div>
                                                        </li>
                                                    <?php endif; ?>
                                                <?php endforeach; ?>
                                            </ul>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </section>

        <?php 
	}
}

$widgets_manager->register( new Bixola_Portfolio_Info() );