<?php
namespace BixolaCore\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use \Elementor\Utils;
use \Elementor\Control_Media;

use \Elementor\Group_Control_Border;
use \Elementor\Group_Control_Box_Shadow;
use \Elementor\Group_Control_Text_Shadow;
use \Elementor\Group_Control_Typography;
Use \Elementor\Core\Schemes\Typography;
use \Elementor\Group_Control_Background;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Bixola Core
 * @since 1.0.0
 */
class Bixola_Heading extends Widget_Base {

	/**
	 * Retrieve the widget name.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'bixola_heading';
	}

	/**
	 * Retrieve the widget title.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Heading', 'bixola-core' );
	}

	/**
	 * Retrieve the widget icon.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'bixola-icon';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * Note that currently Elementor supports only one category.
	 * When multiple categories passed, Elementor uses the first one.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'bixola_core' ];
	}

	/**
	 * Retrieve the list of scripts the widget depended on.
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'bixola-core' ];
	}

	/**
	 * Register the widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function register_controls() {

        $this->start_controls_section(
            '_content_design_layout',
            [
                'label' => esc_html__('Design Layout', 'bixola-core'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'design_style',
            [
                'label' => esc_html__('Select Layout', 'bixola-core'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'layout-1' => esc_html__('Layout 1', 'bixola-core'),
                    'layout-2' => esc_html__('Layout 2', 'bixola-core'),
                    'layout-3' => esc_html__('Layout 3', 'bixola-core'),
                ],
                'default' => 'layout-1',
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_content_title',
            [
                'label' => esc_html__('Title & Content', 'bixola-core'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'section_subheading',
            [
                'label' => esc_html__('Subheading', 'bixola-core'),
                'description' => bixola_get_allowed_html_desc( 'basic' ),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Services', 'bixola-core'),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'section_title',
            [
                'label' => esc_html__('Title', 'bixola-core'),
                'description' => bixola_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'default' => __('What services we provide to our <br> customers business', 'bixola-core'),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'section_title_tag',
            [
                'label' => esc_html__('Title HTML Tag', 'bixola-core'),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'h1' => [
                        'title' => esc_html__('H1', 'bixola-core'),
                        'icon' => 'eicon-editor-h1'
                    ],
                    'h2' => [
                        'title' => esc_html__('H2', 'bixola-core'),
                        'icon' => 'eicon-editor-h2'
                    ],
                    'h3' => [
                        'title' => esc_html__('H3', 'bixola-core'),
                        'icon' => 'eicon-editor-h3'
                    ],
                    'h4' => [
                        'title' => esc_html__('H4', 'bixola-core'),
                        'icon' => 'eicon-editor-h4'
                    ],
                    'h5' => [
                        'title' => esc_html__('H5', 'bixola-core'),
                        'icon' => 'eicon-editor-h5'
                    ],
                    'h6' => [
                        'title' => esc_html__('H6', 'bixola-core'),
                        'icon' => 'eicon-editor-h6'
                    ]
                ],
                'default' => 'h2',
                'toggle' => false,
            ]
        );

        $this->add_control(
            'section_description',
            [
                'label' => esc_html__('Description', 'bixola-core'),
                'description' => bixola_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXTAREA,
                'default' => __('It is a long established fact that a reader will be distracted by the readable content of a page when looking at its layout.', 'bixola-core'),
                'label_block' => true,
                'condition' => [
                    'design_style' => 'layout-2',
                ],
            ]
        );

        $this->add_responsive_control(
			'section_title_align',
			[
				'type' => Controls_Manager::CHOOSE,
				'label' => esc_html__( 'Alignment', 'bixola-core' ),
				'options' => [
					'left' => [
						'title' => esc_html__( 'Left', 'bixola-core' ),
						'icon' => 'eicon-text-align-left',
					],
					'center' => [
						'title' => esc_html__( 'Center', 'bixola-core' ),
						'icon' => 'eicon-text-align-center',
					],
					'right' => [
						'title' => esc_html__( 'Right', 'bixola-core' ),
						'icon' => 'eicon-text-align-right',
					],
				],
				'devices' => [ 'desktop', 'tablet', 'mobile' ],
				'prefix_class' => 'content-align-%s',
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-section' => 'text-align: {{VALUE}};'
                ],
                'condition' => [
                    'design_style' => 'layout-1',
                ],
			]
		);

        $this->end_controls_section();

        $this->start_controls_section(
            '_content_button',
            [
                'label' => esc_html__( 'Button', 'bixola-core' ),
                'tab' => Controls_Manager::TAB_CONTENT,
                'condition' => [
                    'design_style' => [ 'layout-3' ],
                ],
            ]
        );

        $this->add_control(
            'section_button_text',
            [
                'label' => esc_html__( 'Button Text', 'bixola-core' ),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__( 'Learn More', 'bixola-core' ),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'section_button_link_type',
            [
                'label' => esc_html__('Button Link Type', 'bixola-core'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '1' => 'Custom Link',
                    '2' => 'Internal Page',
                ],
                'default' => '1',
                'label_block' => true,
            ]
        );

        $this->add_control(
            'section_button_link',
            [
                'label' => esc_html__('Button link', 'bixola-core'),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => esc_html__('https://your-link.com', 'bixola-core'),
                'show_external' => false,
                'default' => [
                    'url' => '#',
                    'is_external' => true,
                    'nofollow' => true,
                    'custom_attributes' => '',
                ],
                'condition' => [
                    'section_button_link_type' => '1',
                ],
                'label_block' => true,
            ]
        );
        $this->add_control(
            'section_button_page_link',
            [
                'label' => esc_html__('Select Button Page', 'bixola-core'),
                'type' => Controls_Manager::SELECT2,
                'label_block' => true,
                'options' => bixola_get_all_pages(),
                'condition' => [
                    'section_button_link_type' => '2',
                ]
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_style_design_layout',
            [
                'label' => esc_html__( 'Design Layout',  'text-domain'  ),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );
        
        $this->add_responsive_control(
            'design_layout_margin',
            [
                'label' => esc_html__( 'Margin', 'text-domain' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-section' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        
        $this->add_responsive_control(
            'design_layout_padding',
            [
                'label' => esc_html__( 'Padding', 'text-domain' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-section' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        
        $this->add_control(
            'design_layout_background',
            [
                'label' => esc_html__( 'Background', 'text-domain' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-section' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
            '_style_title',
            [
                'label' => esc_html__( 'Title / Content', 'bixola-core' ),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            '_heading_style_section_title',
            [
                'type' => Controls_Manager::HEADING,
                'label' => esc_html__( 'Title', 'bixola-core' ),
            ]
        );

        $this->add_responsive_control(
            'section_title_bottom_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'text-domain' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'section_title_color',
            [
                'label' => esc_html__( 'Text Color', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-title' => 'color: {{VALUE}}',
                    
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'section_title_typography',
                'selector' => '{{WRAPPER}} .bixola-el-title',
            ]
        );

        $this->add_control(
            '_heading_style_section_subheading',
            [
                'type' => Controls_Manager::HEADING,
                'label' => esc_html__( 'Subheading', 'bixola-core' ),
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'section_subheading_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'bixola-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-subheading' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'section_subheading_color',
            [
                'label' => esc_html__( 'Text Color', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-subheading' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'section_subheading_border_color',
            [
                'label' => esc_html__( 'Border Color', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-subheading:before' => 'background-color: {{VALUE}}',
                ],
                'condition' => [
                    'design_style' => 'layout-1',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'section_subheading_typography',
                'selector' => '{{WRAPPER}} .bixola-el-subheading',
            ]
        );

        $this->add_control(
            '_heading_style_section_description',
            [
                'label' => esc_html__( 'Description', 'bixola-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
                'condition' => [
                    'design_style' => 'layout-2',
                ],
            ]
        );

        $this->add_responsive_control(
            'section_description_spacing',
            [
                'label' => esc_html__( 'Bottom Spacing', 'bixola-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-description' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'design_style' => 'layout-2',
                ],
            ]
        );

        $this->add_control(
            'section_description_color',
            [
                'label' => esc_html__( 'Color', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-description' => 'color: {{VALUE}}',
                    
                ],
                'condition' => [
                    'design_style' => 'layout-2',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'section_description_typography',
                'selector' => '{{WRAPPER}} .bixola-el-description',
                'condition' => [
                    'design_style' => 'layout-2',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
			'_style_button',
			[
				'label' => esc_html__( 'Button', 'bixola-core' ),
				'tab' => Controls_Manager::TAB_STYLE,
                'condition' => [
                    'design_style' => [ 'layout-3' ],
                ],
			]
		);

        $this->start_controls_tabs( 'tabs_button_style' );

        $this->start_controls_tab(
            'button_tab',
            [
                'label' => esc_html__( 'Normal', 'bixola-core' ),
            ]
        );

        $this->add_control(
            'button_color',
            [
                'label'     => esc_html__( 'Color', 'bixola-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-button'    => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'button_background',
            [
                'label'     => esc_html__( 'Background', 'bixola-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-button' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'label'    => esc_html__( 'Border', 'bixola-core' ),
                'name'     => 'button_border',
                'selector' => '{{WRAPPER}} .bixola-el-button',
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name'     => 'button_box_shadow',
                'selector' => '{{WRAPPER}} .bixola-el-button',
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'button_hover_tab',
            [
                'label' => esc_html__( 'Hover', 'bixola-core' ),
            ]
        );

        $this->add_control(
            'button_color_hover',
            [
                'label'     => esc_html__( 'Color', 'bixola-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-button:hover' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'button_background_hover',
            [
                'label'     => esc_html__( 'Background', 'bixola-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-button::before' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'label'    => esc_html__( 'Border', 'bixola-core' ),
                'name'     => 'button_border_hover',
                'selector' => '{{WRAPPER}} .bixola-el-button:hover',
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name'     => 'button_box_shadow_hover',
                'selector' => '{{WRAPPER}} .bixola-el-button:hover',
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->add_control(
            'button_border_radius',
            [
                'label'     => esc_html__( 'Border Radius', 'bixola-core' ),
                'type'      => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .bixola-el-button' => 'border-radius: {{SIZE}}{{UNIT}}',
                    '{{WRAPPER}} .bixola-el-button::before' => 'border-radius: {{SIZE}}{{UNIT}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'label'    => esc_html__( 'Typography', 'bixola-core' ),
                'name'     => 'button_typography',
                'selector' => '{{WRAPPER}} .bixola-el-button',
            ]
        );

        $this->add_control(
            'button_padding',
            [
                'label'      => esc_html__( 'Padding', 'bixola-core' ),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .bixola-el-button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'button_margin',
            [
                'label'      => esc_html__( 'Margin', 'bixola-core' ),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .bixola-el-button' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

		$this->end_controls_section();
	}

	/**
	 * Render the widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();
        
		?>

        <?php if ( $settings['design_style']  == 'layout-1' ):
            $this->add_render_attribute('title_args', 'class', 'bixola-el-title section-title__title');
            ?>

            <div class="bixola-el-section section-title">
                <?php if($settings['section_subheading']): ?>
                    <div class="section-title__tagline-box">
                        <p class="bixola-el-subheading section-title__tagline">
                            <?php print twinkle_kses($settings['section_subheading']); ?>
                        </p>
                    </div>
                <?php endif; ?>
                <?php
                    if ( !empty($settings['section_title' ]) ) :
                        printf( '<%1$s %2$s>%3$s</%1$s>',
                            tag_escape( $settings['section_title_tag'] ),
                            $this->get_render_attribute_string( 'title_args' ),
                            twinkle_kses( $settings['section_title' ] )
                            );
                    endif;
                ?>
            </div>

        <?php elseif ( $settings['design_style']  == 'layout-2' ): 
            $this->add_render_attribute('title_args', 'class', 'bixola-el-title section-title-three__title');
            ?>

            <div class="bixola-el-section industries-served__top">
                <div class="row">
                    <div class="col-xl-6 col-lg-6">
                        <div class="industries-served__left">
                            <div class="section-title-three text-left">
                                <?php if($settings['section_subheading']): ?>
                                    <div class="section-title-three__tagline-box">
                                        <p class="bixola-el-subheading section-title-three__tagline">
                                            <?php print twinkle_kses($settings['section_subheading']); ?>
                                        </p>
                                    </div>
                                <?php endif; ?>
                                <?php
                                    if ( !empty($settings['section_title' ]) ) :
                                        printf( '<%1$s %2$s>%3$s</%1$s>',
                                            tag_escape( $settings['section_title_tag'] ),
                                            $this->get_render_attribute_string( 'title_args' ),
                                            twinkle_kses( $settings['section_title' ] )
                                            );
                                    endif;
                                ?>
                            </div>
                        </div>
                    </div>
                    <div class="col-xl-6 col-lg-6">
                        <div class="industries-served__right">
                            <?php if($settings['section_description']): ?>
                                <p class="bixola-el-description industries-served__text">
                                    <?php print twinkle_kses($settings['section_description']); ?>
                                </p>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>

        <?php elseif ( $settings['design_style']  == 'layout-3' ): 
            $this->add_render_attribute('title_args', 'class', 'bixola-el-title section-title-three__title');

            // Link
            if ('2' == $settings['section_button_link_type']) {
                $this->add_render_attribute('bixola-button-arg', 'href', get_permalink($settings['section_button_page_link']));
                $this->add_render_attribute('bixola-button-arg', 'target', '_self');
                $this->add_render_attribute('bixola-button-arg', 'rel', 'nofollow');
                $this->add_render_attribute('bixola-button-arg', 'class', 'bixola-el-button team-six__btn thm-btn');
            } else {
                if ( ! empty( $settings['section_button_link']['url'] ) ) {
                    $this->add_link_attributes( 'bixola-button-arg', $settings['section_button_link'] );
                    $this->add_render_attribute('bixola-button-arg', 'class', 'bixola-el-button team-six__btn thm-btn');
                }
            }

            ?>

            <div class="bixola-el-section team-six__top">
                <div class="team-six__left">
                    <div class="section-title-three text-left">
                        <?php if($settings['section_subheading']): ?>
                            <div class="section-title-three__tagline-box">
                                <p class="bixola-el-subheading section-title-three__tagline">
                                    <?php print twinkle_kses($settings['section_subheading']); ?>
                                </p>
                            </div>
                        <?php endif; ?>
                        <?php
                            if ( !empty($settings['section_title' ]) ) :
                                printf( '<%1$s %2$s>%3$s</%1$s>',
                                    tag_escape( $settings['section_title_tag'] ),
                                    $this->get_render_attribute_string( 'title_args' ),
                                    twinkle_kses( $settings['section_title' ] )
                                    );
                            endif;
                        ?>
                    </div>
                </div>
                <?php if (!empty($settings['section_button_text'])) : ?>
                    <div class="team-six__btn-box">
                        <a <?php echo $this->get_render_attribute_string( 'bixola-button-arg' ); ?>>
                            <?php echo $settings['section_button_text']; ?>
                        </a>
                    </div>
                <?php endif; ?>
            </div>

        <?php endif; ?>

        <?php
	}
}

$widgets_manager->register( new Bixola_Heading() );