<?php
namespace BixolaCore\Widgets;

use \Elementor\Widget_Base;
use \Elementor\Controls_Manager;
use \Elementor\Group_Control_Image_Size;
use \Elementor\Repeater;
use \Elementor\Utils;
use \Elementor\Control_Media;

use \Elementor\Group_Control_Border;
use \Elementor\Group_Control_Box_Shadow;
use \Elementor\Group_Control_Text_Shadow;
use \Elementor\Group_Control_Typography;
use \Elementor\Group_Control_Background;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Bixola Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class Bixola_Features extends Widget_Base {

	/**
	 * Retrieve the widget name.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'bixola_features';
	}

	/**
	 * Retrieve the widget title.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Features', 'bixola-core' );
	}

	/**
	 * Retrieve the widget icon.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'bixola-icon';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * Note that currently Elementor supports only one category.
	 * When multiple categories passed, Elementor uses the first one.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'bixola_core' ];
	}

	/**
	 * Retrieve the list of scripts the widget depended on.
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'bixola-core' ];
	}

	/**
	 * Register the widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function register_controls() {

        $this->start_controls_section(
            '_content_design_layout',
            [
                'label' => esc_html__('Design Layout', 'bixola-core'),
            ]
        );

        $this->add_control(
            'design_style',
            [
                'label' => esc_html__('Select Layout', 'bixola-core'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'layout-1' => esc_html__('Layout 1', 'bixola-core'),
                    'layout-2' => esc_html__('Layout 2', 'bixola-core'),
                ],
                'default' => 'layout-1',
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_content_title',
            [
                'label' => esc_html__( 'Title',  'text-domain'  ),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );
        
        $this->add_control(
            'section_subheading',
            [
                'label' => esc_html__( 'Subheading', 'text-domain' ),
                'description' => bixola_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'Features', 'text-domain' ),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'section_title',
            [
                'label' => esc_html__( 'Title', 'text-domain' ),
                'description' => bixola_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'Grow Your Brand By Zeroing In On Your Target Market', 'text-domain' ),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'section_description',
            [
                'label' => esc_html__( 'Description', 'text-domain' ),
                'description' => bixola_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXTAREA,
                'default' => __( 'When I first moved to Israel, I began working with an organization that helps new immigrants find professional success by providing education.', 'text-domain' ),
                'label_block' => true,
                'condition' => [
                    'design_style' => 'layout-2',
                ],
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
            '_content_features_list',
            [
                'label' => esc_html__('Features List', 'bixola-core'),
                'tab' => Controls_Manager::TAB_CONTENT,
                'condition' => [
                    'design_style' => 'layout-1',
                ],
            ]
        );

        $repeater = new Repeater();

        $repeater->add_control(
            'features_image_hover_image',
            [
                'label' => esc_html__('Hover Image', 'bixola-core'),
                'type' => Controls_Manager::MEDIA,
            ]
        );

        $repeater->add_control(
            'features_shape_switch',
            [
                'label' => esc_html__( 'Shape ON/OFF', 'text-domain' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Show', 'text-domain' ),
                'label_off' => esc_html__( 'Hide', 'text-domain' ),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $repeater->add_control(
			'features_icon_switcher',
			[
				'label' => esc_html__( 'Image Type', 'bixola-core' ),
				'type' => Controls_Manager::SELECT,
				'default' => 'icon',
				'options' => [
					'image' => esc_html__( 'Image', 'bixola-core' ),
					'icon' => esc_html__( 'Icon', 'bixola-core' ),
				],
			]
		);

        $repeater->add_control(
            'features_image_icon',
            [
                'label' => esc_html__('Image Icon', 'bixola-core'),
                'type' => Controls_Manager::MEDIA,
                'condition' => [
                    'features_icon_switcher' => 'image',
                ],
            ]
        );

        if (bixola_is_elementor_version('<', '2.6.0')) {
            $repeater->add_control(
                'features_icon',
                [
                    'label' => esc_html__('Icon', 'bixola-core'),
                    'type' => Controls_Manager::ICON,
                    'label_block' => true,
                    'default' => 'icon-house',
                    'condition' => [
                        'features_icon_switcher' => 'icon',
                    ],
                ]
            );
        } else {
            $repeater->add_control(
                'features_selected_icon',
                [
                    'label' => esc_html__('Icon', 'bixola-core'),
                    'type' => Controls_Manager::ICONS,
                    'fa4compatibility' => 'icon',
                    'label_block' => true,
                    'default' => [
                        'value' => 'icon-house',
                        'library' => 'solid',
                    ],
                    'condition' => [
                        'features_icon_switcher' => 'icon',
                    ],
                ]
            );
        }

        $repeater->add_control(
            'features_title',
            [
                'label' => esc_html__('Title', 'bixola-core'),
                'description' => bixola_get_allowed_html_desc( 'basic' ),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'features_subheading',
            [
                'label' => esc_html__('Subtitle', 'bixola-core'),
                'description' => bixola_get_allowed_html_desc( 'basic' ),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'features_link_switcher',
            [
                'label' => esc_html__( 'Add link', 'bixola-core' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Yes', 'bixola-core' ),
                'label_off' => esc_html__( 'No', 'bixola-core' ),
                'return_value' => 'yes',
                'default' => 'yes',
                'separator' => 'before',
            ]
        );

        $repeater->add_control(
            'features_link_text',
            [
                'label' => esc_html__('Button Text', 'bixola-core'),
                'description' => bixola_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'default' => __('Read More', 'bixola-core'),
                'label_block' => true,
                'condition' => [
                    'features_link_switcher' => 'yes'
                ]
            ]
        );

        $repeater->add_control(
            'features_link_type',
            [
                'label' => esc_html__( 'Link Type', 'bixola-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '1' => 'Custom Link',
                    '2' => 'Internal Page',
                ],
                'default' => '1',
                'condition' => [
                    'features_link_switcher' => 'yes'
                ]
            ]
        );

        $repeater->add_control(
            'features_link',
            [
                'label' => esc_html__( 'Link', 'bixola-core' ),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => esc_html__( 'https://your-link.com', 'bixola-core' ),
                'show_external' => true,
                'default' => [
                    'url' => '#',
                    'is_external' => false,
                    'nofollow' => false,
                ],
                'condition' => [
                    'features_link_type' => '1',
                    'features_link_switcher' => 'yes',
                ]
            ]
        );

        $repeater->add_control(
            'features_page_link',
            [
                'label' => esc_html__( 'Select Page Link', 'bixola-core' ),
                'type' => Controls_Manager::SELECT2,
                'label_block' => true,
                'options' => bixola_get_all_pages(),
                'condition' => [
                    'features_link_type' => '2',
                    'features_link_switcher' => 'yes',
                ]
            ]
        );

        $repeater->add_control(
            'animation_delay',
            [
                'label' => esc_html__( 'Animation delay', 'text-domain' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '100ms' => esc_html__( '100ms', 'text-domain' ),
                    '200ms' => esc_html__( '200ms', 'text-domain' ),
                    '300ms' => esc_html__( '300ms', 'text-domain' ),
                    '400ms' => esc_html__( '400ms', 'text-domain' ),
                    '500ms' => esc_html__( '500ms', 'text-domain' ),
                    '600ms' => esc_html__( '600ms', 'text-domain' ),
                    '700ms' => esc_html__( '700ms', 'text-domain' ),
                    '800ms' => esc_html__( '800ms', 'text-domain' ),
                    '900ms' => esc_html__( '900ms', 'text-domain' ),
                ],
            ]
        );

        $this->add_control(
            'features_list',
            [
                'label' => esc_html__('Features List', 'bixola-core'),
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'features_subheading' => __('Our Industries', 'bixola-core'),
                        'features_title' => __('Banking & capital market', 'bixola-core'),
                        'animation_delay' => '100ms',
                    ],
                    [
                        'features_subheading' => __('Our Industries', 'bixola-core'),
                        'features_title' => __('Banking & capital market', 'bixola-core'),
                        'animation_delay' => '200ms',
                    ],
                    [
                        'features_subheading' => __('Our Industries', 'bixola-core'),
                        'features_title' => __('Logistic & transportation', 'bixola-core'),
                        'animation_delay' => '300ms',
                    ],
                    [
                        'features_subheading' => __('Our Industries', 'bixola-core'),
                        'features_title' => __('Justice & defence security', 'bixola-core'),
                        'animation_delay' => '400ms',
                    ],
                ],
                'title_field' => '{{{ features_title }}}',
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_content_features_list_two',
            [
                'label' => esc_html__('Features List', 'bixola-core'),
                'tab' => Controls_Manager::TAB_CONTENT,
                'condition' => [
                    'design_style' => 'layout-2',
                ],
            ]
        );

        $repeater = new Repeater();

        $repeater->add_control(
			'features_icon_switcher',
			[
				'label' => esc_html__( 'Image Type', 'bixola-core' ),
				'type' => Controls_Manager::SELECT,
				'default' => 'icon',
				'options' => [
					'image' => esc_html__( 'Image', 'bixola-core' ),
					'icon' => esc_html__( 'Icon', 'bixola-core' ),
				],
			]
		);

        $repeater->add_control(
            'features_image_icon',
            [
                'label' => esc_html__('Image Icon', 'bixola-core'),
                'type' => Controls_Manager::MEDIA,
                'condition' => [
                    'features_icon_switcher' => 'image',
                ],
            ]
        );

        $repeater->add_control(
            'features_selected_icon',
            [
                'label' => esc_html__('Icon', 'bixola-core'),
                'type' => Controls_Manager::ICONS,
                'fa4compatibility' => 'icon',
                'label_block' => true,
                'default' => [
                    'value' => 'icon-house',
                    'library' => 'solid',
                ],
                'condition' => [
                    'features_icon_switcher' => 'icon',
                ],
            ]
        );

        $repeater->add_control(
            'features_title',
            [
                'label' => esc_html__('Title', 'bixola-core'),
                'description' => bixola_get_allowed_html_desc( 'basic' ),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'features_link_switcher',
            [
                'label' => esc_html__( 'Add link', 'bixola-core' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Yes', 'bixola-core' ),
                'label_off' => esc_html__( 'No', 'bixola-core' ),
                'return_value' => 'yes',
                'default' => 'yes',
                'separator' => 'before',
            ]
        );

        $repeater->add_control(
            'features_link_text',
            [
                'label' => esc_html__('Button Text', 'bixola-core'),
                'description' => bixola_get_allowed_html_desc( 'intermediate' ),
                'type' => Controls_Manager::TEXT,
                'default' => __('Read More', 'bixola-core'),
                'label_block' => true,
                'condition' => [
                    'features_link_switcher' => 'yes'
                ]
            ]
        );

        $repeater->add_control(
            'features_link_type',
            [
                'label' => esc_html__( 'Link Type', 'bixola-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '1' => 'Custom Link',
                    '2' => 'Internal Page',
                ],
                'default' => '1',
                'condition' => [
                    'features_link_switcher' => 'yes'
                ]
            ]
        );

        $repeater->add_control(
            'features_link',
            [
                'label' => esc_html__( 'Link', 'bixola-core' ),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => esc_html__( 'https://your-link.com', 'bixola-core' ),
                'show_external' => true,
                'default' => [
                    'url' => '#',
                    'is_external' => false,
                    'nofollow' => false,
                ],
                'condition' => [
                    'features_link_type' => '1',
                    'features_link_switcher' => 'yes',
                ]
            ]
        );

        $repeater->add_control(
            'features_page_link',
            [
                'label' => esc_html__( 'Select Page Link', 'bixola-core' ),
                'type' => Controls_Manager::SELECT2,
                'label_block' => true,
                'options' => bixola_get_all_pages(),
                'condition' => [
                    'features_link_type' => '2',
                    'features_link_switcher' => 'yes',
                ]
            ]
        );

        $repeater->add_control(
            'animation_delay',
            [
                'label' => esc_html__( 'Animation delay', 'text-domain' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '100ms' => esc_html__( '100ms', 'text-domain' ),
                    '200ms' => esc_html__( '200ms', 'text-domain' ),
                    '300ms' => esc_html__( '300ms', 'text-domain' ),
                    '400ms' => esc_html__( '400ms', 'text-domain' ),
                    '500ms' => esc_html__( '500ms', 'text-domain' ),
                    '600ms' => esc_html__( '600ms', 'text-domain' ),
                    '700ms' => esc_html__( '700ms', 'text-domain' ),
                    '800ms' => esc_html__( '800ms', 'text-domain' ),
                    '900ms' => esc_html__( '900ms', 'text-domain' ),
                ],
            ]
        );

        $this->add_control(
            'features_list_two',
            [
                'label' => esc_html__('Features List', 'bixola-core'),
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'features_title' => __('Banking & <br> capital market', 'bixola-core'),
                        'animation_delay' => '100ms',
                    ],
                    [
                        'features_title' => __('Business <br> Audit', 'bixola-core'),
                        'animation_delay' => '200ms',
                    ],
                    [
                        'features_title' => __('Identifying <br> problems solutions', 'bixola-core'),
                        'animation_delay' => '300ms',
                    ],
                    [
                        'features_title' => __('Solutions <br> your problem', 'bixola-core'),
                        'animation_delay' => '400ms',
                    ],
                    [
                        'features_title' => __('Consultation <br> with our experts', 'bixola-core'),
                        'animation_delay' => '500ms',
                    ],
                    [
                        'features_title' => __('Manage & <br> Investment', 'bixola-core'),
                        'animation_delay' => '600ms',
                    ],
                ],
                'title_field' => '{{{ features_title }}}',
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            '_content_button',
            [
                'label' => esc_html__( 'Button', 'text-domain' ),
                'tab' => Controls_Manager::TAB_CONTENT,
                'condition' => [
                    'design_style' => 'layout-1',
                ],
            ]
        );
        
        $this->add_control(
            'section_button_description',
            [
                'label' => esc_html__( 'Button Text', 'text-domain' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'We help take your small business to the next level. Dont hesitate to', 'text-domain' ),
                'label_block' => true,
            ]
        );
        
        $this->add_control(
            'section_button_text',
            [
                'label' => esc_html__( 'Button Text', 'text-domain' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'contact us.', 'text-domain' ),
                'label_block' => true,
            ]
        );
        
        $this->add_control(
            'section_button_link_type',
            [
                'label' => esc_html__( 'Button Link Type', 'text-domain' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '1' => 'Custom Link',
                    '2' => 'Internal Page',
                ],
                'default' => '1',
                'label_block' => true,
            ]
        );
        
        $this->add_control(
            'section_button_link',
            [
                'label' => esc_html__( 'Button link', 'text-domain' ),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => esc_html__('https://your-link.com', 'text-domain'),
                'show_external' => false,
                'default' => [
                    'url' => '#',
                    'is_external' => true,
                    'nofollow' => true,
                    'custom_attributes' => '',
                ],
                'condition' => [
                    'section_button_link_type' => '1',
                ],
                'label_block' => true,
            ]
        );
        $this->add_control(
            'section_button_page_link',
            [
                'label' => esc_html__( 'Select Button Page', 'text-domain' ),
                'type' => Controls_Manager::SELECT2,
                'label_block' => true,
                'options' => bixola_get_all_pages(),
                'condition' => [
                    'section_button_link_type' => '2',
                ]
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
            '_content_column',
            [
                'label' => esc_html__( 'Column',  'bixola-core'  ),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );
        
        $this->add_control(
            'column_desktop',
            [
                'label' => esc_html__( 'Columns for Desktop', 'bixola-core' ),
                'description' => esc_html__( 'Screen width equal to or greater than 1200px', 'bixola-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    12 => esc_html__( '1 Columns', 'bixola-core' ),
                    6 => esc_html__( '2 Columns', 'bixola-core' ),
                    4 => esc_html__( '3 Columns', 'bixola-core' ),
                    3 => esc_html__( '4 Columns', 'bixola-core' ),
                    5 => esc_html__( '5 Columns (For Carousel Item)', 'bixola-core' ),
                    2 => esc_html__( '6 Columns', 'bixola-core' ),
                    1 => esc_html__( '12 Columns', 'bixola-core' ),
                ],
                'separator' => 'before',
                'default' => 4,
                'style_transfer' => true,
            ]
        );

        $this->add_control(
            'column_laptop',
            [
                'label' => esc_html__( 'Columns for Large', 'bixola-core' ),
                'description' => esc_html__( 'Screen width equal to or greater than 992px', 'bixola-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    12 => esc_html__( '1 Columns', 'bixola-core' ),
                    6 => esc_html__( '2 Columns', 'bixola-core' ),
                    4 => esc_html__( '3 Columns', 'bixola-core' ),
                    3 => esc_html__( '4 Columns', 'bixola-core' ),
                    5 => esc_html__( '5 Columns (For Carousel Item)', 'bixola-core' ),
                    2 => esc_html__( '6 Columns', 'bixola-core' ),
                    1 => esc_html__( '12 Columns', 'bixola-core' ),
                ],
                'separator' => 'before',
                'default' => 4,
                'style_transfer' => true,
            ]
        );

        $this->add_control(
            'column_tablet',
            [
                'label' => esc_html__( 'Columns for Tablet', 'bixola-core' ),
                'description' => esc_html__( 'Screen width equal to or greater than 768px', 'bixola-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    12 => esc_html__( '1 Columns', 'bixola-core' ),
                    6 => esc_html__( '2 Columns', 'bixola-core' ),
                    4 => esc_html__( '3 Columns', 'bixola-core' ),
                    3 => esc_html__( '4 Columns', 'bixola-core' ),
                    5 => esc_html__( '5 Columns (For Carousel Item)', 'bixola-core' ),
                    2 => esc_html__( '6 Columns', 'bixola-core' ),
                    1 => esc_html__( '12 Columns', 'bixola-core' ),
                ],
                'separator' => 'before',
                'default' => 6,
                'style_transfer' => true,
            ]
        );

        $this->add_control(
            'column_mobile',
            [
                'label' => esc_html__( 'Columns for Mobile', 'bixola-core' ),
                'description' => esc_html__( 'Screen width less than 767px', 'bixola-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    12 => esc_html__( '1 Columns', 'bixola-core' ),
                    6 => esc_html__( '2 Columns', 'bixola-core' ),
                    4 => esc_html__( '3 Columns', 'bixola-core' ),
                    3 => esc_html__( '4 Columns', 'bixola-core' ),
                    5 => esc_html__( '5 Columns (For Carousel Item)', 'bixola-core' ),
                    2 => esc_html__( '6 Columns', 'bixola-core' ),
                    1 => esc_html__( '12 Columns', 'bixola-core' ),
                ],
                'separator' => 'before',
                'default' => 12,
                'style_transfer' => true,
            ]
        );
        
        $this->end_controls_section();

        $this->start_controls_section(
            '_section_features_list_style',
            [
                'label' => __( 'Features List', 'bixola-core' ),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            '_heading_layout',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Layout', 'bixola-core' ),
            ]
        );

        $this->add_responsive_control(
            'features_list_padding',
            [
                'label' => __( 'Padding', 'bixola-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .key-featured__item' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'features_list_border_color',
            [
                'label' => __( 'Border', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .key-featured__item:first-child' => 'border-color: {{VALUE}}',
                    '{{WRAPPER}} .key-featured__item' => 'border-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            '_heading_title',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Title', 'bixola-core' ),
                'separator' => 'before'
            ]
        );

        $this->add_control(
            'title_color',
            [
                'label' => __( 'Color', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .key-featured__item-header h4' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'title_typography',
                'selector' => '{{WRAPPER}} .key-featured__item-header h4',
            ]
        );

        $this->add_control(
            '_heading_subtitle',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __( 'Subtitle', 'bixola-core' ),
                'separator' => 'before'
            ]
        );

        $this->add_control(
            'subtitle_color',
            [
                'label' => __( 'Color', 'bixola-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .key-featured__item-body p' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'subtitle_typography',
                'selector' => '{{WRAPPER}} .key-featured__item-body p',
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
			'bixola_button_style',
			[
				'label' => __( 'Button', 'bixola-core' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

        $this->start_controls_tabs( 'tabs_button_style' );

        $this->start_controls_tab(
            'button_tab',
            [
                'label' => esc_html__( 'Normal', 'bixola-core' ),
            ]
        );

        $this->add_control(
            'button_color',
            [
                'label'     => esc_html__( 'Color', 'bixola-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .rs-btn.btn-transparent'    => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'button_background',
            [
                'label'     => esc_html__( 'Background', 'bixola-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .rs-btn.btn-transparent' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'label'    => esc_html__( 'Border', 'bixola-core' ),
                'name'     => 'button_border',
                'selector' => '{{WRAPPER}} .rs-btn.btn-transparent',
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name'     => 'button_box_shadow',
                'selector' => '{{WRAPPER}} .rs-btn.btn-transparent',
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'button_hover_tab',
            [
                'label' => esc_html__( 'Hover', 'bixola-core' ),
            ]
        );

        $this->add_control(
            'button_color_hover',
            [
                'label'     => esc_html__( 'Color', 'bixola-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .rs-btn.btn-transparent:hover' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'button_background_hover',
            [
                'label'     => esc_html__( 'Background', 'bixola-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .rs-btn.btn-transparent::before' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'label'    => esc_html__( 'Border', 'bixola-core' ),
                'name'     => 'button_border_hover',
                'selector' => '{{WRAPPER}} .rs-btn.btn-transparent:hover',
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name'     => 'button_box_shadow_hover',
                'selector' => '{{WRAPPER}} .rs-btn.btn-transparent:hover',
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->add_control(
            'button_border_radius',
            [
                'label'     => esc_html__( 'Border Radius', 'bixola-core' ),
                'type'      => Controls_Manager::SLIDER,
                'selectors' => [
                    '{{WRAPPER}} .rs-btn.btn-transparent' => 'border-radius: {{SIZE}}px;',
                    '{{WRAPPER}} .rs-btn.btn-transparent::before' => 'border-radius: {{SIZE}}px;',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'label'    => esc_html__( 'Typography', 'bixola-core' ),
                'name'     => 'button_typography',
                'selector' => '{{WRAPPER}} .rs-btn.btn-transparent',
            ]
        );

        $this->add_control(
            'button_padding',
            [
                'label'      => esc_html__( 'Padding', 'bixola-core' ),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .rs-btn.btn-transparent' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'button_margin',
            [
                'label'      => esc_html__( 'Margin', 'bixola-core' ),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .rs-btn.btn-transparent' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

		$this->end_controls_section();
	}

	/**
	 * Render the widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

        if ('2' == $settings['section_button_link_type']) {
            $this->add_render_attribute('section-button-arg', 'href', get_permalink($settings['section_button_page_link']));
            $this->add_render_attribute('section-button-arg', 'target', '_self');
            $this->add_render_attribute('section-button-arg', 'rel', 'nofollow');
        } else {
            if ( ! empty( $settings['section_button_link']['url'] ) ) {
                $this->add_link_attributes( 'section-button-arg', $settings['section_button_link'] );
            }
        }

	?>
        <?php if ( $settings['design_style']  == 'layout-1' ) : ?>

            <section class="feature-three">
                <div class="container">
                    <div class="section-title-three text-center">
                        <?php if ( !empty( $settings['section_subheading'] ) ) : ?>
                            <div class="section-title-three__tagline-box">
                                <p class="section-title-three__tagline">
                                    <?php print twinkle_kses($settings['section_subheading']); ?>
                                </p>
                            </div>
                        <?php endif; ?>
                        <?php if ( !empty( $settings['section_title'] ) ) : ?>
                            <h2 class="section-title-three__title">
                                <?php print twinkle_kses($settings['section_title']); ?>
                            </h2>
                        <?php endif; ?>
                    </div>
                    <div class="row">
                    <?php foreach ($settings['features_list'] as $item) : 

                        if ( !empty($item['features_image_hover_image']['url']) ) {
                            $features_image_hover_image = !empty($item['features_image_hover_image']['id']) ? wp_get_attachment_image_url( $item['features_image_hover_image']['id'], 'full') : $item['features_image_hover_image']['url'];
                            $features_image_hover_image_alt = get_post_meta($item["features_image_hover_image"]["id"], "_wp_attachment_image_alt", true);
                        }
                    
                        // Link
                        if ('2' == $item['features_link_type']) {
                            $link = get_permalink($item['features_page_link']);
                            $target = '_self';
                            $rel = 'nofollow';
                        } else {
                            $link = !empty($item['features_link']['url']) ? $item['features_link']['url'] : '';
                            $target = !empty($item['features_link']['is_external']) ? '_blank' : '';
                            $rel = !empty($item['features_link']['nofollow']) ? 'nofollow' : '';
                        }

                    ?>
                    <div class="col-xl-<?php print esc_attr($settings['column_desktop']); ?> col-lg-<?php print esc_attr($settings['column_laptop']); ?> col-md-<?php print esc_attr($settings['column_tablet']); ?> col-sm-<?php print esc_attr($settings['column_mobile']); ?> wow fadeInUp" data-wow-delay="<?php print esc_attr($item['animation_delay']); ?>">
                        <div class="feature-three__single">
                            <div class="feature-three__bg" style="background-image: url(<?php print esc_url($features_image_hover_image); ?>);"></div>
                            <?php if ( !empty( $item['features_shape_switch'] ) ) : ?>
                                <div class="feature-three__shape-1"></div>
                                <div class="feature-three__shape-2"></div>
                                <div class="feature-three__shape-3"></div>
                            <?php endif; ?>
                            <?php if (!empty($item['features_image_icon']['url'])): ?>
                                <div class="feature-three__icon">
                                    <img class="img-fluid" src="<?php echo $item['features_image_icon']['url']; ?>" alt="<?php echo get_post_meta(attachment_url_to_postid($item['features_image_icon']['url']), '_wp_attachment_image_alt', true); ?>">
                                </div>
                            <?php endif; ?>
                            <?php if ( !empty($item['features_selected_icon']['value'])) : ?>
                                <div class="feature-three__icon">
                                    <?php bixola_render_icon($item, 'features_selected_icon'); ?>
                                </div>
                            <?php endif; ?>
                            <?php if ( !empty( $item['features_subheading' ] ) ) : ?>
                                <p class="feature-three__sub-title">
                                    <?php echo twinkle_kses($item['features_subheading' ]); ?>
                                </p>
                            <?php endif; ?>
                            <h3 class="feature-three__title">
                                <a target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>" href="<?php echo esc_url($link); ?>">
                                    <?php echo twinkle_kses($item['features_title' ]); ?>
                                </a>
                            </h3>
                            <div class="feature-three__btn">
                                <a target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>" href="<?php echo esc_url($link); ?>">
                                    <?php echo twinkle_kses($item['features_link_text' ]); ?>
                                </a>
                            </div>
                        </div>
                    </div>
                    <?php endforeach; ?>
                    <p class="feature-three__bottom-text">
                        <?php print twinkle_kses($settings['section_button_description']); ?>
                        <a <?php echo $this->get_render_attribute_string( 'section-button-arg' ); ?>>
                            <?php print twinkle_kses($settings['section_button_text']); ?>
                        </a>
                    </p>
                </div>
            </section>

        <?php elseif ( $settings['design_style']  == 'layout-2' ) : ?>
            
            <section class="industries-served">
                <div class="container">
                    <div class="industries-served__top">
                    <div class="row">
                        <div class="col-xl-6 col-lg-6">
                            <div class="industries-served__left">
                                <div class="section-title-three text-left">
                                    <?php if ( !empty( $settings['section_subheading'] ) ) : ?>
                                        <div class="section-title-three__tagline-box">
                                            <p class="section-title-three__tagline">
                                                <?php print twinkle_kses($settings['section_subheading']); ?>
                                            </p>
                                        </div>
                                    <?php endif; ?>
                                    <?php if ( !empty( $settings['section_title'] ) ) : ?>
                                        <h2 class="section-title-three__title">
                                            <?php print twinkle_kses($settings['section_title']); ?>
                                        </h2>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>
                        <div class="col-xl-6 col-lg-6">
                            <?php if ( !empty( $settings['section_description'] ) ) : ?>
                                <div class="industries-served__right">
                                    <p class="industries-served__text">
                                        <?php print twinkle_kses($settings['section_description']); ?>
                                    </p>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                    </div>
                    <div class="industries-served__bottom">
                    <div class="row">
                        <?php foreach ($settings['features_list_two'] as $item) : 

                            // Link
                            if ('2' == $item['features_link_type']) {
                                $link = get_permalink($item['features_page_link']);
                                $target = '_self';
                                $rel = 'nofollow';
                            } else {
                                $link = !empty($item['features_link']['url']) ? $item['features_link']['url'] : '';
                                $target = !empty($item['features_link']['is_external']) ? '_blank' : '';
                                $rel = !empty($item['features_link']['nofollow']) ? 'nofollow' : '';
                            }

                        ?>
                        <div class="col-xl-<?php print esc_attr($settings['column_desktop']); ?> col-lg-<?php print esc_attr($settings['column_laptop']); ?> col-md-<?php print esc_attr($settings['column_tablet']); ?> col-sm-<?php print esc_attr($settings['column_mobile']); ?> wow fadeInUp" data-wow-delay="<?php print esc_attr($item['animation_delay']); ?>">
                            <div class="industries-served__single">
                                <div class="industries-served__inner">
                                    <?php if (!empty($item['features_image_icon']['url'])): ?>
                                        <div class="industries-served__icon">
                                            <img class="img-fluid" src="<?php echo $item['features_image_icon']['url']; ?>" alt="<?php echo get_post_meta(attachment_url_to_postid($item['features_image_icon']['url']), '_wp_attachment_image_alt', true); ?>">
                                        </div>
                                    <?php endif; ?>
                                    <?php if ( !empty($item['features_selected_icon']['value'])) : ?>
                                        <div class="industries-served__icon">
                                            <?php bixola_render_icon($item, 'features_selected_icon'); ?>
                                        </div>
                                    <?php endif; ?>
                                    <h3 class="industries-served__title">
                                        <a target="<?php echo esc_attr($target); ?>" rel="<?php echo esc_attr($rel); ?>" href="<?php echo esc_url($link); ?>">
                                            <?php echo twinkle_kses($item['features_title' ]); ?>
                                        </a>
                                    </h3>
                                </div>
                            </div>
                        </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            </div>
        </section>

        <?php endif; ?>

        <?php
	}
}

$widgets_manager->register( new Bixola_Features() );