<?php
namespace BixolaCore\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;
use \Elementor\Repeater;
use \Elementor\Utils;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Bixola Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class Bixola_Advanced_Tab extends Widget_Base {

	/**
	 * Retrieve the widget name.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'bixola_advanced_tab';
	}

	/**
	 * Retrieve the widget title.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Advanced Tab', 'bixola-core' );
	}

	/**
	 * Retrieve the widget icon.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'bixola-icon';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * Note that currently Elementor supports only one category.
	 * When multiple categories passed, Elementor uses the first one.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'bixola_core' ];
	}

	/**
	 * Retrieve the list of scripts the widget depended on.
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'bixola-core' ];
	}

	/**
	 * Register the widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function register_controls() {

		$this->start_controls_section(
			'_content_title',
			[
				'label' => esc_html__( 'Title & Content',  'text-domain'  ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
            'bixola_design_style',
            [
                'label' => esc_html__('Select Layout', 'bixola-core'),
                'type' => Controls_Manager::HIDDEN,
                'options' => [
                    'layout-1' => esc_html__('Layout 1', 'bixola-core'),
                    'layout-2' => esc_html__('Layout 2', 'bixola-core'),
                ],
                'default' => 'layout-1',
            ]
        );

		$this->add_control(
			'section_subheading',
			[
				'label' => esc_html__( 'Subheading', 'text-domain' ),
				'description' => bixola_get_allowed_html_desc( 'intermediate' ),
				'type' => Controls_Manager::TEXT,
				'default' => __( 'What Do We Do?', 'text-domain' ),
				'label_block' => true,
			]
		);

		$this->add_control(
			'section_title',
			[
				'label' => esc_html__( 'Title', 'text-domain' ),
				'description' => bixola_get_allowed_html_desc( 'intermediate' ),
				'type' => Controls_Manager::TEXT,
				'default' => __( 'Check Our Best Services', 'text-domain' ),
				'label_block' => true,
			]
		);

		$this->add_control(
			'section_description',
			[
				'label' => esc_html__( 'Description', 'text-domain' ),
				'description' => bixola_get_allowed_html_desc( 'intermediate' ),
				'type' => Controls_Manager::TEXTAREA,
				'default' => __( 'It is a long established fact that a reader will be distracted by the readable content of a page when looking at its layout.', 'text-domain' ),
				'label_block' => true,
			]
		);
		
		$this->end_controls_section();

		$this->start_controls_section(
            '_content_advanced_tab',
            [
                'label' => __('Advanced Tabs', 'bixola-core'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $repeater = new Repeater();

        $repeater->add_control(
            'active_tab',
            [
                'label' => __('Is Active Tab?', 'bixola-core'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => __('Yes', 'bixola-core'),
                'label_off' => __('No', 'bixola-core'),
                'return_value' => 'yes',
                'frontend_available' => true,
            ]
        );

		$repeater->add_control(
			'advanced_tab_icon_switcher',
			[
				'label' => esc_html__( 'Image Type', 'bixola-core' ),
				'description' => esc_html__( 'Icon options are available for layout 2', 'bixola-core' ),
				'type' => Controls_Manager::SELECT,
				'default' => 'icon',
				'options' => [
					'image' => esc_html__( 'Image', 'bixola-core' ),
					'icon' => esc_html__( 'Icon', 'bixola-core' ),
				],
			]
		);

        $repeater->add_control(
            'advanced_tab_image',
            [
                'label' => esc_html__('Image Icon', 'bixola-core'),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
                'condition' => [
                    'advanced_tab_icon_switcher' => 'image',
                ],
            ]
        );

        $repeater->add_control(
			'advanced_tab_icon',
			[
				'label' => esc_html__('Icon', 'bixola-core'),
				'type' => Controls_Manager::ICONS,
				'fa4compatibility' => 'icon',
				'label_block' => true,
				'default' => [
					'value' => 'icon-house',
					'library' => 'solid',
				],
				'condition' => [
					'advanced_tab_icon_switcher' => 'icon',
				],
			]
		);

        $repeater->add_control(
            'title',
            [
                'type' => Controls_Manager::TEXT,
                'label' => __('Title', 'bixola-core'),
                'dynamic' => [
                    'active' => true,
				],
				'label_block' => true,
            ]
        );

        $repeater->add_control(
            'template',
            [
                'label' => __('Section Template', 'bixola-core'),
                'placeholder' => __('Select a section template for as tab content', 'bixola-core'),
  
                'type' => Controls_Manager::SELECT2,
                'options' => get_elementor_templates()
            ]
        );

        $this->add_control(
            'tabs',
            [
                'show_label' => false,
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'title_field' => '{{title}}',
                'default' => [
                    [
                        'title' => 'Business Audit',
                    ],
                    [
                        'title' => 'Tax Strategy',
					],
                    [
                        'title' => 'Financial Advices',
					],
                    [
                        'title' => 'Insurance Strategy',
					],
                    [
                        'title' => 'Start Ups',
					],
                    [
                        'title' => 'Manage Investment',
					],
                ]
            ]
        );

        $this->end_controls_section();
		
		$this->start_controls_section(
			'_content_design_style',
			[
				'label' => esc_html__( 'Design Layout',  'text-domain'  ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);
		
		$this->add_responsive_control(
			'design_layout_margin',
			[
				'label' => esc_html__( 'Margin', 'text-domain' ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em' ],
				'selectors' => [
					'{{WRAPPER}} .bixola-el-section' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'design_layout_padding',
			[
				'label' => esc_html__( 'Padding', 'text-domain' ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px' ],
				'selectors' => [
					'{{WRAPPER}} .bixola-el-section' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'design_layout_background',
			[
				'label' => esc_html__( 'Background', 'text-domain' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .bixola-el-section' => 'background-color: {{VALUE}}',
				],
			]
		);
		
		$this->end_controls_section();

		$this->start_controls_section(
			'_style_title',
			[
				'label' => esc_html__( 'Title & Content',  'text-domain'  ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);
		
		$this->add_control(
			'_heading_style_section_title',
			[
				'label' => esc_html__( 'Title', 'text-domain' ),
				'type' => Controls_Manager::HEADING,
			]
		);
		
		$this->add_responsive_control(
			'section_title_spacing',
			[
				'label' => esc_html__( 'Bottom Spacing', 'text-domain' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => ['px'],
				'selectors' => [
					'{{WRAPPER}} .bixola-el-section-title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);
		
		$this->add_control(
			'section_title_color',
			[
				'label' => __( 'Color', 'text-domain' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .bixola-el-section-title' => 'color: {{VALUE}}',
				],
			]
		);
		
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'section_title_typography',
				'selector' => '{{WRAPPER}} .bixola-el-section-title',
			]
		);

		$this->add_control(
			'_heading_style_section_subheading',
			[
				'label' => esc_html__( 'Subheading', 'text-domain' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);
		
		$this->add_responsive_control(
			'section_subheading_spacing',
			[
				'label' => esc_html__( 'Bottom Spacing', 'text-domain' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => ['px'],
				'selectors' => [
					'{{WRAPPER}} .bixola-el-section-subheading' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);
		
		$this->add_control(
			'section_subheading_color',
			[
				'label' => __( 'Color', 'text-domain' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .bixola-el-section-subheading' => 'color: {{VALUE}}',
				],
			]
		);
		
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'section_subheading_typography',
				'selector' => '{{WRAPPER}} .bixola-el-section-subheading',
			]
		);

		$this->add_control(
			'_heading_style_section_description',
			[
				'type' => Controls_Manager::HEADING,
				'label' => __( 'Description', 'text-domain' ),
				'separator' => 'before',
			]
		);
		
		$this->add_responsive_control(
			'section_description_spacing',
			[
				'label' => __( 'Bottom Spacing', 'text-domain' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => ['px'],
				'selectors' => [
					'{{WRAPPER}} .bixola-el-section-description' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);
		
		$this->add_control(
			'section_description_color',
			[
				'label' => __( 'Color', 'text-domain' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .bixola-el-section-description' => 'color: {{VALUE}}',
				],
			]
		);
		
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'section_description_typography',
				'selector' => '{{WRAPPER}} .bixola-el-section-description',
			]
		);
		
		$this->end_controls_section();

		$this->start_controls_section(
			'_style_advance_tab',
			[
				'label' => __( 'Advanced Tab', 'bixola-core' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

		$this->start_controls_tabs( '_advanced_tabs' );
		
		$this->start_controls_tab(
			'advanced_tab_normal_tab',
			[
				'label' => esc_html__( 'Normal', 'bixola-core' ),
			]
		);
		
		$this->add_control(
			'advanced_tab_color',
			[
				'label' => esc_html__( 'Color', 'bixola-core' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .services-four__tab-buttons-single' => 'color: {{VALUE}}',
				],
			]
		);
		
		$this->add_control(
			'advanced_tab_background',
			[
				'label' => esc_html__( 'Background', 'bixola-core' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .services-four__tab-buttons-single' => 'background-color: {{VALUE}}',
					'{{WRAPPER}} .services-four__tab-buttons-single:before' => 'border-top-color: {{VALUE}}',
					'{{WRAPPER}} .services-four__tab-buttons-single:after' => 'border-top-color: {{VALUE}}',
				],
			]
		);
		
		$this->end_controls_tab();
		
		$this->start_controls_tab(
			'advanced_tab_hover_tab',
			[
				'label' => esc_html__( 'Hover', 'bixola-core' ),
			]
		);
		
		$this->add_control(
			'advanced_tab_color_hover',
			[
				'label' => esc_html__( 'Color', 'bixola-core' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .services-four__tab-buttons-single:hover .services-four__tab-buttons-title' => 'color: {{VALUE}}',
				],
			]
		);
		
		$this->add_control(
			'advanced_tab_background_hover',
			[
				'label' => esc_html__( 'Background', 'bixola-core' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .services-four__tab-buttons-single:after' => 'background-color: {{VALUE}}',
					'{{WRAPPER}} .services-four__tab-buttons-single:hover:before' => 'border-top-color: {{VALUE}}',
				],
			]
		);
		
		$this->end_controls_tab();
		
		$this->start_controls_tab(
			'advanced_tab_active_tab',
			[
				'label' => esc_html__( 'Active', 'bixola-core' ),
			]
		);
		
		$this->add_control(
			'advanced_tab_color_active',
			[
				'label' => esc_html__( 'Color', 'bixola-core' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .services-four__main-tab .tab-buttons .tab-btn.active-btn .services-four__tab-buttons-title' => 'color: {{VALUE}}',
				],
			]
		);
		
		$this->add_control(
			'advanced_tab_background_active',
			[
				'label' => esc_html__( 'Background', 'bixola-core' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .services-four__main-tab .tab-buttons .tab-btn.active-btn .services-four__tab-buttons-single:after' => 'background-color: {{VALUE}}',
					'{{WRAPPER}} .services-four__main-tab .tab-buttons .tab-btn.active-btn .services-four__tab-buttons-single:before' => 'border-top-color: {{VALUE}}',
				],
			]
		);
		
		$this->end_controls_tab();
		
		$this->end_controls_tabs();

		$this->end_controls_section();
	}

	/**
	 * Render the widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		?>

		<?php if ( $settings['bixola_design_style']  == 'layout-1' ): ?>

			<section class="bixola-el-section services-four">
				<div class="container">
					<div class="services-four__top">
						<div class="row">
							<div class="col-xl-6 col-lg-6">
								<div class="services-four__left">
									<div class="section-title-three text-left">
										<?php if ( !empty( $settings['section_subheading'] ) ) : ?>
											<div class="section-title-three__tagline-box">
												<p class="bixola-el-section-subheading section-title-three__tagline">
													<?php print twinkle_kses($settings['section_subheading']); ?>
												</p>
											</div>
										<?php endif; ?>
										<?php if ( !empty( $settings['section_title'] ) ) : ?>
											<h2 class="bixola-el-section-title section-title-three__title">
												<?php print twinkle_kses($settings['section_title']); ?>
											</h2>
										<?php endif; ?>
									</div>
								</div>
							</div>
							<div class="col-xl-6 col-lg-6">
								<?php if ( !empty( $settings['section_description'] ) ) : ?>
									<div class="services-four__right">
										<p class="bixola-el-section-description services-four__text">
											<?php print twinkle_kses($settings['section_description']); ?>
										</p>
									</div>
								<?php endif; ?>
							</div>
						</div>
					</div>
					<div class="services-four__bottom">
						<div class="services-four__main-tab tabs-box">
							<ul class="tab-buttons clearfix list-unstyled">
								<?php foreach ($settings['tabs'] as $key => $tab):
									$active = $tab['active_tab'] ? 'active-btn' : '';
								?>
								<li data-tab="#tab-<?php echo esc_attr($key); ?>" class="tab-btn <?php echo esc_attr($active); ?>">
									<div class="services-four__tab-buttons-single">
										<?php if ( $tab['advanced_tab_icon_switcher']  == 'image' ): ?>
											<div class="services-four__tab-buttons-icon">
												<img class="img-fluid" src="<?php echo $tab['advanced_tab_image_icon']['url']; ?>" alt="<?php echo get_post_meta(attachment_url_to_postid($tab['advanced_tab_image_icon']['url']), '_wp_attachment_image_alt', true); ?>">
											</div>
										<?php elseif ( $tab['advanced_tab_icon_switcher']  == 'icon' ): ?>
											<div class="services-four__tab-buttons-icon">
												<?php bixola_render_icon($tab, 'advanced_tab_icon' ); ?>
											</div>
										<?php endif; ?>
										<h4 class="services-four__tab-buttons-title">
											<?php echo twinkle_kses($tab['title']); ?>
										</h4>
									</div>
								</li>
								<?php endforeach; ?>
							</ul>
							<div class="tabs-content">
								<?php foreach ($settings['tabs'] as $key => $tab):
									$active = $tab['active_tab'] ? 'active-tab' : '';
								?>
								<div class="tab <?php echo esc_attr($active); ?>" id="tab-<?php echo esc_attr($key); ?>">
									<?php echo \Elementor\Plugin::instance()->frontend->get_builder_content($tab['template'], true); ?>
								</div>
								<?php endforeach; ?>
							</div>
						</div>
					</div>
				</div>
			</section>
		
		<?php elseif ( $settings['bixola_design_style']  == 'layout-2' ): ?>
		
		<?php endif; ?>

            

		<?php
	}

}
$widgets_manager->register( new Bixola_Advanced_Tab() );